#ifndef _LIGO_EVENTCONDITIONLOGIC_H
#define _LIGO_EVENTCONDITIONLOGIC_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: ConditionLogic						*/
/*                                                         		*/
/* Module Description: Defines a boolean event condition		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: ConditionLogic.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "events/Condition.hh"
#include "events/ConditionPtr.hh"


namespace events {


/** A boolean expression for event conditions.
   
    @memo Defines an event condition
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ConditionLogic : public Condition  {
   public:
      /** Boolean operations supported by the condition.
          @memo Boolean operations.
       ******************************************************************/
      enum operation {
      /// Always false
      opFalse = 0,
      /// Always true
      opTrue = 1,
      /// identity condition
      opIdentity = 2,
      /// not condition
      opNot = 3,
      /// and condition
      opAnd = 4,
      /// or condition
      opOr = 5,
      /// xor condition
      opXor = 6
      };
   
      /** Creates an event boolean condition with zero arguments.
          @memo Constructor
       ******************************************************************/
      explicit ConditionLogic (operation op) 
      : mOp (op), mPtr1(0), mPtr2(0) {
      }
      /** Creates an event boolean condition with zero arguments.
          @memo Constructor
       ******************************************************************/
      explicit ConditionLogic (bool value) 
      : mOp (value ? opTrue : opFalse), mPtr1(0), mPtr2(0) {
      }
      /** Creates an event boolean condition with one argument.
          @memo Constructor
       ******************************************************************/
      explicit ConditionLogic (const Condition& cond,
                        operation op = opIdentity) 
      : mOp (op), mPtr1 (cond), mPtr2(0) {
      }
      explicit ConditionLogic (const ConditionPtr& cond,
                        operation op = opIdentity) 
      : mOp (op), mPtr1 (cond), mPtr2(0) {
      }
      /** Creates an event boolean condition with two arguments.
          @memo Constructor
       ******************************************************************/
      explicit ConditionLogic (const Condition& cond1,
                        const Condition& cond2, operation op) 
      : mOp (op), mPtr1 (cond1), mPtr2 (cond2) {
      }
      explicit ConditionLogic (const ConditionPtr& cond1,
                        const ConditionPtr& cond2, operation op) 
      : mOp (op), mPtr1 (cond1), mPtr2 (cond2) {
      }
   
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      virtual ConditionLogic* Copy() const {
         return new ConditionLogic (*this); }
   
      /** Evaluate the condition.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, bool& val) const;
   
   protected:
      /** Default constructor.
          @memo Default constructor
       ******************************************************************/
      ConditionLogic () : mOp (opFalse) {
      }
   
   private:
      /// Operation
      operation		mOp;
      /// First argument
      ConditionPtr	mPtr1;
      /// Second argument
      ConditionPtr	mPtr2;
   };

/** @name Bolean operators
    The following comparison operations are supported:
    \begin{verbatim}
    !Condition
    Condition && Condition
    Condition || Condition
    \end{verbatim}
   
    @memo Defines an event condition boolean operators
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{
   /** Boolean operations for event conditions (not).
       @memo Event condition logic (not)
       @return true if boolean comparision true
    *********************************************************************/
   inline ConditionLogic operator! (const Condition& cond) {
      return ConditionLogic (cond, ConditionLogic::opNot); }
   inline ConditionLogic operator! (const ConditionPtr& cond) {
      return ConditionLogic (cond, ConditionLogic::opNot); }
   /** Boolean operations for event conditions (and).
       @memo Event condition logic (and)
       @return true if boolean comparision true
    *********************************************************************/
   inline ConditionLogic operator&& (const Condition& cond1,
                     const Condition& cond2) {
      return ConditionLogic (cond1, cond2, ConditionLogic::opAnd); }
   inline ConditionLogic operator&& (const ConditionPtr& cond1,
                     const ConditionPtr& cond2) {
      return ConditionLogic (cond1, cond2, ConditionLogic::opAnd); }
   /** Boolean operations for event conditions (or).
       @memo Event condition logic (or)
       @return true if boolean comparision true
    *********************************************************************/
   inline ConditionLogic operator|| (const Condition& cond1,
                     const Condition& cond2) {
      return ConditionLogic (cond1, cond2, ConditionLogic::opOr); }
   inline ConditionLogic operator|| (const ConditionPtr& cond1,
                     const ConditionPtr& cond2) {
      return ConditionLogic (cond1, cond2, ConditionLogic::opOr); }

//@}

/** @name Shift a condition
    When a condition is shifted the specified number of events are removed
    from the argument list and the condition is evaluated on the remaining
    event arguments.
   
    @memo Defines the condition shift operation
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
//@{

/** Shifted condition.
    @memo Shifted condition
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ShiftCondition : public Condition {
   public:
      /** Creates a shifted condition.
          @memo Constructor
          @param cond Condition to be shifted
          @param howmuch Number of events to shift
       ******************************************************************/
      explicit ShiftCondition (const Condition& cond, int howmuch = 1) :
      mShift (howmuch), mPtr (cond) {
      }
      explicit ShiftCondition (const ConditionPtr& cond, int howmuch = 1) :
      mShift (howmuch), mPtr (cond) {
      }
      /** Returns a copy of the event condition. This method must be 
          overriden by all descendents.
          @memo Copy the event
          @return event copy
       ******************************************************************/
      virtual ShiftCondition* Copy() const {
         return new ShiftCondition (*this); }
   
      /** Evaluate the condition.
          @memo Evaluate
          @param arg Event argument list
          @param val Condition value (return)
          @return True if condition could be evaluated
       ******************************************************************/
      virtual bool Evaluate (const Argument& arg, bool& val) const;
   
   private:
      /// Shift amount
      int		mShift;
      /// Condition 
      ConditionPtr 	mPtr;
   };

/** Shift function
    @memo Shift a condition
    @param cond Condition to be shifted
    @param howmuch Number of events to shift
    @return Shifted condition
 ************************************************************************/
   ShiftCondition Shift (const Condition& cond, int howmuch = 1);
   ShiftCondition Shift (const ConditionPtr& cond, int howmuch = 1);

   inline ShiftCondition Shift (const Condition& cond, int howmuch) {
      return ShiftCondition (cond, howmuch); }
   inline ShiftCondition Shift (const ConditionPtr& cond, int howmuch) {
      return ShiftCondition (cond, howmuch); }
//@}

}

#endif // _LIGO_EVENTCONDITIONLOGIC_H
