/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Argument						*/
/*                                                         		*/
/* Module Description: Argument on which functions and conditions	*/
/*                     are evaluated					*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Argument.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_EVENTARGUMENT_H
#define _LIGO_EVENTARGUMENT_H


#include <complex>
#include "events/Event.hh"
#include "events/Window.hh"


namespace events {

/** The event argument class contains the list of arguments which are
    passed to event functions and event conditions. At this moment
    it is just a wrapper around an event window.
   
    @memo Defines an event function argument
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Argument {
   public:
      /// Value type
      typedef Window::value_type value_type;
      /// Pointer type
      typedef Window::pointer pointer;
      /// Const pointer type
      typedef Window::const_pointer const_pointer;
      /// Reference type
      typedef Window::reference reference;
      /// Const reference type
      typedef Window::const_reference const_reference;
      /// Iterator type
      typedef Window::iterator iterator;
      /// Const iterator type
      typedef Window::const_iterator const_iterator;
   
      /** Creates an argument from an event window.
          @memo Constructor
       ******************************************************************/
      Argument (Window& window) : mWindow (window) {
      }
   
      /** Get the event window.
          @memo Get event window
       ******************************************************************/
      Window& GetWindow () {
         return mWindow; }
      /** Get the event window.
          @memo Get event window
       ******************************************************************/
      const Window& GetWindow () const {
         return mWindow; }
   
      /** Get window order. How many current or coincidence events are 
          selected?
          @memo Window order
       ******************************************************************/
      int GetOrder() const {
         return mWindow.GetOrder(); }
      /** Access the current event(s).
          @memo Current event access
       ******************************************************************/
      reference operator() (int n) {
         return mWindow(n); }
      /** Access the current event(s).
          @memo Current event access
       ******************************************************************/
      const_reference operator() (int n) const {
         return mWindow(n); }
   
      /** First event in window.
          @memo First
       ******************************************************************/
      reference Front() {
         return mWindow.Front(); }
      /** First event in window.
          @memo First
       ******************************************************************/
      const_reference Front() const {
         return mWindow.Front(); }
      /** Last event in window.
          @memo Last
       ******************************************************************/
      reference Back() {
         return mWindow.Back(); }
      /** Last event in window.
          @memo Last
       ******************************************************************/
      const_reference Back() const {
         return mWindow.Back(); }
      /** Beginning of list.
          @memo Begin
       ******************************************************************/
      iterator Begin() {
         return mWindow.Begin(); }
      /** Beginning of list.
          @memo Begin
       ******************************************************************/
      const_iterator Begin() const {
         return mWindow.Begin(); }
      /** End of list.
          @memo End
       ******************************************************************/
      iterator End() {
         return mWindow.End(); }
      /** End of list.
          @memo End
       ******************************************************************/
      const_iterator End() const {
         return mWindow.End(); }
      /** Event list access.
          @memo List access
       ******************************************************************/
      reference operator[] (int n) {
         return mWindow[n]; }
      /** Event list access.
          @memo List access
       ******************************************************************/
      const_reference operator[] (int n) const {
         return mWindow[n]; }
      /** Number of events within window.
          @memo Size
       ******************************************************************/
      int Size() const {
         return mWindow.Size(); }
      /** No events?.
          @memo No events?
       ******************************************************************/
      bool Empty() const {
         return mWindow.Empty(); }
   
   private:
      /// Reference to event window
      Window&		mWindow;
   };


}

#endif // _LIGO_EVENTARGUMENT_H
