//
//    Methods for the Window base class.
//
//    21/07/99  JGZ   I added the symmetric calculation method to setWindow
//                    and rescaled the window coefficients by 1/RMS to give
//                    unit power normalization. By integrating the Hanning
//                    window squared, I calculate the window RMS to be 
//                    sqrt(0.375)=0.6123724. The direct RMS calculation for
//                    an 8192 point window gives 0.6123351
//
#include "DVecType.hh"
#include "Interval.hh"
#include "Time.hh"
#include "TSeries.hh"
#include "window_api.hh"
#include "constant.hh"
#include <cmath>

//--------------------------------------  Window constructor
window_api::window_api(void)
  : mRMS(0.0)
{}

//--------------------------------------  Copy constructor
window_api::window_api (const window_api& w)
  : mRMS(0.0)
{
   *this = w;
}

//--------------------------------------  Window destructor
window_api::~window_api() {
}

//--------------------------------------  Assignment operator
window_api& window_api::operator= (const window_api& w) {
   mRMS = w.mRMS;
   if (mWindow.get() && w.mWindow.get()) *mWindow = *w.mWindow;
   else if (w.mWindow.get())             mWindow.reset(w.mWindow->clone());
   else if (mWindow.get())               mWindow->Clear();
   return *this;
}

//--------------------------------------  Return true if symmetric around x=pi
bool 
window_api::isPiSymmetric(void) {
    return true;
}

//--------------------------------------  Set length N window contents
void 
window_api::setWindow(int N) {
    if (getLength() == N) return;

    //----------------------------------  Allocate storage.
    if (mWindow.get()) mWindow->ReSize(N);
    else               mWindow.reset(new DVecType<element_type>(N));
    element_type* pWindow=reinterpret_cast<element_type*>(mWindow->refData());

    //----------------------------------  Fill in the window vector.
    double SumSq = 0;
    double dPhi(0);
    if (N>1) dPhi = twopi/double(N-1); // Really N-1?
    int n2 = N/2;
    if (isPiSymmetric()) {
        for (int i=0; i<n2; i++) {  // loop over independent entries
	    double Xn       = WinFc(dPhi*double(i));
	    pWindow[i]      = Xn;
	    pWindow[N-1-i]  = Xn;
	    SumSq          += Xn*Xn;
	}
	SumSq *= 2.0;
	if ((N & 1)) {                //  One more entry for odd N.
	    pWindow[n2] = WinFc(pi);
	    SumSq += pWindow[n2] * pWindow[n2];
	}
    } else {                          //  No symmetry assumption
        for (int i=0; i<N; i++) {
	    pWindow[i] = WinFc(dPhi*double(i));
	    SumSq += pWindow[i] * pWindow[i];
	}
    }

    //----------------------------------  Now scale by 1/RMS to maintain power
    mRMS = std::sqrt(SumSq/N);
    double Norm = 1.0/mRMS;
    *mWindow *= Norm;
}

//--------------------------------------  Apply window to the specified series
TSeries 
window_api::apply(const TSeries& ts) {
    TSeries r(ts);
    unsigned int N = ts.getNSample();
    if (!N) return r;
    setWindow(N);
    DVector* dv = r.refDVect();
    switch (dv->getType()) {
    case DVector::t_short:
    case DVector::t_int:
        r.Convert(DVecType<element_type>::getDataType());
	dv = r.refDVect();
        break;
    case DVector::t_long:
        r.Convert(DVector::t_double);
	dv = r.refDVect();
        break;
    default:
        break;
    }
    dv->mpy(0, *mWindow, 0, N);
    return r;
}

//---------------------------------------  All data are valid
void
window_api::dataCheck(const TSeries& ts) const {
    return;
}

int
window_api::getLength(void) const {
    if (!mWindow.get()) return 0;
    return mWindow->getLength();
}

bool
window_api::isDataValid(const TSeries& ts) const {
    return true;
}
