/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIG_TRIGWRITER_HH
#define TRIG_TRIGWRITER_HH

#include "lmsg/MsgTypes.hh"
#include "TrigProc.hh"
#include <string>


namespace trig {

    class Segment;
    class TrigBase;

    /**  The %TrigWriter class is an API for writing triggers to various 
     *  trigger output databases and formats. It is used by the TrigClient 
     *  class. The usage mode is as follows:
     *  <ol>
     *    <li> Construct the TrigWriter nased class with \e e.g. 
     *         mWriter = new LdasDBWriter;
     *    <li> Set the default processID with:
     *         mWriter->setProcess(const trig::TrigProc&);
     *    <li> Write segments/triggers with 
     *         mWriter->addSegment(const trig::Segment&)
     *         or with
     *         addTrigger(const trig::TrigBase&)
     *    <li> The data are written to a file explicitly with:
     *          write(const std::string& file);
     *  </ol>
     *  @memo Trigger writer API
     *  @author J. Zweizig
     *  @version 1.0; Last Modified September 23, 2005
     */
    class TrigWriter {
    public:
	/**  Enumerate the trigger logging mode.
	 */
	enum trig_mode {
	    kNone,         ///< No write is active.
	    kMgr,          ///< Write to trigger manager
	    kWriter,       ///< Default trigger file writer
	    kSegWrt,       ///< Default segment file writer
	    kS6Seg,        ///< S6 segment file writer.
	    kSBTrig,       ///< Sngl_burst trigger xml file writer
	    kHdf5Trig      ///< Sngl_burst trigger hdf5 file writer
	};

    public:
	/**  All local data are initialized to their default values.
	 *  @memo Default constructor.
	 */
	TrigWriter(void);

	/**  Close out writer and delete all local data.
	  *  \brief Destructor.
	  */
	virtual ~TrigWriter(void);

	/**  The trigger is written to the trigger output stream and associated 
	 *  with the specified process ID. In buffered writers, the trigger 
	 *  (and process ID if unique) are entered in the writer tables until
	 *  the buffer is flushed by a write command.
	 *  \brief Write a trigger to the output stream.
	 *  \param t Constant reference to the trigger to be written.
	 *  \param p Constant reference to a descriptor of the process that 
	 *          generated trigger \a t.
	 *  \return lmsg error code.
	 */
	virtual lmsg::error_type addTrigger(const TrigBase& t, 
					    const TrigProc& p)=0;

	/**  Write a trigger associated with the default processID (as 
	 *  specified by the setProcess method).
	 *  \brief Write a trigger.
	 *  \param t Constant reference to the trigger to be written.
	 *  \return lmsg error code
	 */
	lmsg::error_type addTrigger(const TrigBase& t);

	/**  The segment is written to the segment output stream and associated 
	 *  with the specified process ID. In buffered writers, the segment
	 *  (and process ID if unique) are entered in the writer tables until
	 *  the buffer is flushed by a write command.
	 *  \brief memo Write a segment.
	 *  \param s Constant reference to %Segment to be written to the stream.
	 *  \param p Constant reference to descriptor of the process that 
	 *           generated segment \a s.
	 *  \return lmsg error code
	 */ 
	virtual lmsg::error_type addSegment(const Segment& s, 
					    const TrigProc& p)=0;

	/**  Write a segment associated with the default processID (as 
	 *  specified by the setProcess method).
	 *  \brief Write a segment.
	 *  \param s Constant reference to %Segment to be added to the list.
	 *  \return lmsg error code
	 */
	lmsg::error_type addSegment(const Segment& s);

	/**  Clear the internal trigger and process tables unique to the
	 *  interval \c t\>=start and \c t<end.
	 *  \brief Remove triggers and segments in a specified interval.
	 *  \param start Start time.
	 *  \param end   End time.
	 */
	virtual void clear(const Time& start, const Time& end) = 0;

	/**  Test whether the trigger and segment lists are empty before the
	 *  specified time.
	 *  \brief Test for no triggers/segments before the specified time.
	 *  \param t End time for test.
	 *  \return True if no buffered triggers or segments start before 
	 *          time \a t.
	 */
	virtual bool empty(const Time& t) const;

	/**  Get the debug message level.
	 *  \brief Debug level
	 *  \return Current debug message level.
	 */
	virtual lmsg::index_type getDebug(void) const;

	/**  Return the time of the first available trigger or segment.
	 *  \brief Earliest trigger time
	 *  \return Start time of the  first available trigger or segment.
	 */
	virtual Time getEarly(void) const;

	/**  Get the trigger writer mode
	 *  \brief Writer mode 
	 *  \return Enumerated trigger writer mode.
	 */
	virtual trig_mode getMode(void) const = 0;

	/**  Get the number of segments starting before the specified time 
	 *  currently buffered by the writer.
	 *  \brief Get number of buffered segments.
	 *  \param t End time for segment count
	 *  \return Number of selected segments.
	 */
	virtual int getNSegs(const Time& t) const;

	/**  Get the number of triggers starting before the specified time  
	 *  currently buffered by the writer.
	 *  \brief Get number of buffered triggers.
	 *  \param t End time for trigger count
	 *  \return Number of selected triggers.
	 */
	virtual int getNTrigs(const Time& t) const;

	/**  Return a constant reference to the current default process
	 *  descriptor.
	 *  \brief Get a reference to the process descriptor.
	 *  \return Constant reference to default process descriptor.
	 */
	const TrigProc& refProcess(void) const;

	/**  Set the debug message level.
	 *  \brief Set debug level
	 *  \param d Debug message level.
	 */
	virtual void setDebug(lmsg::index_type d);

	/**  Set the default process descriptor.
	  *  \brief Set default process
	  *  \param p Constant reference to the TrigProc descriptor
	  *  \return lmsg error code
	  */
	virtual lmsg::error_type setProcess(const TrigProc& p);

	/**  Flush the writer buffers of any stored segments, triggers or
	  *  process entries.
	  *  \brief Flush buffered segments or triggers to output.
	  *  \param file Output file name.
	  *  \param strt Earliest start time of segments/triggers to be written.
	  *  \param end  End time of segments/triggers to be written.
	  *  \return lmsg error code
	  */
	virtual lmsg::error_type write(const std::string& file, 
				       const Time& strt, 
				       const Time& end) const=0;

    protected:
	/**  Return a non-constant reference to the default Process descriptor.
	 *  @memo Get a reference to the process descriptor.
	 *  @return Reference to default process descriptor.
	 */
	TrigProc& refProcess(void);

    private:
	lmsg::index_type mDebug;
	TrigProc mProcess;
    };

} // namespace trig

inline lmsg::error_type
trig::TrigWriter::addTrigger(const TrigBase& t) {
    return addTrigger(t, mProcess);
}

inline lmsg::error_type
trig::TrigWriter::addSegment(const Segment& s) {
    return addSegment(s, mProcess);
}

inline trig::TrigProc&
trig::TrigWriter::refProcess(void) {
    return mProcess;
}

inline const trig::TrigProc&
trig::TrigWriter::refProcess(void) const {
    return mProcess;
}

#endif // TRIG_TRIGWRITER_HH
