/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIG_SBTRIGWRITER_HH
#define TRIG_SBTRIGWRITER_HH

#include "TrigWriter.hh"
#include "Segment.hh"
#include "SBTrigger.hh"
#include "TrigProc.hh"
#include <list>

namespace trig {

    /**  The SBTrigWriter class writes triggers to the LDAS meta-database.
     *  @memo Trigger writer
     *  @author J. Zweizig
     *  @version 1.0; Last Modified September 23, 2005
     */
    class SBTrigWriter : public TrigWriter {
    public:
	/**  Trigger database writer constructor.
	 */
	SBTrigWriter(void);

	/**  Trigger database writer destructor.
	 */
	~SBTrigWriter(void);

	/**  The segment is written to the segment output stream and associated 
	 *  with the specified process ID. In buffered writers, the segment
	 *  (and process ID if unique) are entered in the writer tables until
	 *  the buffer is flushed by a write command.
	 *  \brief memo Write a segment.
	 *  \param s Constant reference to %Segment to be written to the stream.
	 *  \param p Constant reference to descriptor of the process that 
	 *           generated segment \a s.
	 *  \return lmsg error code
	 */ 
	lmsg::error_type addSegment(const Segment& s, const TrigProc& p);

	/**  The specified trigger is added to the list of triggers to be 
	 *  written and is linked to the appropriate entry in the process 
	 *  list. If the process isn't already in the internal process list, 
	 *  it is added to the list.
	 *  @memo Add a trigger to the database.
	 *  @param t Trigger to be added to the list
	 *  @param p Descriptor of the process that generated the trigger.
	 *  @return Always OK.
	 */
	lmsg::error_type addTrigger(const TrigBase& t, const TrigProc& p);

	void clear(const Time& start, const Time& end);

	Time getEarly(void) const;
	TrigWriter::trig_mode getMode(void) const;
	int getNTrigs(const Time& t) const;
	lmsg::error_type setProcess(const TrigProc& p);
	lmsg::error_type write(const std::string& file, const Time& start, 
			       const Time& end) const;

    private:
	typedef std::list<TrigProc>       proc_list;
	typedef proc_list::iterator       proc_iter;
	typedef proc_list::const_iterator const_proc_iter;
	typedef std::list<SBTrigger>      trig_list;
	typedef trig_list::iterator       trig_iter;
	typedef trig_list::const_iterator const_trig_iter;

    private:
	proc_iter insert_proc(const TrigProc& p);

    private:
	int       mNPrcWrt;
	proc_list mPrcList;
	trig_list mTrgList;
    };

    //====================================  Inline methods.
    inline TrigWriter::trig_mode 
    SBTrigWriter::getMode(void) const {
	return kSBTrig;
    }


} // namespace trig

#endif // SBTRIGWRITER_HH
