/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIG_LDASDBWRITER_HH
#define TRIG_LDASDBWRITER_HH

#include "TrigWriter.hh"
#include "Segment.hh"
#include "TrigBase.hh"
#include "TrigProc.hh"
#include <list>

namespace trig {

    /**  The LdasDBWriter class writes triggers to the LDAS meta-database.
     *  @memo Trigger writer
     *  @author J. Zweizig
     *  @version 1.0; Last Modified September 23, 2005
     */
    class LdasDBWriter : public TrigWriter {
    public:
	/**  Trigger database writer constructor.
	 */
	LdasDBWriter(void);

	/**  Trigger database writer destructor.
	 */
	~LdasDBWriter(void);

	/**  The specified trigger is added to the list of triggers to be 
	 *  written and is linked to the appropriate entry in the process 
	 *  list. If the process isn't already in the internal process list, 
	 *  it is added to the list.
	 *  @memo Add a trigger to the database.
	 *  @param t Trigger to be added to the list
	 *  @param p Descriptor of the process that generated the trigger.
	 *  @return Always OK.
	 */
	lmsg::error_type addTrigger(const TrigBase& t, const TrigProc& p);

	/**  The specified segment is added to the list of segments to be 
	 *  written and is linked to the appropriate entry in the process 
	 *  list. If the process isn't already in the internal process list, 
	 *  it is added to the list. Segments  with a zero or negative 
	 *  activity code are ignored.
	 *  @memo Add a segment to the database.
	 *  @param s Segment to be added to the list
	 *  @param p Descriptor of the process that generated the segment.
	 *  @return Always OK.
	 */
	lmsg::error_type addSegment(const Segment& s,  const TrigProc& p);

	void clear(const Time& start, const Time& end);

	Time getEarly(void) const;
	TrigWriter::trig_mode getMode(void) const;
	int getNSegs(const Time& t) const;
	int getNTrigs(const Time& t) const;
	lmsg::error_type setProcess(const TrigProc& p);
	lmsg::error_type write(const std::string& file, const Time& start, 
			       const Time& end) const;

    private:
	typedef std::list<TrigProc>       proc_list;
	typedef proc_list::iterator       proc_iter;
	typedef proc_list::const_iterator const_proc_iter;
	typedef std::list<Segment>        seg_list;
	typedef seg_list::iterator        seg_iter;
	typedef seg_list::const_iterator  const_seg_iter;
	typedef std::list<TrigBase>       trig_list;
	typedef trig_list::iterator       trig_iter;
	typedef trig_list::const_iterator const_trig_iter;

    private:
	proc_iter insert_proc(const TrigProc& p);

    private:
	int       mNPrcWrt;
	proc_list mPrcList;
	seg_list  mSegList;
	trig_list mTrgList;
    };

    //====================================  Inline methods.
    inline TrigWriter::trig_mode 
    LdasDBWriter::getMode(void) const {
	return kWriter;
    }


} // namespace trig

#endif // TRIG_LDASDBWRITER_HH
