#include "hosttable.hh"
#include <stdlib.h>
#include <iostream>
#include <fstream>

using namespace std;
using namespace lmsg;

static const char* wSpace = " \t";

//=======================================  Test for same network
const unsigned long defMask = 0xffff0000;

inline bool
interNet(MsgAddr::ipaddr_t addr) {
    return (addr >> 24) != 10;
}

inline bool
SameNet(MsgAddr::ipaddr_t a, MsgAddr::ipaddr_t b, unsigned long m) {
    if (interNet(a) && interNet(b)) return true;
    return !((a ^ b) & m);
}

inline bool
SameNet(const MsgAddr& a, const MsgAddr& b, unsigned long m) {
    return SameNet(a.getIPAddr(), b.getIPAddr(), m);
}

//======================================  Print an IP address
void
HostEntry::dumpIP(ostream& out, ip_addr addr) {
    out << ((addr >> 24) & 255) << "." 
	<< ((addr >> 16) & 255) << "."  
	<< ((addr >>  8) & 255) << "." 
	<< ( addr        & 255);
}

//======================================  Print an IP address
HostEntry::ip_addr
getIP(const char* & ptr) {
    HostEntry::ip_addr rc(0);
    for (int i=0 ; i<4 ; i++) {
        rc <<= 8;
	rc += strtol(ptr, const_cast<char**>(&ptr), 0);
	if (i==3 || *ptr != '.') break;
	ptr++;
    }
    return rc;
}

//======================================  Construct an empty host entry
HostEntry::HostEntry(void) 
  : mAddr(0), mMask(0)
{
}

//======================================  Construct and set a host entry
HostEntry::HostEntry(const std::string& in) 
  : mAddr(0), mMask(0)
{
    const char* ptr = in.c_str();
    mAddr = getIP(ptr);
    if (*ptr++ == '/') mMask = getIP(ptr);
    return;
}

//======================================  Copy a host entry
HostEntry::HostEntry(const HostEntry& x) 
  : mAddr(x.mAddr), mMask(x.mMask) 
{
}

//======================================  Compare two entries
bool 
HostEntry::operator==(const HostEntry& x) const NOEXCEPT {
    return mAddr == x.mAddr && mMask == x.mMask;
}

//======================================  See if hosts are the same
bool 
HostEntry::isNode(ip_addr x) const NOEXCEPT {
    return x == mAddr;
}

//======================================  Is specified host on the same net?
bool 
HostEntry::sameNet(ip_addr x) const NOEXCEPT {
    return SameNet(x, mAddr, mMask);
}

//======================================  Write the host entry IP and mask
void 
HostEntry::dump(ostream& out) const {
    dumpIP(out, mAddr);
    cout << "/";
    dumpIP(out, mMask);
}

//======================================  Construct an empty host list
HostList::HostList(void) {}

//======================================  Construct a host-linst and fill it.
HostList::HostList(std::string& in) {
    string::size_type len = in.size();
    string::size_type pos = in.find_first_not_of(wSpace);
    while (pos < len) {
	string::size_type lEntry = in.find_first_of(wSpace, pos) - pos;
	mHostVect.push_back(HostEntry(in.substr(pos, lEntry)));
	pos += lEntry;
	if (pos < len) pos = in.find_first_not_of(wSpace, pos);
    }
}

//======================================  Copy the Host list
HostList::HostList(const HostList& x) 
  : mHostVect(x.mHostVect)
{
}

//======================================  Destroy the host list.
HostList::~HostList(void) {
    mHostVect.clear();
}

//======================================  Add a new host entry
void 
HostList::addEntry(const HostEntry& x) {
    mHostVect.push_back(x);
}

//======================================  Test if specified node is in list.
bool 
HostList::isNode(HostEntry::ip_addr x) const {
    for (size_type i=0 ; i<mHostVect.size() ; i++) {
        if (mHostVect[i].isNode(x)) return true;
    }
    return false;
}

//======================================  Get IP address on specified net.
HostEntry::ip_addr 
HostList::netAlias(HostEntry::ip_addr net) const {
    for (size_type i=0 ; i < size() ; i++) {
        if (mHostVect[i].sameNet(net)) return mHostVect[i].getAddr();
    }
    return 0;
}

//======================================  Get number of entries in host list
HostList::size_type 
HostList::size(void) const {
    return mHostVect.size();
}

//======================================  Get the ith host entry.
const HostEntry& 
HostList::operator[](size_type i) const {
    return mHostVect[i];
}

//======================================  Print out the host list.
void 
HostList::dump(ostream& out) const {
    for (size_type i=0 ; i<size() ; i++) {
        if (i) out << " ";
        mHostVect[i].dump(out);
    }
    out << endl;
}

//======================================  Construct an empty host table
HostTable::HostTable(void) 
  : mDebug(false)
{
}

//======================================  Construct and initialize a table
HostTable::HostTable(std::istream& in)  
  : mDebug(false)
{
    setTable(in);
}

//======================================  Construct a table from a file.
HostTable::HostTable(const char* infile)  
  : mDebug(false)
{
    ifstream in(infile);
    setTable(in);
}

//======================================  Set the table from a stream.
void
HostTable::setTable(std::istream& in) {
    char line[1024];
    std::string Alias;
    while (in.getline(line, sizeof(line)).good()) {
        line[in.gcount()] = 0;
        std::string lineString(line);
	size_type lData = lineString.find("#");
	if (lData < lineString.size()) lineString.erase(lData);
	else                           lData = lineString.size();
	if (lData && lineString[lData-1] == '\n') lData--;
	if (lData && lineString[lData-1] == '\\') {
	    Alias += lineString.substr(0, lData-1);
	} else {
	    Alias += lineString;
	    if (!Alias.empty()) {
	        mHostVList.push_back(HostList(Alias));
		Alias.erase();
	    }
	}
    }
    if (mDebug) dump(cout);
}

//=======================================  Find the specified node list.
HostTable::host_iter 
HostTable::find(const MsgAddr& addr) const {
    HostEntry::ip_addr x = addr.getIPAddr();
    for (host_iter i=mHostVList.begin() ; i != end() ; i++) {
        if (i->isNode(x)) return i;
    }
    return end();
}

//=======================================  Number of networks for node.
HostTable::size_type 
HostTable::getNNet(const MsgAddr& req) const {
    host_iter iter = find(req);
    if (iter == end()) return 1;
    return iter->size();
}

//=======================================  Target IP accessible by requestor.
MsgAddr
HostTable::getRoute(const MsgAddr& request, const MsgAddr& target) const {
    MsgAddr alias(target);
    alias.setSubProcess(0);
    if (mDebug) cout << "Resolve address: " << target 
		     << " from: " << request << " Use: ";

    //----------------------------------  See if target is single-net
    size_type nTarg = getNNet(target);
    size_type nReq  = getNNet(request);
    host_iter targ  = find(target); 
    host_iter req   = find(request); 

    if (nTarg == 1 && nReq == 1) {
        if (SameNet(target, request, defMask)) return alias;

    } else if (nTarg == 1) {
        HostEntry::ip_addr aliasIP = req->netAlias(target.getIPAddr());
	if (aliasIP) {
	    if (mDebug) cout << alias << " (multinet requestor)." << endl;
	    return alias;
	}

    } else if (nReq == 1) {
        alias.setIPAddr( targ->netAlias(request.getIPAddr()) );
	if (alias.getIPAddr()) {
	    if (mDebug) cout << alias << " (multinet target)." 
			     << endl;
	    return alias;
	}

    //----------------------------------  Look for multi-net requestor match
    } else {
        for (size_type i=0 ; i<nReq ; i++) {
	    HostEntry::ip_addr aliasIP = targ->netAlias((*req)[i].getAddr());
	    if (aliasIP) {
	        alias.setIPAddr(aliasIP);
		if (mDebug) cout << alias << " (multinet requestor & target)."
				 << endl;
		return alias;
	    }
	}
    }

    //----------------------------------  Look for a tunnel
    for (const_tunnel_iter i=mTunnels.begin() ; i != mTunnels.end() ; i++) {
        if (i->isAccessible(target)) {
	    alias = i->getAddr();
	    alias.setSubProcess(target.getSubProcess());
	    alias = getRoute(request, alias);
	    alias.setSubProcess(target.getSubProcess());
	    if (mDebug) cout << alias << " (via tunnel)." << endl;
	    return alias;
	}
    }

    if (mDebug) cout << "0 (no common network)." << endl;
    return MsgAddr(0);
}

//=======================================  Dump network
void 
HostTable::dump(ostream& out) const {
    for (host_iter i=mHostVList.begin() ; i != end() ; i++) {
        i->dump(out);
    }
}

//=======================================  Add a tunnel.
void 
HostTable::addTunnel(TunnelEntry::ip_addr netip, TunnelEntry::ip_addr netmask,
		     const MsgAddr& addr) {
    mTunnels.push_back(TunnelEntry(netip, netmask, addr));
}

//=======================================  Remove a tunnel from the list.
void 
HostTable::remTunnel(const MsgAddr& addr) {
    for (tunnel_iter i=mTunnels.begin() ; i != mTunnels.end() ; i++) {
        if (i->isTunnel(addr)) {
	    mTunnels.erase(i);
	    return;
	}
    }
}
