#ifndef NAMESERVER_HH
#define NAMESERVER_HH
#include "AppServer.hh"
#include "lmsg/MsgHandler.hh"
#include "NameMsg.hh"
#include "hosttable.hh"
#include "SigFlag.hh"
#include <list>
#include <fstream>

/**  @name Name Server
  *  The name server process allows a server process to register it 
  *  port address with a name and type code. The name server keeps 
  *  a table of equivalent host addresses so that it can translate
  *  an IP address to one on the same network as the client process.
  *
  * {\bf Tunnel interactions:} \\
  *  The Name server may work closely with one or more tunnel processes
  *  to provide communication with processes on disjoint networks. If a 
  *  request is received for a target address that is not reachable by 
  *  the requesting process, the name server searches its list of tunnels 
  *  for one that connects the requester's network to the target network.
  *  If an appropriate tunnel is found, the name server returns the tunnel's
  *  server port address and a gives the index of the target process entry 
  *  as a process ID. The tunnel then routes the request message to the 
  *  target address (resolved by an IDget request) and forwards the
  *  resulting reply to the original sender.
  *
  *  The Name server maintains a list of Tunnel processes. The tunnel process
  *  Registers itself with a name of #"_Tunnel_<net><mask>"# and a process 
  *  type of #lmsg::p_Tunnel#. #<net># and #<mask># are 8-digit hex numbers 
  *  giving the network address and network mask of the Target network.
  *
  * {\bf Running the Name Server:} \\
  *  The name server process is run with the following command line:
\begin{verbatim}
   NameServer [-debug <level>] [-port <ip-port>] [-hosts <table>]
\end{verbatim}
  *  Where the meanings of the flags are summarized in the following table:
  *
  * \begin{tabular}{|l|l|}
  * #-debug <level>#       & Specifies a debug printout level. The default 
  *                          level (0) indicates no debugging messages are 
  *                          to be produced. The message volume increases 
  *                          with the level. \\
  * #-hosts <table>#       & Location of the host table file. \\
  * #-port <ip-port>#      & Specify the name service port [23456]. 
  * \end{tabular}
  *
  * {\Large Name Server Inputs:} \\
  *
  *  {\bf Equivalent Host Table} \\
  *  The "Equivalent Host" table contains a list of know hosts which are 
  *  connected to more than one network. This information is used to 
  *  translate the server IP address information in cases where either 
  *  the target server or the client hosts are connected to more than one
  *  network. The file contains network addresses of the form
  *  #<ip-address>/<net-mask>#. Each line in the equivalent host file 
  *  contains two or more network addresses for the same multiply connected
  *  host.
  *
  * {\bf Messages:} \\
  * The name server accepts messages as defined in NameMsg.hh. The 
  * following message types are handled:
  *
  * \begin{tabular}{|l|l|}
  * #AddName# & The specified process name is registered. A NameStatus
  * message is returned \\
  * #FindName# & The specified process name is looked up in the name
  * server dictionary and a #NameStatus# message is returned. \\
  * #RemoveName# & The specified process name entry is removed from the 
  * name server dictionary. \\
  * #NameIndex# & Request for an index. \\
  * #IDget# & Get the name and address associated with the spcified ID. \\
  * #<other># & Other message types are ignored and no reply is returned.
  *\end{tabular}
  *
  * {\bf Signals:} \\
  * Signals are used to cause asynchronous state changes.
  *
  * \begin{center}
  * \begin{tabular}{|l|l|} 
  * #SIGINT#  & Terminate cleanly. \\
  * #SIGTERM# & Terminate cleanly.
  * \end{tabular}
  * \end{center}
  *
  *  {\Large Name Server Outputs:}
  *
  * {\bf Messages:} \\
  * The trigger manager accepts monitor messages as defined 
  * in NameMsg.hh. The following message types are handled:
  *
  * \begin{tabular}{|l|l|}
  * #Ack# & Reply to a successful remove request. \\
  * #NameStatus# & Message in reply to an #AddName#, #FindName# or
  * NameIndex request containing the name(s) and its address and type. \\
  * #NAck# & Reply to an unsucessful remove request. \\
  *\end{tabular}
  *
  * @memo    DMT Name to Address Translation Service
  * @author  J. Zweizig
  * @version 1.0; Modified April 26, 2001
  */
//@{
//@}

static const short default_server_port = 23456;
using namespace lmsg;

class NameEntry : public NameData {
public:
    NameEntry(const NameData& x);
    int getIdent(void) const;
    void setIdent(int id);
private:
    int mIdent;
};

inline int
NameEntry::getIdent(void) const {
    return mIdent;
}

class NameServer : public AppServer {
  typedef std::list<NameEntry> NameDictionary;
  typedef NameDictionary::iterator dict_iter;

public:
  NameServer(int argc, const char* argv[]);
  ~NameServer(void);
  void ProvideService(void);
  unsigned int getDebug(void) const;
  dict_iter find(const char* name);
  dict_iter findByID(int ID);
  error_type addHandler(const MsgHeader& hdr, AddName& msg);
  error_type remHandler(const MsgHeader& hdr, const RemoveName& msg);
  error_type findHandler(const MsgHeader& hdr, const FindName& msg);
  error_type indexHandler(const MsgHeader& hdr, const NameIndex& msg);
  error_type idGetHandler(const MsgHeader& hdr, const IDget& msg);

private:
  SigFlag         mTerm;
  SigFlag         mPoll;
  unsigned int    mDebug; 
  NameDictionary  mDictionary;
  bool            mActive;
  HostTable       mTable;
  int             mLastIdent;
};

inline unsigned int 
NameServer::getDebug(void) const {
    return mDebug;
}

template<class Msg>
class NameHandler : public MsgHandler, public Msg {
public:
  NameHandler(void);
  ~NameHandler(void);
  error_type handleMsg(AppServer& app, const MsgHeader& hdr);
  Message& getMessage(void);
  const char* getFormat(void) const;
};

template<class Msg>
inline NameHandler<Msg>::~NameHandler(void) 
{
}

template<class Msg>
inline NameHandler<Msg>::NameHandler(void) 
{
}

template<class Msg>
inline Message&
NameHandler<Msg>::getMessage(void) {
    return *this;
}


template<class Msg>
inline const char*
NameHandler<Msg>::getFormat(void) const {
    return "0A";
}

typedef NameHandler<AddName>    HandleNameAdd;
typedef NameHandler<FindName>   HandleNameLookup;
typedef NameHandler<RemoveName> HandleNameRemove;
typedef NameHandler<NameIndex>  HandleNameIndex;
typedef NameHandler<IDget>      HandleIDget;

#endif
