/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "MonDataMsg.hh"
#include "lmsg/TransInput.hh"
#include "lmsg/TransOutput.hh"
#include "DVecType.hh"
#include <iostream>
#include <string.h>

using namespace lmsg;
using namespace std;

using containers::fSeries;

template<class T, unsigned int MsgID>
MonDataMsg<T, MsgID>::MonDataMsg(void) {
}

template<class T, unsigned int MsgID>
MonDataMsg<T, MsgID>::~MonDataMsg(void) {
}

template<class T, unsigned int MsgID>
MonDataMsg<T, MsgID>::MonDataMsg(const char* name, const T& ival) 
  : mName(name), mValue(ival)
{
}

template<class T, unsigned int MsgID>
MonDataMsg<T, MsgID>::MonDataMsg(TransInput& istr) {
    setData(istr);
}

//--------------------------------------  Set data methods
template<>
void
MDM_TSeries::setData(TransInput& istr) {
    istr >> mName;

    string Title;
    Time Start;
    int  nData;
    double dT, F0;
    istr >> Title >> Start >> dT >> F0 >> nData;
    float* dVec = new float[nData];
    istr.read(dVec, nData);
    mValue = TSeries(Start, Interval(dT), nData, dVec);
    mValue.setName(Title.c_str());
    mValue.setF0(F0);
    delete[] dVec;
}

template<>
void
MDM_FSeries::setData(TransInput& istr) {
    istr >> mName;

    string Title;
    Time Start;
    int  nData;
    double F0, dF, dT;
    istr >> Title >> F0 >> dF >> Start >> dT >> nData;
    fComplex* dVec = new fComplex[nData];
    istr.read((float*)dVec, nData*2);
    mValue = FSeries(F0, dF, Start, dT, nData, dVec);
    delete[] dVec;
    mValue.setName(Title.c_str());
}

template<>
void
MDM_FSpectrum::setData(TransInput& istr) {
    istr >> mName;

    string Title;
    Time Start;
    int nData, Count;
    double F0, dF, dT;
    istr >> Title >> F0 >> dF >> Start >> dT >> nData >> Count;
    float* dVec = new float[nData];
    istr.read(dVec, nData);
    mValue = FSpectrum(F0, dF, Start, dT, nData, dVec);
    delete[] dVec;
    mValue.setName(Title.c_str());
    mValue.setCount(Count);
}

template<>
void
MDM_DFT::setData(TransInput& istr) {
    istr >> mName;

    string Title;
    Time Start;
    int  nData;
    double F0, dF, dT;
    istr >> Title >> F0 >> dF >> Start >> dT >> nData;
    DVectC* dVec = new DVectC(nData);
    istr.read(reinterpret_cast<float*>(dVec->refData()), nData*2);
    mValue.setData(F0, dF, dVec);
    mValue.setTimeSpan(Start, dT);
    mValue.setName(Title.c_str());
}

template<>
void
MDM_PSD::setData(TransInput& istr) {
    istr >> mName;

    string Title;
    Time Start;
    int nData, Count;
    double F0, dF, dT;
    istr >> Title >> F0 >> dF >> Start >> dT >> nData >> Count;
    DVectF* dVec = new DVectF(nData);
    istr.read(dVec->refTData(), nData);
    mValue.setData(F0, dF, dVec);
    mValue.setTimeSpan(Start, dT);
    mValue.setName(Title.c_str());
    //mValue.setCount(Count);
}

template<>
void
MDM_ASD::setData(TransInput& istr) {
    istr >> mName;

    string Title;
    Time Start;
    int nData, Count;
    double F0, dF, dT;
    istr >> Title >> F0 >> dF >> Start >> dT >> nData >> Count;
    DVectF* dVec = new DVectF(nData);
    istr.read(dVec->refTData(), nData);
    mValue.setData(F0, dF, dVec);
    mValue.setTimeSpan(Start, dT);
    mValue.setName(Title.c_str());
    //mValue.setCount(Count);
}

template<>
void 
MDM_Histogram1::setData(TransInput& istr) {
  istr >> mName;

  string title, xlabel, nlabel;
  int nbinx,nentries,bintype,errflag;
  Time tstamp;

  istr>>title>>xlabel>>nlabel>>tstamp>>nbinx>>nentries>>bintype>>errflag;

  double* stats = new double[4];
  istr.read(stats,4);
  double* array = new double[nbinx+2];
  istr.read(array,nbinx+2);
  double* xbins = new double[nbinx+1];
  istr.read(xbins,nbinx+1);

  mValue.SetTitle(title.c_str());
  mValue.SetXLabel(xlabel.c_str());
  mValue.SetNLabel(nlabel.c_str());
  mValue.SetTime(tstamp);
  mValue.SetBinLowEdges(nbinx,xbins);
  mValue.SetBinType(bintype);
  mValue.SetBinContents(array);
  mValue.SetNEntries(nentries);
  mValue.PutStats(stats);

  if (errflag) {
    double* error = new double[nbinx+2];
    istr.read(error,nbinx+2);
    mValue.SetBinErrors(error);
  }

}


template<class T, unsigned int MsgID>
void
MonDataMsg<T, MsgID>::setData(TransInput& istr) {
    istr >> mName >> mValue;
}

//--------------------------------------  Get data methods
template<>
void
MDM_TSeries::getData(TransOutput& ostr) const {
    ostr << mName;

    string Title(mValue.getName());
    Time Start = mValue.getStartTime();
    int nData  = mValue.getNSample();
    double dT  = mValue.getTStep();
    double F0  = mValue.getF0();
    ostr << Title << Start << dT << F0 << nData;
    float* dVec = new float[nData];
    mValue.getData(nData, dVec);
    ostr.write(dVec, nData);
    delete[] dVec;
}

template<>
void
MDM_FSeries::getData(TransOutput& ostr) const {
    ostr << mName;

    string Title(mValue.getName());
    Time Start = mValue.getStartTime();
    int nData  = mValue.getNStep() + 1;
    double dT  = mValue.getEndTime() - Start;
    double F0  = mValue.getLowFreq();
    double dF  = mValue.getFStep();
    fComplex* dVec = new fComplex[nData];
    nData = mValue.getData(nData, dVec);
    ostr << Title << F0 << dF << Start << dT << nData;
    ostr.write((float*)dVec, nData*2);
    delete[] dVec;
}

template<>
void
MDM_FSpectrum::getData(TransOutput& ostr) const {
    ostr << mName;

    string Title(mValue.getName());
    Time Start = mValue.getStartTime();
    int nData  = mValue.getNStep() + 1;
    double dT  = mValue.getEndTime() - Start;
    double F0  = mValue.getLowFreq();
    double dF  = mValue.getFStep();
    int Count  = mValue.getCount();
    float* dVec = new float[nData];
    nData = mValue.getData(nData, dVec);
    ostr << Title << F0 << dF << Start << dT << nData << Count;
    ostr.write(dVec, nData);
    delete[] dVec;
}

template<>
void
MDM_DFT::getData(TransOutput& ostr) const {
    ostr << mName;

    string Title(mValue.getName());
    Time Start = mValue.getStartTime();
    int nData  = mValue.size();
    double dT  = mValue.getEndTime() - Start;
    double F0  = mValue.getLowFreq();
    double dF  = mValue.getFStep();
    ostr << Title << F0 << dF << Start << dT << nData;
    if (nData) {
	fComplex* dVec = new fComplex[nData];
	mValue.getData(nData, dVec);
	ostr.write((float*)dVec, nData*2);
	delete[] dVec;
    }
}

template<>
void
MDM_PSD::getData(TransOutput& ostr) const {
    ostr << mName;

    string Title(mValue.getName());
    Time Start = mValue.getStartTime();
    int nData  = mValue.size();
    double dT  = mValue.getEndTime() - Start;
    double F0  = mValue.getLowFreq();
    double dF  = mValue.getFStep();
    //int Count  = mValue.getCount();
    int Count  = 1;
    ostr << Title << F0 << dF << Start << dT << nData << Count;
    if (nData) {
	float* dVec = new float[nData];
	mValue.getData(nData, dVec);
	ostr.write(dVec, nData);
	delete[] dVec;
    }
}

template<>
void
MDM_ASD::getData(TransOutput& ostr) const {
    ostr << mName;

    string Title(mValue.getName());
    Time Start = mValue.getStartTime();
    int nData  = mValue.size();
    double dT  = mValue.getEndTime() - Start;
    double F0  = mValue.getLowFreq();
    double dF  = mValue.getFStep();
    //int Count  = mValue.getCount();
    int Count  = 1;
    ostr << Title << F0 << dF << Start << dT << nData << Count;
    if (nData) {
	float* dVec = new float[nData];
	mValue.getData(nData, dVec);
	ostr.write(dVec, nData);
	delete[] dVec;
    }
}

template<>
void
MDM_Histogram1::getData(TransOutput& ostr) const {
  ostr << mName;

  string title(mValue.GetTitle());
  string xlabel(mValue.GetXLabel());
  string nlabel(mValue.GetNLabel());
  Time tstamp = mValue.GetTime();
  int nbinx = mValue.GetNBins();
  int nentries = mValue.GetNEntries();
  int bintype = mValue.GetBinType();
  int errflag = (int)mValue.IsErrorFlagON();
  double* stats = new double[4];
  mValue.GetStats(stats);
  double* array = new double[nbinx+2];
  mValue.GetBinContents(array);
  double* xbins = new double[nbinx+1];
  mValue.GetBinLowEdges(xbins);

  ostr<<title<<xlabel<<nlabel<<tstamp<<nbinx<<nentries<<bintype<<errflag;
  ostr.write(stats,4);
  ostr.write(array,nbinx+2);
  ostr.write(xbins,nbinx+1);

  if(errflag) {
    double* error = new double[nbinx+2];
    mValue.GetBinErrors(error);
    ostr.write(error,nbinx+2);
    delete[] error;
  }

  delete[] stats;
  delete[] array;
  delete[] xbins;
  
}

template<class T, unsigned int MsgID>
void
MonDataMsg<T, MsgID>::getData(TransOutput& ostr) const {
    ostr << mName << mValue;
}

//--------------------------------------  Get value methods
template<class T, unsigned int MsgID>
T
MonDataMsg<T, MsgID>::getValue(void) const {
    return mValue;
}

template<class T, unsigned int MsgID>
const char*
MonDataMsg<T, MsgID>::getName(void) const {
    return mName.c_str();
}

template<class T, unsigned int MsgID>
size_type 
MonDataMsg<T, MsgID>::size(void) const {
    return sizeof(short)+mName.size()+sizeof(T);
}

template<class T, unsigned int MsgID>
MsgHeader::MsgType
MonDataMsg<T, MsgID>::getType(void) const {
    return MsgID;
}

//--------------------------------------   Request message
MDM_Request::MDM_Request(void) {
}

MDM_Request::MDM_Request(const char* Name) 
  : mName(Name)
{
}

MDM_Request::~MDM_Request(void) {
}

MsgHeader::MsgType
MDM_Request::getType(void) const {
    return mdm_Request;
}

size_type
MDM_Request::size(void) const {
    return sizeof(short) + mName.size();
}

void
MDM_Request::getData(TransOutput& ostr) const {
    ostr << mName;
}

void
MDM_Request::setData(TransInput& istr) {
    istr >> mName;
}

const char*
MDM_Request::getName(void) const {
    return mName.c_str();
}

//--------------------------------------   Index message
MDM_Index::MDM_Index(void) {
}

MDM_Index::MDM_Index(const string& Index) 
  : mIndex(Index)
{
}

MDM_Index::~MDM_Index(void) {
}

MsgHeader::MsgType
MDM_Index::getType(void) const {
    return mdm_Index;
}

size_type
MDM_Index::size(void) const {
    return sizeof(short) + mIndex.size();
}

void
MDM_Index::getIndex(string& x) const {
    x = mIndex;
}

void
MDM_Index::getData(TransOutput& ostr) const {
    ostr << mIndex;
}

void
MDM_Index::setData(TransInput& istr) {
    istr >> mIndex;
}

//--------------------------------------------  Explicit instantiations
template class MonDataMsg<long,       mdm_Int>;
template class MonDataMsg<double,     mdm_Double>;
template class MonDataMsg<string,     mdm_String>;
template class MonDataMsg<TSeries,    mdm_TSeries>;
template class MonDataMsg<FSeries,    mdm_FSeries>;
template class MonDataMsg<FSpectrum,  mdm_FSpectrum>;
template class MonDataMsg<Histogram1, mdm_Histogram1>;

template class MonDataMsg<containers::DFT, mdm_FSeries>;
template class MonDataMsg<containers::PSD, mdm_FSpectrum>;
template class MonDataMsg<containers::ASD, mdm_FSpectrum>;
