/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <unistd.h>
#include "AlarmMgr.hh"
#include "AlarmMsg.hh"
// #include "AlarmTable.hh"
#include "Time.hh"
#include "lmsg/ErrorList.hh"
#include <fstream>
#include "PConfig.h"
#include <sstream>
#include <sys/time.h>
#include <cstdlib>

//--------------------------- Not currently in use
#include "html/color.hh"
#include "html/size.hh"
#include "html/document.hh"
#include "html/text.hh"
#include "html/writer.hh"
#include "html/Table2C.hh"
//------------------------------------------------

using namespace std;
using namespace lmsg;

static const char* AlarmMgrVsn = "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Services/AlarmMgr/AlarmMgr.cc 7582 2016-03-01 22:25:48Z john.zweizig@LIGO.ORG $";

namespace lmsg {
//
//    Alarm Manager message handlers.
//
//    All alarm manager messages are passed on to the appropriate handler
//    method from the AlarmMgr class. Note that handlers assume that the 
//    server application is the alarm manager.
//
template<>
error_type
Handler<ALM_Define>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleDefine(hdr, mMessage);
}

template<>
error_type
Handler<ALM_Set>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleSet(hdr, mMessage);
}

template<>
error_type
Handler<ALM_Request>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleRequest(hdr, mMessage);
}

template<>
error_type
Handler<ALM_Acknowledge>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleAcknowledge(hdr, mMessage);
}

template<>
error_type
Handler<ALM_GetActive>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleGetActive(hdr, mMessage);
}

template<>
error_type
Handler<ALM_GetDefined>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleGetDefined(hdr, mMessage);
}

template<>
error_type
Handler<ALM_Prolong>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleProlong(hdr, mMessage);
}

template<>
error_type
Handler<ALM_Cancel>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleCancel(hdr, mMessage);
}

template<>
error_type
Handler<ALM_SetDisable>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleSetDisable(hdr, mMessage);
}

template<>
error_type
Handler<ALM_GetTemplate>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleGetTemplate(hdr, mMessage);
}

template<>
error_type
Handler<ALM_Remove>::handleMsg(AppServer& app, const MsgHeader& hdr) 
{
    return ((AlarmMgr&) app).HandleRemove(hdr, mMessage);
}

template class Handler< ALM_Define >;
template class Handler< ALM_Set >;
template class Handler< ALM_Request >;
template class Handler< ALM_Acknowledge >;
template class Handler< ALM_GetActive >;
template class Handler< ALM_GetDefined >;
template class Handler< ALM_Prolong >;
template class Handler< ALM_Cancel >;
template class Handler< ALM_SetDisable >;
template class Handler< ALM_GetTemplate >;
template class Handler< ALM_Remove >;
}

//
//    I/O functions
//
static ostream&
operator<<(ostream& out, const AlarmID& a) {
    return out << a.getMonitor() << ":" << a.getAlarm();
}

//
//    Main function of the alarm manager. Construct it and run it.
//
int
main(int argc, const char* argv[]) {
    AlarmMgr x(argc, argv);
    return x.Application();
}

//
//    Alarm manager constructor.
//
//    The only constructor passes the command line argument list as 
//    arguments in the usual form. The constructor first parses the 
//    command line and then sets up message and signal handling.
//
AlarmMgr::AlarmMgr(int argc, const char* argv[]) 
  : AppServer(AlarmServerName, o_register, p_Server), mTimer(10.0),
    mHack(0), mTerm(SIGTERM), mNext(0, 0)
{
    cout << "Starting alarm manager version: " << AlarmMgrVsn << endl;

    //----------------------------------  Set default values

    //----------------------------------  Parse arguments
    for (int i=1 ; i<argc ; i++) {
        string argi(argv[i]);
        if (argi == "-debug") {
	    if (i >= argc-1)            setDebug(1);
	    else if (*argv[i+1] == '-') setDebug(1);
	    else                        setDebug(strtol(argv[++i],0,0));
	} else {
	    cerr << "Unrecognized argument: " << argv[i] << endl;
	    cerr << "Command syntax is as follows:" << endl;
	    cerr << argv[0] << " [-debug [<level>]] " << endl;
	    mTerm.setSig0();
	    return;
	}
    }

    // mCTable.readFile(accessTable.c_str());
    // mCTable.reset();

    //----------------------------------  Define message handlers
    addHandler(almDefine,      new Handler<ALM_Define>);
    addHandler(almSet,         new Handler<ALM_Set>);
    addHandler(almAcknowledge, new Handler<ALM_Acknowledge>);
    addHandler(almGetActive,   new Handler<ALM_GetActive>);
    addHandler(almGetDefined,  new Handler<ALM_GetDefined>);
    addHandler(almRequest,     new Handler<ALM_Request>);
    addHandler(almProlong,     new Handler<ALM_Prolong>);
    addHandler(almCancel,      new Handler<ALM_Cancel>);
    addHandler(almSetDisable,  new Handler<ALM_SetDisable>);
    addHandler(almGetTemplate, new Handler<ALM_GetTemplate>);
    addHandler(almRemove,      new Handler<ALM_Remove>);

    //----------------------------------  Define Signal handlers
    mTerm.add(SIGINT);
    mForce.add(SIGHUP);
    mForce.add(SIGALRM);

    //----------------------------------  Start the timer
    setTimer();
}

//======================================  Alarm manager destructor
AlarmMgr::~AlarmMgr(void) {
    close();
}

//======================================  Set the timer
void 
AlarmMgr::setTimer(void) {
    itimerval time;
    long nsec = long(mTimer);
    time.it_interval.tv_sec  = 0;
    time.it_interval.tv_usec = 0;
    time.it_value.tv_sec  = nsec;
    time.it_value.tv_usec = long((mTimer - nsec)*1000000);
    setitimer(ITIMER_REAL, &time, 0);
}

//======================================  Main alarm manager function
int 
AlarmMgr::Application(void) {
    while (!mTerm) {
        //------------------------------  Check that I'm registered
        if (!isRegistered()) {
	    error_type rc = register_name(AlarmServerName, p_Server);
	    if (rc) cerr << "Alarm Manager not registered!" << endl;
	}

        //------------------------------  See if an alarm has expired
	mNow.set();
	if (mNext != Time(0,0) && mNext < mNow.getTime()) {
	    Expire(mNow.getTime());
	    continue;
	}

        //------------------------------  Wait for/handle a message
 	wtime_type tWait;
 	if (!mNext) {
 	    tWait = -1;
 	    if (getDebug() > 2) 
 	        cout << "AlarmMgr waiting indefinitely for a message" << endl;
 	} else {
 	    tWait = mNext - mNow.getTime();
 	    if (getDebug() > 2) cout << "AlarmMgr waiting " << tWait 
 				     << " for a message" << endl;
 	}

	if (waitMsg(tWait)) {
 	    if (getDebug() > 2) cout << "AlarmMgr received a message" << endl;
	    mNow.set();
            try {
	        error_type rc = handleMessage();
		if (rc != OK && rc != Continue) {
		    cerr << mNow << "  Error " << rc << " in handleMessage()" 
			 << endl;
		}
	    } catch (exception& e) {
	        cerr << mNow << "Exception handling message: " << e.what() 
		     << endl;
	    }
	}

	//------------------------------  Handle watchdog if alarmed
	if (mForce) {
	}
    }
    return 0;
}

//======================================  Handle a registration request
void
AlarmMgr::Expire(const Time& now) 
{
    mNext = Time(0);
    activ_iter next;
    for (activ_iter i=mActiveDB.begin() ; i != mActiveDB.end() ; i=next ) {
        next = i;
	next++;
	Time expiry = i->second.getExpire();
        if (expiry <= now) {
	    if (getDebug()) {
	        cout << mNow << "  Alarm: " 
		     << static_cast<AlarmID&>(i->second)
		     << " expired." << endl;
	    }
	    mActiveDB.erase(i);
	} else if (!mNext || expiry < mNext) {
	    mNext = expiry;
	}
    }
}

//======================================  Handle a registration request
error_type
AlarmMgr::HandleAcknowledge(const MsgHeader& hdr, const ALM_Acknowledge& msg) 
{
    activ_iter i = mActiveDB.find( msg.refData() );
    if (getDebug()) cout << mNow << "  Alarm: " 
			 << static_cast<AlarmID&>(i->second)
			 << " acknowledged." << endl;
    i->second.setAcknowledge(true);
    return reply(hdr, ALM_Ack());
}

//======================================  Handle a registration request
error_type
AlarmMgr::HandleCancel(const MsgHeader& hdr, const ALM_Cancel& msg) 
{
    activ_iter i = mActiveDB.find( msg.refData() );
    if (i == mActiveDB.end()) {
      cout << mNow << "  Alarm handle " << msg.refData().getInt()
	   << " not found to cancel." << endl;
        return reply(hdr, ALM_Ack());
    } else if (getDebug()) {
        cout << mNow << "  Alarm: " << static_cast<AlarmID&>(i->second)
	     << " canceled." << endl;
    }
    mActiveDB.erase( i );
    return reply(hdr, ALM_Ack());
}

//======================================  Handle an alarm definition message
error_type
AlarmMgr::HandleDefine(const MsgHeader& hdr, const ALM_Define& msg) 
{
    const_alarm_iter i = findAlarm(msg.refData());
    if (i != mAlarmDB.end()) {
        cout << mNow << "  Warning: Alarm " << static_cast<const AlarmID&>(*i)
	     << " was not redefined." << endl;
    } else {
        mAlarmDB.push_back(msg.refData());
        if (getDebug()) {
	    cout << mNow << "  Alarm " 
		 << static_cast<const AlarmID&>(mAlarmDB.back())
		 << " defined." << endl;
	}
    }
    return reply(hdr, ALM_Ack());
}

//======================================  Handle an active alarm request
error_type
AlarmMgr::HandleGetActive(const MsgHeader& hdr, const ALM_GetActive& msg) 
{
    vector<AlarmHandle> aList;
    for (activ_iter i=mActiveDB.begin() ; i != mActiveDB.end() ; i++ ) {
        if (i->second.match(msg.refData())) aList.push_back(i->first);
    }
    // note a bug in gcc 2.95.3 generates a warning if I use return reply(...);
    error_type rc = reply(hdr, ALM_ActiveReply(aList));
    return rc;
}

//======================================  Handle a get defined request.
error_type
AlarmMgr::HandleGetDefined(const MsgHeader& hdr, const ALM_GetDefined& msg) 
{
    stringstream aList;
    bool first(true);
    for (const_alarm_iter i=mAlarmDB.begin() ; i != mAlarmDB.end() ; i++ ) {
        if (i->match(msg.refData())) {
	    if (!first) aList << " ";
	    else        first = false;
	    aList << static_cast<const AlarmID&>(*i);
	}
    }
    if (first) aList << " ";
    aList << ends;
    // note a bug in gcc 2.95.3 generates a warning if I use return reply(...)
    error_type rc = reply(hdr, ALM_DefinedReply(aList.str()));
    return rc;
}

//======================================  Handle a registration request
error_type
AlarmMgr::HandleGetTemplate(const MsgHeader& hdr, const ALM_GetTemplate& msg) 
{
    alarm_iter iter=findAlarm(msg.refData());
    if (iter == mAlarmDB.end()) return reply(hdr, ALM_NAck());
    error_type rc = reply(hdr, ALM_Status(*iter));
    return rc;
} 

//======================================  Add a alarm to the database.
error_type
AlarmMgr::HandleProlong(const MsgHeader& hdr, const ALM_Prolong& msg) {
    AlarmHandle myHandle( msg.refData().first );
    activ_iter i = mActiveDB.find( myHandle );
    if (i == mActiveDB.end())  {
        if (getDebug()) cout << mNow << " Prolongation failed, handle ["
			     << myHandle.getInt() << "] not active." << endl;
        return reply(hdr, ALM_NAck() );
    }
    if (getDebug()) cout << mNow << "  Alarm: " 
			 << static_cast<AlarmID&>(i->second)
			 << "[" << myHandle.getInt() << "] prolonged by " 
			 << msg.refData().second << " seconds" << endl;
    i->second.prolong( msg.refData().second );
    return reply(hdr, ALM_Ack() );
}

//======================================  Handle a registration request
error_type
AlarmMgr::HandleRemove(const MsgHeader& hdr, const ALM_Remove& msg) 
{
    alarm_iter iter=findAlarm(msg.refData());
    if (iter == mAlarmDB.end()) return reply(hdr, ALM_NAck());
    mAlarmDB.erase(iter);
    error_type rc = reply(hdr, ALM_Ack());
    return rc;
} 

//======================================  Add a alarm to the database.
error_type
AlarmMgr::HandleRequest(const MsgHeader& hdr, const ALM_Request& msg) {
    AlarmHandle myHandle( msg.refData() );
    return reply(hdr, ALM_Status(mActiveDB[myHandle]));
}

//======================================  Log an action
void 
AlarmMgr::logAction(const string& action, const const_activ_iter& it) {
    cout << mNow << " " << static_cast<const AlarmID&>(it->second) << "[" 
	 << it->first.getInt() << "] " << action << " time(" 
	 << it->second.getTimeout() << ") param(" 
	 << it->second.getParameters() << ")" << endl;
}

//======================================  Add a alarm to the database.
error_type
AlarmMgr::HandleSet(const MsgHeader& hdr, const ALM_Set& msg) {

    //----------------------------------  Get the alarm definition
    const AlarmData& request(msg.refData());
    const_alarm_iter a = findAlarm(request);
    if (a == mAlarmDB.end()) {
        if (getDebug()) cout << mNow << " Alarm " 
			     << static_cast<const AlarmID&>(request)
			     << " is not defined." << endl;
	return reply(hdr, ALM_Reference(AlarmHandle()));
    } else if ((a->getFlags() & AlarmData::kDisable) != 0) {
        if (getDebug()) cout << mNow << " Alarm " 
			     << static_cast<const AlarmID&>(request)
			     << " is disabled." << endl;
	return reply(hdr, ALM_Reference(AlarmHandle()));
    }

    //----------------------------------  Build up a composite
    AlarmData myAlarm(*a);
    myAlarm.setStartTime(mNow.getTime());
    if (request.getTimeout()) myAlarm.setTimeout(request.getTimeout());
    if (!myAlarm.getTimeout()) {
        cout << mNow << " Alarm " << static_cast<const AlarmID&>(request)
	     << " has zero duration." << endl;
	return reply(hdr, ALM_Reference(AlarmHandle()));
    }
    myAlarm.minSeverity(request.getSeverity());
    if (!myAlarm.getSeverity()) {
        cout << mNow << " Alarm " << static_cast<const AlarmID&>(request)
	     << " has zero severity." << endl;
	return reply(hdr, ALM_Reference(AlarmHandle()));
    }
    myAlarm.jamFlags(request.getFlags());
    const char* desc = request.getDescription();
    if (*desc) myAlarm.setDescription(desc);
    const char* param = request.getParameters();
    if (*param) myAlarm.setParameters(param);

    //----------------------------------  Check for existing alarm if retrigger
    AlarmHandle myHandle;
    if ((myAlarm.getFlags() & AlarmData::kReTrigger)) {
        for (activ_iter i=mActiveDB.begin(); i != mActiveDB.end(); ++i) {
	    if (i->second == myAlarm) {
	        myHandle = i->first;
		Interval tDiff = myAlarm.getExpire() - i->second.getExpire();
		if (tDiff > Interval(0.0)) i->second.prolong(tDiff);
		if (getDebug()) logAction("retriggered", i);
	        break;
	    }
	}
    }

    //----------------------------------  Add to database and update stats
    if (!myHandle) {
        myHandle = AlarmHandle::getUnique();
	mActiveDB[myHandle] = myAlarm;
	logAction("added", mActiveDB.find(myHandle));
    }

    //----------------------------------  Update next expitration time.
    Time expires(myAlarm.getExpire());
    if (!mNext || expires < mNext) mNext = expires;
    return reply(hdr, ALM_Reference(myHandle));
}

//======================================  Handle a registration request
error_type
AlarmMgr::HandleSetDisable(const MsgHeader& hdr, const ALM_SetDisable& msg) 
{
    alarm_iter iter=findAlarm(msg.refData().first);
    if (iter == mAlarmDB.end()) return reply(hdr, ALM_NAck());
    if (msg.refData().second) {
        iter->jamFlags  (AlarmData::kDisable);
	cout << mNow << "Disabled alarm: " << static_cast<AlarmID&>(*iter) 
	     << endl;
    } else {
        iter->clearFlags(AlarmData::kDisable);
	cout << mNow << "Enabled alarm: " << static_cast<AlarmID&>(*iter) 
	     << endl;
    }
    error_type rc = reply(hdr, ALM_Ack());
    return rc;
} 

//======================================  Write out process statistics
void
AlarmMgr::writeStats(void) const {

    //----------------------------------  Build the file name
    string ofile;
    const char* dir = getenv("DMTHTMLOUT");
    if (dir) {
        ofile = dir;
	ofile += "/";
    }
    ofile += "AlarmMgr.html";

    //----------------------------------  Build an html document
    html::document doc("Alarm statistics");
    char time[32];
    doc.setBackgroundColor(html::color("white"));
    LocalStr(Now(), time, "%M %d, %Y %02H:%02N");
    html::text hdr(string("Alarm statistics at: ") + time);
    hdr.setSize(html::size(2));
    html::block cb("center");
    cb.addObject(hdr);
    doc.addObject(cb);

    html::Table2C parTbl("AlarmMgr Parameters");
    // parTbl.addRow("Timer", double(mTimer));
    doc.addObject(parTbl);

    //----------------------------------  Add the Alarm statistics table
    // doc.addObject(mCTable.makeHtmlTable());

    //----------------------------------  Write the document
    ofstream f(ofile.c_str());
    html::writer w(f);
    doc.write(w);
}

//=====================================  Find alarm in the definition database
AlarmMgr::const_alarm_iter
AlarmMgr::findAlarm(const AlarmID& a) const {
    for (const_alarm_iter i=mAlarmDB.begin() ; i != mAlarmDB.end() ; i++) {
        if (static_cast<const AlarmID&>(*i) == a) return i;
    }
    return mAlarmDB.end();
}

AlarmMgr::alarm_iter
AlarmMgr::findAlarm(const AlarmID& a) {
    for (alarm_iter i=mAlarmDB.begin() ; i != mAlarmDB.end() ; i++) {
        if (static_cast<const AlarmID&>(*i) == a) return i;
    }
    return mAlarmDB.end();
}
