#ifndef CHINTERFACE_HH
#define CHINTERFACE_HH
#include <string>
#include <map>
#include <vector>
#include "TSeries.hh"
#include "Time.hh"
#include "Interval.hh"
#include "Histogram1.hh"
#include "DaccAPI.hh"

#include "chConstructs.hh"

namespace channel {
/** @name Channel Interfacing class
  * The chInterface class is designed to ease the production of DMT Monitors
  * through a variety of means:
  * \begin{enumerate}
  *   \item It contains a standard data structure for each channel.
  *   \item It allows for multiple filters and sequences of filtering schemes.
  *   \item It allows for multiple user-defined event-finding schemes.
  *   \item It allows for multiple user-defined statistics-calculating schemes.
  *   \item It allows for multiple user-defined handling of results.
  * \end{enumerate}
  * By creating a chInterface instance for each channel, one is able to iterate
  * as needed through the channels and collect the desired data from each
  * channel.  A DMT monitor could look for correlations among channels in this
  * way without having the normal overhead due to administration. 
  * @memo An easier way to manage channels.
  * @author Rauha Rahkola
  * @version 0.1 May 29th, 2002
  ***************************************************************************/
//@{

/** @name Channel Constructs
  * Essential chInterface components
  * @memo chInterface components
  * @author Rauha Rahkola
  * @version 0.1 May 29th, 2002
  ***************************************************************************/
//@{
//@Include: chConstructs.doc
//@}

/** So the goal here is to be able to create a generic channel interface class
  * which can be used to create a bank of filtered data and apply various
  * glitch and statistics definitions to them, then format the results for
  * some legible output.  For example, let's assume we have:
  * \begin{itemize}
  *   \item an event-finding scheme #MyGlitch : GlitchConstruct glitch#,
  *   \item a statistics scheme     #MyStats : StatisticsConstruct stats#,
  *   \item and a kind of result    #MyResult : ResultConstruct results#.
  * \end{itemize}
  * Then for each #chInterface *ch#,
  * \begin{verbatim}
  ch->addFilter("myFilter", z,p,k);
  ch->addGlitch("myGlitch", glitch);
  ch->addStatistic("myStat", stats);
  ch->addResultType("myResult", results, true);
  ch->applyFilter(filterID);
  for (FBciter i = ch->mFBank.begin(); i != ch->mFBank.end(); i++) {
    ch->logResult( resultID_1, ch->applyStatistic(statID, i) );
    ch->logResult( resultID_1, ch->applyGlitch(glitchID, i) );
    ch->triggerResult( resultID_2, ch->applyGlitch(glitchID, i) );
  }
  \end{verbatim}
  * In addition, we would like to be able to support "real-time" updating,
  * so we should have "reset" and "copy" functions, which will destroy a
  * channel and copy a channel configuration to another channel, respectively.
  *
  * Finally, we would like to be able to configure the output for log files
  * such that users can define their own output format and understand them.
  * @memo general Interface class for DMT monitors to keep track of channels
  * @author Rauha Rahkola
  * @version 0.1 May 29th, 2002
  *********************************************************************/
class chInterface
{
public:
  /*
   * Typedefs
   */
  /// Each channel can make use of multiple filters.
  typedef std::vector<FilterStatus> FilterBank;
  /// A pointer will go through each filter in turn, checking for glitches.
  typedef FilterBank::const_iterator FBciter;

private:
  /*
   * Typedefs
   */
  /// A pointer will go through each filter in turn, checking for glitches.
  typedef FilterBank::iterator FBiter;
  /// Each channel can set up various means to determine glitches
  typedef std::map<std::string,GlitchConstruct *> GlitchBank;
  /// A pointer will go through each mapped GlitchConstruct;
  typedef GlitchBank::const_iterator GBciter;
  /// A pointer will go through each mapped GlitchConstruct;
  typedef GlitchBank::iterator GBiter;
  /// Each channel can set up various statistics to keep track of
  typedef std::map<std::string, StatisticsConstruct *> StatBank;
  /// A pointer will go through each mapped StatisticsConstruct;
  typedef StatBank::const_iterator SBciter;
  /// A pointer will go through each mapped StatisticsConstruct;
  typedef StatBank::iterator SBiter;
  /// Each channel can set up various log parameters & formats to output
  typedef std::map<std::string, ResultConstruct *> ResultBank;
  /// A pointer will go through each mapped ResultConstruct;
  typedef ResultBank::const_iterator RBciter;
  /// A pointer will go through each mapped ResultConstruct;
  typedef ResultBank::iterator RBiter;

  /*
   * Channel identifiers/parameters
   */
  /// channel name
  std::string mName;
  /// sampling frequency
  unsigned int mSampleFreq;
  /// which Operating State Condition (if any) should be satisfied
  std::string mOSCSatisfy;
  /// debug status
  unsigned int mDebug;
  /// initialized status
  bool mInit;

  /*
   * Channel structures
   */
  /// A map of glitch definitions and glitch status variables
  GlitchBank mGBank;
  /// A list of current glitch results
  std::list<glitchResult> mGResults;
  /// A map of statistics definitions and statistics status variables
  StatBank   mSBank;
  /// A list of current statistics results
  std::list<statResult>   mSResults;
  /// A map of result definitions and result structures
  ResultBank mRBank;

  /* 
   * Channel data
   */
  /// Pointer to Raw channel data;
  TSeries    *mData;
  /// Pointer to Dacc data accessor (doesn't need allocation)
  DaccAPI *mDaccPtr;
  /// Temporary time holder
  Time        mTmpTime;

  /*
   * Private functions
   */
  /// references a FilterStatus object (by name)-- could also just use FBank[i]
  FilterStatus* refFilter(const std::string& filterID) {
    //NOTE: Commented the following code because it generates a "cannot
    //NOTE: convert type iter to FilterStatus*" error.
    // for( FBiter i=FBank.begin(); i != FBank.end(); i++ )
    // if( i->getName() == filterID ) return i;
    for( unsigned int i=0; i < FBank.size(); i++ )
      if( FBank[i].getName() == filterID ) return &FBank[i];
    return NULL;
  }

public:
  /*
   * Note: FBank needs to be accessible, as most uses of chInterface will
   * involve cycling over mFBank items.  That's why the other banks are 'maps'
   * and FBank is a 'vector'.
   */
  /// A vector of filters to be applied to a particular channel
  FilterBank FBank;


 //*****************************
 //  standard class functions
 //*****************************
  /// Create a chInterface instance
  chInterface(unsigned int debug=0);
  
  /// Create a chInterface instance with some preset parameters
  explicit chInterface(std::string name, std::string osc="", DaccAPI *dacc=0,
		       unsigned int debug=0);

  /// Create a chInterface instance, using another chInterface
  chInterface(const chInterface& ch);
  
  /// Destroy a chInterface instance
  ~chInterface(void);

  /// copy a chInterface instance
  chInterface& operator = (const chInterface& ch);


 //*****************************
 //  chInterface modifiers
 //*****************************
  /// sets the name of the channel interface
  void setName(std::string name) { mName = name; }

  /// sets the operating state condition required to be satisfied
  void setOSC(std::string osc) { if (osc.size() > 3) mOSCSatisfy = osc; }

  /// points chInterface to the proper data accessor
  bool setDacc(DaccAPI *dacc) {
    bool dacc_return(false);
    mDaccPtr = dacc;
    try           { mDaccPtr->setDebug(mDebug); }
    catch (int n) { dacc_return |= n; }
    return dacc_return;
  }

  /// sets the sample rate of the channel
  void setSampleRate(unsigned int rate) {
    mSampleFreq = rate;
    for (FBiter i=FBank.begin(); i != FBank.end(); i++) i->setSampleRate(rate);
  }

 //*****************************
 //  chInterface accessors
 //*****************************
  /// returns the name of the channel interface
  const std::string getName(void) const { return mName; }

  /// returns the name of the relevant operating state condition
  const std::string getOSC(void) const { return mOSCSatisfy; }

  /// returns a reference to the TSeries data
  TSeries ** getData(void) {
    //    if (!mData) mData = new TSeries();
    return &mData;
  }


 //******************************
 //  chInterface miscellaneous functions
 //******************************
  /// initializes any unset parameters in filters/glitches/statistics/results
  bool Init(void);

  /// Is the new channel data a continuation of the previous data
  bool isTSContinuous(void);

  /// Is the chInterface fully initialized?
  bool isInitialized(void) const { return mInit; }


 //*****************************
 //  chInterface functions related to filters
 //*****************************
  /// references a FilterStatus object (by name)-- could also just use FBank[i]
  const FilterStatus& refFilter(const std::string& filterID) const {
    for( FBciter i=FBank.begin(); i != FBank.end(); i++ )
      if( i->getName() == filterID ) return *i;
    throw;
  }

  /// adds a FilterStatus object using vector poles and zeroes
  void addFilter(const std::string& filterID, std::vector<dComplex>* poles, 
		 std::vector<dComplex>* zeroes, double gain=1.0,
		 const std::string& useFilterID="");

  /// adds a FilterStatus object by reading poles and zeroes from char*'s
  void addFilter( const std::string& filterID, const char *pstr,
		  const char *zstr,double gain=1.0,
		  const std::string& useFilterID="" );

  /// applies (one or all) filters to the channel data
  void applyFilters(const std::string& filterID="");

  /// resets (one or all) FilterStatus objects
  bool ResetFilters(const std::string& filterID="");


 //*****************************
 //  chInterface functions related to glitches
 //*****************************
  /// references a GlitchConstruct object (by name)
  GlitchConstruct* refGlitch(const std::string& glitchID) {
    return (mGBank.find(glitchID) != mGBank.end()) ?
      mGBank.find(glitchID)->second : NULL;
  }

  /// adds a GlitchConstruct-- defaults to base class definition of glitch
  void addGlitch(const std::string& glitchID);

  /// adds a GlitchConstruct-- for adding your own derived glitch construct
  void addGlitch(const std::string& glitchID, GlitchConstruct& glitch);

  /// applies all glitch-tests to one or more (possibly filtered) data
  const std::list<glitchResult>* applyGlitch(const std::string& filterID="");

  /// applies a single glitch-test to (possibly filtered) data
  const std::list<glitchResult>*
  applyGlitch(const std::string& glitchID, FilterStatus& filter );

  /** dumps formatted output of (one or more) glitches
    * @param out   the ostream to direct output to
    * @param total the number of results to dump, or '0' for all of them
    */
  void dumpGlitches( std::ostream& out, unsigned int total=0 );

//    /// resets (one or all) GlitchConstruct objects
//    bool ResetGlitches(std::string glitchID="");


 //*****************************
 //  chInterface functions related to statistics
 //*****************************
  /// references a StatisticsConstruct object (by name)
  StatisticsConstruct* refStats(const std::string& statID) {
    return (mSBank.find(statID) != mSBank.end()) ?
      mSBank.find(statID)->second : NULL;
  }

  /** adds a StatisticsConstruct-- defaults to base-class definition of
    * a statistic (calculates the rms of (possibly-filtered) data)
    */
  void addStatistic(const std::string& statID, bool cumulative=false);

  /// adds a StatisticsConstruct-- for adding your own derived statistics
  void addStatistic(const std::string& statID, StatisticsConstruct& statistic);

  /// applies all statistics to one or more (possibly filtered) data
  const std::list<statResult>* applyStats(const std::string& filterID="");

  /// applies a single statistics construct to (possibly filtered) data
  const std::list<statResult>*
  applyStats(const std::string& statID, FilterStatus& filter );

  /** dumps formatted output of (one or more) statistics
    * @param out   the ostream to direct output to
    * @param total the number of results to dump, or '0' for all of them
    */
  void dumpStats( std::ostream& out, unsigned int total=0 );

  /// resets (one or all) StatisticsConstruct objects
  bool ResetStats(const std::string& statID="", bool cumulative=false);


 //*****************************
 //  chInterface functions related to results
 //*****************************
  /// references a ResultConstruct object (by name)
  ResultConstruct* refResult(const std::string& resultID) {
    return (mRBank.find(resultID) != mRBank.end()) ?
      mRBank.find(resultID)->second : NULL;
  }

  /// adds a ResultConstruct object to the list
  void addResult(const std::string& resultID, bool cumulative=false);

  /// adds a ResultConstruct-- for adding your own derived results
  void addResult(const std::string& resultID, ResultConstruct& result);

  /** logs the result of applying (one or all) glitches to (possibly filtered)
    * data
    * @param resultID the result's identifying string, or "" to use all
    *   results with the same parameters
    * @param results  a list of glitchResult structures to be iterated through
    * @param outmask  a bitmask for redirect of the output.  Bits are:
    *   \begin{tabular}{cl}
    *   #0#   & defaults to internally-defined mask\\
    *   #2^0# & standard output\\
    *   #2^1# & output to internally-defined ostream\\
    *   #2^2# & output to ostream 'out' in function call\\
    *   #2^3# & output to Trigger Manager\\
    *   #2^4# & output to DMTViewer\\
    *   #2^5# & output to trend files\\
    *   #2^6# & signals ResultConstruct to "dump" its contents (subject to the
    *           wishes of any derived class).  Note that this bit will only get
    *           passed to the ResultConstruct when it writes the last
    *           glitchResult
    *   \end{tabular}
    *   When no bits are set, results will be processed internally.
    * @param out      an ostream to be used for output, if bit #2^2# is set
    *   in 'outmask'
    * @param ignorestate a flag whether or not to ignore the state of the
    *   glitchResults.  This would be helpful, e.g., for a dump when the
    *   parent object is destroyed.
    * @return true if all output was successful, false if one or more output
    *   channels was unsuccessful.
    */
  bool writeResult(const std::string& resultID,
                   const std::list<glitchResult>& results,
		   unsigned int outmask=0, std::ostream& out=std::cout,
		   bool ignorestate=false);

  /** logs the result of applying (one or all) statistics to (possibly
    * filtered) data
    * @param resultID the result's identifying string, or "" to use all
    *   results with the same parameters
    * @param results  a list of statResult structures to be iterated through
    * @param outmask  a bitmask for redirect of the output.  Bits are:
    *   \begin{tabular}{cl}
    *   #0#   & defaults to internally-defined mask\\
    *   #2^0# & standard output\\
    *   #2^1# & output to internally-defined ostream\\
    *   #2^2# & output to ostream 'out' in function call\\
    *   #2^3# & output to Trigger Manager\\
    *   #2^4# & output to DMTViewer\\
    *   #2^5# & output to trend files\\
    *   #2^6# & signals ResultConstruct to "dump" its contents (subject to the
    *           wishes of any derived class).  Note that this bit will only get
    *           passed to the ResultConstruct when it writes the last
    *           statResult
    *   \end{tabular}
    *   When no bits are set, results will be processed internally.
    * @param out      an ostream to be used for output, if bit #2^2# is set
    *   in 'outmask'
    * @param ignorestate a flag whether or not to ignore the state of the
    *   statResults.  This would be helpful, e.g., for a dump when the
    *   parent object is destroyed.
    * @return true if all output was successful, false if one or more output
    *   channels was unsuccessful.
    */
  bool writeResult(const std::string& resultID,
		   const std::list<statResult>& results,
		   unsigned int outmask=0, std::ostream& out=std::cout,
		   bool ignorestate=false);

  /** dumps formatted output of (one or more) results
    * @param out   the ostream to direct output to
    * @param total the number of results to dump, or '0' for all of them
    */
  void dumpResults( std::ostream& out, unsigned int total=0);

  /// resets (one or all) ResultConstruct objects
  bool ResetResults(const std::string& resultID="", bool cumulative=false);
}; // end of chInterface class declaration

} // end of namespace channel
//@}

#endif // #ifndef CHINTERFACE_HH
