/************************************************************************
 * The following is a collection of wavelet-analysis related routines
 * @ MAR-1998 IoK, EK
 * @ JUL-2001 - ported to C
 * @ APR-2003 - ported to C++  GJM
 ************************************************************************/

/************************************************************************
 * Top-level call:
 * array : Holds the (input) data, and upon execution the wavelet (output) data
 * size  : size of the array
 * filter: Controls which filter to use
 *         1 -> Haar (or D-2) filter
 *         2 -> Daubechies  4-tap compact support wavelet filter
 *         3 -> Daubechies  8-tap compact support wavelet filter
 *         4 -> Daubechies 16-tap compact support wavelet filter
 *         Anything else -> Does nothing
 *         Rule of thumb : 2^filter is the filter support
 * direction: 1 -> Forward (or decomposition) transform
 *           -1 -> Inverse (or reconstruction) transform
 *           Anything else -> Does nothing
 ************************************************************************/
#include "mywv2lib.hh"

Wavelet::Wavelet (int samples) :
  sqrt2(1.41421356237309504880168872420969808),
  sqrt1_2(0.707106781186547524400844362104849039)
  
{
  size = samples;
  totalScales = getScale(size);
  rawarray = new double[size];
  wavearray = new double[size];
}

Wavelet::Wavelet () :
  sqrt2(1.41421356237309504880168872420969808),
  sqrt1_2(0.707106781186547524400844362104849039)
{
  // create some placeholder pointers to make things consistent
  wavearray = new double[2];    
  rawarray = new double[2];
}

Wavelet::Wavelet (double * inputdata, int samples, char datatype) :
  sqrt2(1.41421356237309504880168872420969808),
  sqrt1_2(0.707106781186547524400844362104849039)
  
{
  size = samples;
  totalScales = getScale(size);
  rawarray = new double[size];
  wavearray = new double[size];
  if (datatype == 'w') {
    for (int i = 0; i < size; i++)
      wavearray[i] = inputdata[i];
  }
  else {
    for (int i = 0; i < size; i++)
      rawarray[i] = inputdata[i];
  }
}



Wavelet::~Wavelet(void)
{
  delete[] rawarray;
  delete[] wavearray;
}

double Wavelet::getWaveletMax(int scale)
{
  /* gets the wavelet maximum for the specified scale */

  double waveletmax = 0;
  int scaleSize = int(size / pow(2.0, scale) + .01);
  int i;
  /* 
     first element of scale n is at position 2 ^ (totalScales - n) - 1
     last element is at 2 ^ (totalScales - n + 1) - 1
  */
  for (i = scaleSize; i < 2 * scaleSize; i++)
    {
      if (fabs(wavearray[i]) > fabs(waveletmax))
	waveletmax = wavearray[i];
    }
  return waveletmax;
}

double Wavelet::getWaveletMax(int scale, int sect, int num_sect)
{
  double waveletmax = 0;
  int scaleSize = int(size / pow(2.0, scale) + .01);
  int i;
  int segment = scaleSize / num_sect;
  for (i = scaleSize + (sect-1)*segment; i < scaleSize + sect*segment; i++)
  {
    if (fabs(wavearray[i]) > fabs(waveletmax))
      waveletmax = wavearray[i];
  }
  return waveletmax;
}
 
int Wavelet::getWaveletMaxTime(int scale)
{
  /* 
     gets the time at which wavelet maximum occurs
     for the specified scale 
  */

  double waveletmax = 0;
  int maxtime = 0;
  int scaleSize = int(size / pow(2.0, scale) + .01);
  int i;
  /* 
     first element of scale n is at position scaleSize
     last element is at 2 * scaleSize - 1
  */
  for (i = scaleSize; i < 2 * scaleSize; i++)
    {
      if (fabs(wavearray[i]) > fabs(waveletmax))
	{
	  waveletmax = wavearray[i];
	  /* 
	     set maxtime to the time at which maximum occurs, correcting
	     for preceding array elements which are not part of the specified
	     scale
	  */
	  maxtime = i - scaleSize;
	}
    }
  return (maxtime + 1);
}

int Wavelet::getWaveletMaxTime(int scale, int sect, int num_sect)
{
  int i;
  double waveletmax = 0;
  int maxtime = 0;
  int scaleSize = int(size / pow(2.0, scale) + .01);
  int segment = scaleSize / num_sect;
  for (i = scaleSize + (sect-1)*segment; i < scaleSize + sect*segment; i++)
  {
    if (fabs(wavearray[i]) > fabs(waveletmax))
    {
      waveletmax = wavearray[i];
      maxtime = i - scaleSize;
    }
  }
  return (maxtime + 1 - (sect-1)*segment);
}


int Wavelet::checkMaxAlign(int * maxtime, int epsilon)
{
  /*
    returns binary number representing alignment of
    wavelet maxima through the scales.
  */

  int i;
  unsigned int alignNum = 0;
  for (i = 0; i < totalScales - 2; i++)
    {
      alignNum = alignNum << 1;
      if (2 * maxtime[i+1] >= maxtime[i] - epsilon &&
	  2 * maxtime[i+1] <= maxtime[i] + epsilon)
	alignNum += 1;    
    }
  return alignNum;
}

int Wavelet::checkNarrowbandAlign(int * maxtime, int epsilon, int minWidth){
  /* 
     Finds all alignments of at least minWidth scales. Return format is 
     [start stop start stop ...]
  */
  int i; 
  int j = 0;
  int k;
  unsigned int alignArray[(totalScales - 2) / minWidth * 2];
  for (i = 0; i < (totalScales - 2) / minWidth; i++)
    alignArray[i] = 0;
  
  for (i = 0, k = 0; i < totalScales - 2; i++)
    {
      if (2 * maxtime[i+1] >= maxtime[i] - epsilon &&
	  2 * maxtime[i+1] <= maxtime[i] + epsilon)
	continue;
      else{
	if (i - j >= minWidth){
	  alignArray[k] = j + 1;
	  alignArray[++k] = i + 1;
	  k++;
	  j = i + 1;
	}
	else 
	  j = i + 1;
      }
    }
  return alignArray[0];
}
    
    
double Wavelet::checkAccuracy(double * inputarray){
  /*
    Compares reconstructed array to input array.
  */
  int i;
  double temp;
  double mse = 0.0;
  double rms;

  for(i=0;i<size;i++)
    {
      temp = inputarray[i] - rawarray[i];
      mse += temp*temp;
    }
  mse/=(double) size;
  rms = sqrt(mse);
  return rms;
}

double Wavelet::getAlignMag (double * maxtime){
    
  /* 
     Generates a measure of the alignment of wavelet maxima by 
     finding the maximum distance away from "perfect" propagation
     and then taking 1 over the square of that number. Result will
     range from 0 to 1.
  */

  int i;
  double dist = 0;
  for (i = 0; i < totalScales - 2; i++){
    if (fabs(maxtime[i] - 2 * maxtime[i+1]) > dist)
      dist = fabs(maxtime[i] - 2 * maxtime[i+1]);
  }
  return 1 / (dist * dist);
}

double Wavelet::deltaE (double * waveletmax){
    
  /*
    Finds the difference in energy content (as measured by
    the square of the wavelet maxima) between low and high 
    scales. Negative deltaE means the signal is low frequency
    biased; positive deltaE means high frequency biased.
  */

  int i;
  double lowFreqE = 0;
  double highFreqE = 0;
  for (i = 0; i < totalScales / 2 - 1; i++)
    highFreqE += pow(waveletmax[i], 2);
  for (i = totalScales / 2; i < totalScales - 1; i++)
    lowFreqE += pow(waveletmax[i], 2);
  return (highFreqE - lowFreqE);
}

int Wavelet::hFoc (double * waveletmax){

  /*
    Finds the maximum of waveletmax, returns the scale at which
    maximum occurs. This scale is the "focus" of the power in the
    signal.
  */

  int i;
  int hfoc = 0;
  double max = 0;
  for (i = 0; i < totalScales - 1; i++)
    {
      if (fabs(waveletmax[i]) > max){
	max = fabs(waveletmax[i]);
	hfoc = i;
      }
    }
  return hfoc;
}

double * Wavelet::getRawArray(void)
{
  return rawarray;
}

double * Wavelet::getWaveArray(void)
{
  return wavearray;
}


int Wavelet::getSize(void)
{
  return size;
}

int 
Wavelet::getScale(int samples)

{
  int scale = 0;
  while (samples != 1)     //calculate J, where 2^J = size
    {
      samples /= 2;
      scale++;
    }
  return scale;
}

void Wavelet::setRawArray  (double * inputdata, int arraySize)
{
  delete[] rawarray;
  delete[] wavearray;
  wavearray = new double[arraySize];
  rawarray = new double[arraySize];
  for (int i = 0; i < arraySize; i++)
    rawarray[i] = inputdata[i];
  size = arraySize;
}

void Wavelet::setWaveArray (double * inputdata, int arraySize)
{
  delete[] rawarray;
  delete[] wavearray;
  wavearray = new double[arraySize];
  rawarray = new double[arraySize];
  for (int i = 0; i < arraySize; i++)
    wavearray[i] = inputdata[i];
  size = arraySize;
}

int Wavelet::transform(int     direction)
{
  // 2 cases: Forward (direction=1) or Inverse (direction=-1) transform
  // returns 1 on successful transform, 0 on failure
  if(direction==1)
    {
      if(!dec())
	return 0;
    }
  else if(direction==(-1))
    {
      if(!rec())
	return 0;
    }
  else
    {
      return 0;
    }
  return 1;
}

/************************************************************************
 * Implements the forward (or decomposition) transform
 * discussed above.
 ************************************************************************/
int Wavelet::dec()
{
  for (int i = 0; i < size; i++)
    wavearray[i] = rawarray[i];   //initialize wavelet coefficients array

  int lsize;
  lsize = size;
  while(lsize>1)
    {
      // The idea is that lsize returns the size of the low-pass band
      // For example, if size=512, then after 1-level of decomposition
      // (is the name of the function more clear now ?) lsize = 256
      // and so on.
      lsize = dec_1level(lsize);
    }
  return lsize;
}

/************************************************************************
 * Implements the inverse (or reconstruction) transform
 * discussed above.
 ************************************************************************/

int Wavelet::rec()
{
  
  int i;
  int j;
  int          lsize=0;
  

  for (i = 0; i < size; i++)
    rawarray[i] = wavearray[i];   //initialize reconstructed array

  // For a given decomposition level, need to find out what is the
  // appropriate size to use.
  for(i=totalScales - 1; i >= 0; i--)
    {
      lsize = size;
      for(j = 0; j < i; j++)
	{
	  lsize = lsize - lsize/2;
	}
      lsize = rec_1level(lsize);
    }
  return lsize;
}

/***********************************************************************
 * HAAR WAVELET SECTION
************************************************************************/

Haar_Wavelet::Haar_Wavelet(int samples):
  Wavelet(samples)
{}
Haar_Wavelet::Haar_Wavelet():
  Wavelet()
{}
Haar_Wavelet::Haar_Wavelet(double * inputdata, int samples, char datatype):
  Wavelet(inputdata, samples, datatype)
{}
const char* 
Haar_Wavelet::getType(void)
{
  return "Haar";
}

/************************************************************************
 * Implements 1-level of decomposition, used when doing forward Haar transform
 * Haar filter: [ 1  1 ] * 1/sqrt(2.0)
 *              [ 1 -1 ]
 ************************************************************************/
int Haar_Wavelet::dec_1level(int lsize)
{
  int i;
  int j;
  int k;
  int l;
  int      size_low;
  int      size_high;
  double   sum;
  double   diff;
  double   temp1;
  double   temp2;


  if(lsize>1)
    {
      double *larray = new double[lsize];
      // sqrt1_2 = sqrt2/2.0;
      size_high = lsize/2;
      // size_high = size/2 (integer arithmetic) should give us the size of the
      // HIGH-PASS filtered band, i.e. the differences. 
      size_low  = lsize - size_high;
      // size of the LOW-PASS filtered band, i.e. the sums.
      k = 0;
      l = 1;
      for(i=0,j=size_low;i<size_high;i++,j++)
	{
	  // For example: i=0, k=0, l=1
	  //              i=1, k=2, l=3
	  //              i=2, k=4, l=5 ...
	  temp1 = wavearray[k];
	  temp2 = wavearray[l];
	  sum   = temp1 + temp2;
	  diff  = temp1 - temp2;
	  larray[i] = sqrt1_2*sum;
	  larray[j] = sqrt1_2*diff;
	  k += 2;
	  l += 2;
	}
      // The following case should only happen when size is odd
      // In this case, we want to keep the "extra" sample on the low-pass band
      // Assume that the "missing" sample is = to the last. In this case,
      // sum = 2*(the last)/sqrt2 = sqrt2*(the last)
      // diff = 0 (no need to store !!!)
      if(size_low!=size_high)
	{
	  larray[size_low-1] = wavearray[lsize-1]*sqrt2;
	}
      for (i=0; i < lsize; i++)
	  wavearray[i] = larray[i];
      delete[] larray;
      return(size_low);
    }
  else
    {
      return(lsize);
    }
}



/************************************************************************
 * Implements 1-level of reconstruction, used when doing inverse Haar transform
 * Haar filter: [ 1  1 ] * 1/sqrt(2.0)
 *              [ 1 -1 ]
 ************************************************************************/
int  Haar_Wavelet::rec_1level(int    lsize)
{
  int i;
  int j;
  int k;
  int l;
  int          size_low;
  int          size_high;
  double       sum;
  double       diff;
  double       temp1;
  double       temp2;
   
  if(lsize>1)
    {
       double *larray = new double[lsize];     
      // sqrt1_2 = sqrt2/2.0;
      size_high = lsize/2;
      // size_high = size/2 (integer arithmetic) should give us the size of the
      // HIGH-PASS filtered band, i.e. the differences. 
      size_low  = lsize - size_high;
      // size of the LOW-PASS filtered band, i.e. the sums.
      k = 0;
      l = 1;
      for(i=0,j=size_low;i<size_high;i++,j++)
	{
	  // For example: i=0, k=0, l=1
	  //              i=1, k=2, l=3
	  //              i=2, k=4, l=5 ...
	  temp1 = rawarray[i];
	  temp2 = rawarray[j];
	  sum   = temp1 + temp2;
	  diff  = temp1 - temp2;
	  larray[k] = sqrt1_2*sum;
	  larray[l] = sqrt1_2*diff;
	  k += 2;
	  l += 2;
	}
      // The following case should only happen when size is odd
      // In this case, we have kept the "extra" sample on the low-pass band
      // assuming that the "missing" sample is = to the last. In this case,
      // sum = 2*(the last)/sqrt2 = sqrt2*(the last)
      // diff = 0 (no need to store !!!)
      // So, sample = sum/SQRT2
      if(size_low!=size_high)
	{
	  larray[lsize-1] = sqrt1_2*rawarray[size_low-1];
	}
      for (i=0; i < lsize; i++)
	rawarray[i] = larray[i];
      delete[] larray;
      return(size_low);
    }
  else
    {
      return(lsize);
    }
}

/************************************************************************
 * DAUBECHIES-4 WAVELET SECTION
 ************************************************************************/

D4_Wavelet::D4_Wavelet(int samples):
  Wavelet(samples)

{
  D4[0] = 0.482962913145;
  D4[1] = 0.836516303738;
  D4[2] = 0.224143868042;
  D4[3] = -0.129409522551;
}
D4_Wavelet::D4_Wavelet():
  Wavelet()

{
  D4[0] = 0.482962913145;
  D4[1] = 0.836516303738;
  D4[2] = 0.224143868042;
  D4[3] = -0.129409522551;
}
D4_Wavelet::D4_Wavelet(double * inputdata, int samples, char datatype):
  Wavelet(inputdata, samples, datatype)

{
  D4[0] = 0.482962913145;
  D4[1] = 0.836516303738;
  D4[2] = 0.224143868042;
  D4[3] = -0.129409522551;
}

const char* 
D4_Wavelet::getType(void)
{
  return "Daubechies-4";
}

/************************************************************************
 * Implements 1-level of decomposition, used when doing Daubechies-4
 * compact support 4-tap wavelet transform
 * Daubechies-4 filter:   
 * [ 0.482962913145  0.836516303738  0.224143868042 -0.129409522551 ] -LOW
 * [-0.129409522551 -0.224143868042  0.836516303738 -0.482962913145 ] -HIGH
 * "Allignment" : 
 *    x(2)   x(3)   x(4)    x(5)  x(6)   x(7)
 *               [ 0.483  0.837  0.224 -0.129 ]         -LOW
 * [-0.129 -0.224  0.837 -0.483 ]                       -HIGH
 ************************************************************************/
int D4_Wavelet::dec_1level(int lsize)
{
  int i;
  int j;
  int k;
  int l;
  int          size_low;
  int          size_high;
  double       temp;
  double       sum;
  double       diff;
  double       sign;

  if(lsize>1)
    {
      double *larray = new double[lsize];
      size_high = lsize/2;
      // size_high = size/2 (integer arithmetic) should give us the size of the
      // HIGH-PASS filtered band, i.e. the differences. 
      size_low  = lsize - size_high;
      // size of the LOW-PASS filtered band, i.e. the sums.
      // adjust size to be even
      lsize = 2*size_high;
      k = 0;
      l = lsize+1;
      for(i=0;i<lsize-1;i+=2)
	{
	  sum  =  0.0;
	  diff =  0.0;
	  sign = -1.0;
	  for(j=0;j<4;j++)
	    {
	      temp  = D4[j];
	      // size=1024, i=0, k= 0, l=1025, j=0  ==>  sum=0  diff=1
	      //                               j=1  ==>  sum=1  diff=0
	      //                               j=2  ==>  sum=2  diff=1023
	      //                               j=3  ==>  sum=3  diff=1022
	      //            i=2, k= 2, l=1027, j=0  ==>  sum=2  diff=3
	      //                               j=1  ==>  sum=3  diff=2
	      //                               j=2  ==>  sum=4  diff=1
	      //                               j=3  ==>  sum=5  diff=0
	      sum  += temp*wavearray[(k+j)%lsize];
	      diff += sign*temp*wavearray[(l-j)%lsize];
	      sign  = -sign;
	    }
	  larray[i]   = sum;
	  larray[i+1] = diff;
	  k += 2;
	  l += 2;
	}
      // The following case should only happen when size is odd
      // In this case, we want to keep the "extra" sample on the low-pass band
      // Assume that the "missing" sample is = to the last. In this case,
      // sum = 2*(the last)/sqrt2 = sqrt2*(the last)
      // diff = 0 (no need to store !!!)
      if(size_low!=size_high)
	{
	  lsize++;
	  wavearray[size_low-1] = wavearray[lsize-1]*sqrt2;
	}
      for(i=0;i<lsize-1;i+=2)
	  {
	    j = i/2;
	    wavearray[j]          = larray[i];
	    wavearray[j+size_low] = larray[i+1];
	  }
	delete[] larray;
      return(size_low);
    }
  else
    {
      return(lsize);
    }
}

/************************************************************************
 * Implements 1-level of reconstruction, used when doing inverse 
 * Daubechies-4 compact support 4-tap wavelet transform
 * Daubechies-4 filter:   
 * [ 0.482962913145  0.836516303738  0.224143868042 -0.129409522551 ] -LOW
 * [-0.129409522551 -0.224143868042  0.836516303738 -0.482962913145 ] -HIGH
 * "Allignment" : 
 *     c[512]  d[1]   c[1]   d[2]   c[2]   d[3]   c[3]    d[4]   c[4]
 *     Notice: "highs" go first and then "lows" i.e.
 *     d(512) c(512) | d(1) c(1) d(2) d(3) .... d(512) c(512) | d(1) c(1)
 *     d4[3]   d4[2]  d4[1]  d4[4]
 *     d4[4]  -d4[1]  d4[2] -d4[3]
 *                      3      2      1      4          -> low
 *                      4     -1      2     -3          -> high
 ************************************************************************/
int D4_Wavelet::rec_1level (int     lsize)
{
  int i;
  int j;
  int k;
  int l;
  int          size_low;
  int          size_high;
  double       temp1;
  double       temp2;
  double       temp3;
  double       temp4;
  double       sum;
  double       diff;
 

  if(lsize>1)
    {
      
      double *larray = new double[lsize];
      size_high = lsize/2;
      // size_high = lsize/2 (integer arithmetic) should give us the size of the
      // HIGH-PASS filtered band, i.e. the differences. 
      size_low  = lsize - size_high;
      // size of the LOW-PASS filtered band, i.e. the sums.
      // adjust lsize to be even
      lsize = 2*size_high;
      for(i=0,k=size_high;i<lsize-1;i+=2,k++)
	{
	  sum  = 0.0;
	  diff = 0.0;
	  for(j=0,l=0;j<4;j+=2,l++)
	    {
	      temp1 = D4[j];
	      temp2 = D4[j+1];
	      temp3 = rawarray[(k-l)%size_high];
	      temp4 = rawarray[(k+l)%size_high+size_low];
	      // For example : i=0, j=0, k=512, l=0, temp3=  0, temp4=512
	      // (size=1024)   i=0, j=2, k=512, l=1, temp3=511, temp4=513
	      //               i=2, j=0, k=513, l=0, temp3=  2, temp4=513
	      //               i=2, j=2, k=513, l=1, temp3=  1, temp4=514
	      //               i=4, j=0, k=514, l=0, temp3=  3, temp4=514
	      //               i=4, j=2, k=514, l=1, temp3=  2, temp4=515
	      //               i=6, j=0, k=515, l=0, temp3=  4, temp4=515
	      //               i=6, j=2, k=515, l=1, temp3=  3, temp4=516
	      sum  += temp1*temp3 + temp2*temp4;
	      diff += temp2*temp3 - temp1*temp4;
	    }
	  larray[i]   = sum;
	  larray[i+1] = diff;
	}
      // The following case should only happen when lsize is odd
      // In this case, we have kept the "extra" sample on the low-pass band
      // assuming that the "missing" sample is = to the last. In this case,
      // sum = 2*(the last)/sqrt2 = sqrt2*(the last)
      // diff = 0 (no need to store !!!)
      // So, sample = sum/SQRT2
      if(size_low!=size_high)
	{
	  lsize++;
	  larray[i]   = rawarray[size_low-1]*sqrt1_2;
	}

      
      for (i=0; i < lsize; i++)
	  rawarray[i] = larray[i];
      delete[] larray;
      return(size_low);
    
    }
  else
    {
      return(lsize);
    }
}

/*********************************************************************
 * DAUBECHIES-8 WAVELET SECTION
 ********************************************************************/

D8_Wavelet::D8_Wavelet(int samples):
  Wavelet(samples)
{
  D8[0] = 0.230377813309l; 
  D8[1] = 0.714846570553;
  D8[2] = 0.630880767930;
  D8[3] = -0.027983769417;
  D8[4] = -0.187034811719;
  D8[5] = 0.030841381836,
  D8[6] = 0.032883011667;
  D8[7] = -0.010597401785;
}
D8_Wavelet::D8_Wavelet():
  Wavelet()
{
  D8[0] = 0.230377813309l; 
  D8[1] = 0.714846570553;
  D8[2] = 0.630880767930;
  D8[3] = -0.027983769417;
  D8[4] = -0.187034811719;
  D8[5] = 0.030841381836;
  D8[6] = 0.032883011667;
  D8[7] = -0.010597401785;
}
D8_Wavelet::D8_Wavelet(double * inputdata, int samples, char datatype):
  Wavelet(inputdata, samples, datatype)
{
  D8[0] = 0.230377813309l; 
  D8[1] = 0.714846570553;
  D8[2] = 0.630880767930;
  D8[3] = -0.027983769417;
  D8[4] = -0.187034811719;
  D8[5] = 0.030841381836,
  D8[6] = 0.032883011667;
  D8[7] = -0.010597401785;
}

const char* 
D8_Wavelet::getType(void)
{
  return "Daubechies-8";
}

/************************************************************************
 * Implements 1-level of decomposition, used when doing Daubechies-8
 * compact support 8-tap wavelet transform
 * Daubechies-8 filter:   
 * [  0.230377813309, 0.714846570553, 0.630880767930, -0.027983769417,
 *   -0.187034811719, 0.030841381836, 0.032883011667, -0.010597401785 ] - LOW
 * [ -0.010597401785,-0.032883011667, 0.030841381836,  0.187034811719,
 *   -0.027983769417,-0.630880767930, 0.714846570553, -0.230377813309 ] - HIGH
 * "Allignment" : 
 *    0  1  2  3  4  5  6  7  8  9 10 11 12 13
 *                      .  .  .  .  .  .  .  .   -LOW
 *    .  .  .  .  .  .  .  .                     -HIGH
 ************************************************************************/
int D8_Wavelet::dec_1level(int     lsize)
{
  int i;
  int j;
  int k;
  int l;
  int          size_low;
  int          size_high;
  double       temp;
  double       sum;
  double       diff;
  double       sign;
   
  if(lsize>1)
    {
      
      double *larray = new double[size];
      size_high = lsize/2;
      // size_high = lsize/2 (integer arithmetic) should give us the size of the
      // HIGH-PASS filtered band, i.e. the differences. 
      size_low  = lsize - size_high;
      // size of the LOW-PASS filtered band, i.e. the sums.
      // adjust lsize to be even
      lsize = 2*size_high;
      k = 0;
      l = 3*lsize+1;
      for(i=0;i<lsize-1;i+=2)
	{
	  sum  =  0.0;
	  diff =  0.0;
	  sign = -1.0;
	  for(j=0;j<8;j++)
	    {
	      temp  = D8[j];
	      // size=1024, i=0, k= 0, l=3075, j=0  ==>  sum=0  diff=1
	      //                               j=1  ==>  sum=1  diff=0
	      //                               j=2  ==>  sum=2  diff=1023
	      //                               j=3  ==>  sum=3  diff=1022
	      //                               j=4  ==>  sum=4  diff=1021
	      //                               j=5  ==>  sum=5  diff=1020
	      //                               j=6  ==>  sum=6  diff=1019
	      //                               j=7  ==>  sum=7  diff=1018
	      //            i=2, k= 2, l=3077, j=0  ==>  sum=2  diff=3
	      //                               j=1  ==>  sum=3  diff=2
	      //                               j=2  ==>  sum=4  diff=1
	      //                               j=3  ==>  sum=5  diff=0
	      //                               j=4  ==>  sum=6  diff=1023
	      //                               j=5  ==>  sum=7  diff=1022
	      //                               j=6  ==>  sum=8  diff=1021
	      //                               j=7  ==>  sum=9  diff=1020
	      sum  += temp*wavearray[(k+j)%lsize];
	      diff += sign*temp*wavearray[(l-j)%lsize];
	      sign  = -sign;
	    }
	  larray[i]   = sum;
	  larray[i + 1] = diff;
	  k += 2;
	  l += 2;
	}
      // The following case should only happen when lsize is odd
      // In this case, we want to keep the "extra" sample on the low-pass band
      // Assume that the "missing" sample is = to the last. In this case,
      // sum = 2*(the last)/sqrt2 = sqrt2*(the last)
      // diff = 0 (no need to store !!!)
      if(size_low!=size_high)
	{
	  lsize++;
	  wavearray[size_low-1] = wavearray[lsize-1]*sqrt2;
	}
      for(i=0;i<lsize-1;i+=2)
	  {
	    j = i/2;
	    wavearray[j]          = larray[i];
	    wavearray[j+size_low] = larray[i+1];
	  }
	delete[] larray;
      return(size_low);
    }
  else
    {
      return(lsize);
    }
}
/************************************************************************
 * Implements 1-level of reconstruction, used when doing inverse 
 * Daubechies-8 compact support 8-tap wavelet transform
 * Daubechies-8 filter:   
 * [  0.230377813309, 0.714846570553, 0.630880767930, -0.027983769417,
 *   -0.187034811719, 0.030841381836, 0.032883011667, -0.010597401785 ] - LOW
 * [ -0.010597401785,-0.032883011667, 0.030841381836,  0.187034811719,
 *   -0.027983769417,-0.630880767930, 0.714846570553, -0.230377813309 ] - HIGH
 ************************************************************************/
int D8_Wavelet::rec_1level(int     lsize)
{
  int i;
  int j;
  int k;
  int l;
  int          size_low;
  int          size_high;
  double       temp1;
  double       temp2;
  double       temp3;
  double       temp4;
  double       sum;
  double       diff;
   
  if(lsize>1)
    {
      double *larray = new double[lsize];
      size_high = lsize/2;
      // size_high = lsize/2 (integer arithmetic) should give us the size of the
      // HIGH-PASS filtered band, i.e. the differences. 
      size_low  = lsize - size_high;
      // size of the LOW-PASS filtered band, i.e. the sums.
      // adjust lsize to be even
      lsize = 2*size_high;
      for(i=0,k=3*size_high;i<lsize-1;i+=2,k++)
	{
	  sum  = 0.0;
	  diff = 0.0;
	  for(j=0,l=0;j<8;j+=2,l++)
	    {
	      temp1 = D8[j];
	      temp2 = D8[j+1];
	      temp3 = rawarray[(k-l)%size_high];
	      temp4 = rawarray[(k+l)%size_high+size_low];
	      // For example : i=0, k=1536, j=0, l=0, temp3=  0, temp4=512
	      // (size=1024)                j=2, l=1, temp3=512, temp4=513
	      //                            j=4, l=2, temp3=511, temp4=514
	      //                            j=6, l=3, temp3=510, temp4=515
	      //               i=2, k=1538, j=0, l=0, temp3=  2, temp4=513
	      //                            j=2, l=1, temp3=  1, temp4=514
	      //                            j=4, l=2, temp3=512, temp4=515
	      //                            j=6, l=3, temp3=511, temp4=516
	      sum  += temp1*temp3 + temp2*temp4;
	      diff += temp2*temp3 - temp1*temp4;
	    }
	  larray[i]   = sum;
	  larray[i+1] = diff;
	}
      // The following case should only happen when size is odd
      // In this case, we have kept the "extra" sample on the low-pass band
      // assuming that the "missing" sample is = to the last. In this case,
      // sum = 2*(the last)/sqrt2 = sqrt2*(the last)
      // diff = 0 (no need to store !!!)
      // So, sample = sum/SQRT2
      if(size_low!=size_high)
	{
	  lsize++;
	  larray[i]   = rawarray[size_low-1]*sqrt1_2;
	}
      for (i=0; i < lsize; i++)
	  rawarray[i] = larray[i];
      delete[] larray;
      return(size_low);
    }
  else
    {
      return(lsize);
    }
}

/*********************************************************************
 * DAUBECHIES-16 WAVELET SECTION
 ********************************************************************/

D16_Wavelet::D16_Wavelet(int samples):
  Wavelet(samples)

{
    D16[0] = 0.054415842243; 
    D16[1] = 0.312871590914; 
    D16[2] = 0.675630736297;
    D16[3] = 0.585354683654;  
    D16[4] = -0.015829105256;
    D16[5] = -0.284015542962;
    D16[6] = 0.000472484574;
    D16[7] = 0.128747426620;
    D16[8] = -0.017369301002;
    D16[9] = -0.044088253931;
    D16[10] = 0.013981027917;
    D16[11] = 0.008746094047;
    D16[12] = -0.004870352993;
    D16[13] = -0.000391740373;
    D16[14] =  0.000675449409;
    D16[15] = -0.000117476784;
}
D16_Wavelet::D16_Wavelet():
  Wavelet()

{
    D16[0] = 0.054415842243; 
    D16[1] = 0.312871590914; 
    D16[2] = 0.675630736297;
    D16[3] = 0.585354683654;  
    D16[4] = -0.015829105256;
    D16[5] = -0.284015542962;
    D16[6] = 0.000472484574;
    D16[7] = 0.128747426620;
    D16[8] = -0.017369301002;
    D16[9] = -0.044088253931;
    D16[10] = 0.013981027917;
    D16[11] = 0.008746094047;
    D16[12] = -0.004870352993;
    D16[13] = -0.000391740373;
    D16[14] =  0.000675449409;
    D16[15] = -0.000117476784;
}

D16_Wavelet::D16_Wavelet(double * inputdata, int samples, char datatype):
  Wavelet(samples)

{
    D16[0] = 0.054415842243; 
    D16[1] = 0.312871590914; 
    D16[2] = 0.675630736297;
    D16[3] = 0.585354683654;  
    D16[4] = -0.015829105256;
    D16[5] = -0.284015542962;
    D16[6] = 0.000472484574;
    D16[7] = 0.128747426620;
    D16[8] = -0.017369301002;
    D16[9] = -0.044088253931;
    D16[10] = 0.013981027917;
    D16[11] = 0.008746094047;
    D16[12] = -0.004870352993;
    D16[13] = -0.000391740373;
    D16[14] =  0.000675449409;
    D16[15] = -0.000117476784;
}

const char* 
D16_Wavelet::getType(void)
{
  return "Daubechies-16";
}

/************************************************************************
 * Implements 1-level of decomposition, used when doing Daubechies-16
 * compact support 16-tap wavelet transform
 * Daubechies-16 filter:
 * [ 0.054415842243,  0.312871590914,  0.675630736297,  0.585354683654,
 *  -0.015829105256, -0.284015542962,  0.000472484574,  0.128747426620,
 *  -0.017369301002, -0.044088253931,  0.013981027917,  0.008746094047,
 *  -0.004870352993, -0.000391740373,  0.000675449409, -0.000117476784 ]
 * "Allignment" : 
 * 0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 21 23 24 25
 *                                           .  .  .  .  .  .  .  .  .  .  .  .
 * .  .  .  .  .  .  .  .  .  .  .  .  .  .  .  .
 ************************************************************************/
int D16_Wavelet::dec_1level(int     lsize)
{
  int i;
  int j;
  int k;
  int l;
  int          size_low;
  int          size_high;
  double       temp;
  double       sum;
  double       diff;
  double       sign;
   
  if(lsize>1)
    {
      double *larray = new double[lsize];
      size_high = lsize/2;
      // size_high = lsize/2 (integer arithmetic) should give us the size of the
      // HIGH-PASS filtered band, i.e. the differences. 
      size_low  = lsize - size_high;
      // size of the LOW-PASS filtered band, i.e. the sums.
      // adjust lsize to be even
      lsize = 2*size_high;
      k = 0;
      l = 7*lsize+1;
      for(i=0;i<lsize-1;i+=2)
	{
	  sum  =  0.0;
	  diff =  0.0;
	  sign = -1.0;
	  for(j=0;j<16;j++)
	    {
	      temp  = D16[j];
	      sum  += temp*wavearray[(k+j)%lsize];
	      diff += sign*temp*wavearray[(l-j)%lsize];
	      sign  = -sign;
	    }
	  larray[i]   = sum;
	  larray[i+ 1] = diff;
	  k += 2;
	  l += 2;
	}
      // The following case should only happen when lsize is odd
      // In this case, we want to keep the "extra" sample on the low-pass band
      // Assume that the "missing" sample is = to the last. In this case,
      // sum = 2*(the last)/sqrt2 = sqrt2*(the last)
      // diff = 0 (no need to store !!!)
      if(size_low!=size_high)
	{
	  lsize++;
	  wavearray[size_low-1] = wavearray[lsize-1]*sqrt2;
	}
      for(i=0;i<lsize-1;i+=2)
	  {
	    j = i/2;
	    wavearray[j]          = larray[i];
	    wavearray[j+size_low] = larray[i+1];
	  }
      delete[] larray;
      return(size_low);
    }
  else
    {
      return(lsize);
    }
}


/************************************************************************
 * Implements 1-level of reconstruction, used when doing inverse 
 * Daubechies-16 compact support 16-tap wavelet transform
 * Daubechies-8 filter:   
 * [ 0.054415842243,  0.312871590914,  0.675630736297,  0.585354683654,
 *  -0.015829105256, -0.284015542962,  0.000472484574,  0.128747426620,
 *  -0.017369301002, -0.044088253931,  0.013981027917,  0.008746094047,
 *  -0.004870352993, -0.000391740373,  0.000675449409, -0.000117476784 ]
 ************************************************************************/
int D16_Wavelet::rec_1level(int     lsize)
{
  int i;
  int j;
  int k;
  int l;
  int          size_low;
  int          size_high;
  double       temp1;
  double       temp2;
  double       temp3;
  double       temp4;
  double       sum;
  double       diff;
   
  if(lsize>1)
    {
      double *larray = new double[lsize];
      size_high = lsize/2;
      // size_high = lsize/2 (integer arithmetic) should give us the size of the
      // HIGH-PASS filtered band, i.e. the differences. 
      size_low  = lsize - size_high;
      // size of the LOW-PASS filtered band, i.e. the sums.
      // adjust lsize to be even
      lsize = 2*size_high;
      for(i=0,k=7*size_high;i<lsize-1;i+=2,k++)
	{
	  sum  = 0.0;
	  diff = 0.0;
	  for(j=0,l=0;j<16;j+=2,l++)
	    {
	      temp1 = D16[j];
	      temp2 = D16[j+1];
	      temp3 = rawarray[(k-l)%size_high];
	      temp4 = rawarray[(k+l)%size_high+size_low];
	      sum  += temp1*temp3 + temp2*temp4;
	      diff += temp2*temp3 - temp1*temp4;
	    }
	  larray[i]   = sum;
	  larray[i+1] = diff;
	}
      // The following case should only happen when lsize is odd
      // In this case, we have kept the "extra" sample on the low-pass band
      // assuming that the "missing" sample is = to the last. In this case,
      // sum = 2*(the last)/sqrt2 = sqrt2*(the last)
      // diff = 0 (no need to store !!!)
      // So, sample = sum/SQRT2
      if(size_low!=size_high)
	{
	  lsize++;
	  larray[i]   = rawarray[size_low-1]*sqrt1_2;
	}
      for (i=0; i < lsize; i++)
	  rawarray[i] = larray[i];
      delete[] larray;
      return(size_low);
    }
  else
    {
      return(lsize);
    }
}

