/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "html/text.hh" 
#include "html/color.hh"

#include "ChannelInfo.hh"
#include "ConfContainer.hh"

#ifdef sun
#include <ieeefp.h>
#else
#include <cmath>
#endif

#ifndef HAVE_ISFINITE
#define isfinite(x) finite(x)
#endif

using namespace std;


template<class T>
bool CalcMean ( int n, const T* data, int step, float& mean) {
   float sum = 0;
   int count = 0;
   for ( int i = 0; i < n; i += step ) {
      if ( abs( data[i] ) < 32767 ) {
	 sum += data[i];
	 ++count;
      }
   }
   
   if ( !count || !isfinite( sum ) ) {
      return false;
   }
   mean = sum/count;
   return true;
}

template<class T>
bool CalcRMS ( int n, const T* data, int step, float& sdev ) {
   int count = 0;
   float sum2 = 0;
   
   for ( int i = 0; i < n; i += step ) {
      if (abs(data[i]) < 65535 ){
	 sum2 += (float)data[i] * data[i];
	 ++count;
      }
   }
   
   if ( !count || !isfinite( sum2 ) ) 
      return false;
   
   sdev = sqrt(sum2 / count);
   return true;
   
}

//==============================================================================
//
//  ChannelInfo
//
//==============================================================================
ChannelInfo::~ChannelInfo () {
   if ( fTSnew ) delete fTSnew;
}

//______________________________________________________________________________
void ChannelInfo::AddConfig ( ConfigInfo& conf ) {
   fList.push_back ( conf );
}

//______________________________________________________________________________
const char* ChannelInfo::GetName () const {
   return fName.c_str();
}

//______________________________________________________________________________
void ChannelInfo::SetCheckLock ( int checklock ) {
   fCheckLock = checklock;
}

//______________________________________________________________________________
bool ChannelInfo::SetName ( const char* name ) {
   fName = name;
   std::string n = name;
   std::string::size_type pos = n.find_first_of (':');
   if ( pos == std::string::npos || pos < 2 ) {
      return false;
   }
   else {
      fIfo = n.substr(0, pos);
      return true;
   }
   
}

//______________________________________________________________________________
void ChannelInfo::Dump () const {
   std::string lock;
   
   switch (fCheckLock)
      {
      case LockStatus::kH1:
	 {
	    lock = "H1";
	    break;
	 }
      case LockStatus::kH2:
	 {
	    lock = "H2";
	    break;
	 }
      case LockStatus::kL1:
	 {
	    lock = "L1";
	    break;
	 }
      default:
	 {
	    lock = "OFF";
	    break;
	 }
      }
   
   cout << fName << "\t" << fIfo <<  "\t" << "CheckLock-" << lock << endl;
   for ( ConfigList::const_iterator itr = fList.begin();
	 itr != fList.end(); ++itr) {
      cout << "\t";
      itr->Dump ();
   }
}

//______________________________________________________________________________
void ChannelInfo::ResetStat ( MonServer& mserv, Trend& trend, const vector<TSeries>& tlist, int ts_interval ) {
   for ( ConfigList::iterator itr = fList.begin();
	 itr != fList.end(); ++itr) {
      itr->ResetStat( fName.c_str(), fIfo.c_str(), mserv, trend, tlist, ts_interval );
   }
}

//______________________________________________________________________________
void ChannelInfo::Register ( DaccAPI& dacc ) {
   fTSnew = new TSeries();
   dacc.addChannel ( fName.c_str(), 0, &fTSnew );
}


//______________________________________________________________________________
void ChannelInfo::ConfigLog ( html::table& results ) {
   for ( ConfigList::iterator itr = fList.begin();
	 itr != fList.end(); ++itr ) {
      results.addRow ();
      int row = results.getNRow () - 1;
      results.insertData (row, 0, html::text (fName.c_str()));
      std::string lock;
      switch (fCheckLock)
	 {
	 case LockStatus::kH1:
	    {
	       lock = "H1";
	       break;
	    }
	 case LockStatus::kH2:
	    {
	       lock = "H2";
	       break;
	    }
	 case LockStatus::kL1:
	    {
	       lock = "L1";
	       break;
	    }
	 case LockStatus::kH1MC:
	    {
	       lock = "H1MC";
	       break;
	    }
	 case LockStatus::kH2MC:
	    {
	       lock = "H2MC";
	       break;
	    }
	 case LockStatus::kL1MC:
	    {
	       lock = "L1MC";
	       break;
	    }
	 default:
	    {
	       lock = "OFF";
	       break;
	    }
	 }
      results.insertData (row, 1, html::text (lock.c_str()));
      itr->ConfigLog ( results );
   }
}

//______________________________________________________________________________
void ChannelInfo::UpdateHistogram ( const Time& t, Trend& trend ) { 
   for ( ConfigList::iterator itr = fList.begin(); 
	 itr != fList.end(); ++itr) {
      itr->UpdateHistogram ( t, trend );
   }
}

//______________________________________________________________________________
bool ChannelInfo::GetMean () {
   int N = fTSold.getNSample ();
   float mean;
   
   if ( fDVType == DVector::t_short ) {
      if ( !CalcMean<short> ( N, (const short*)fTSold.refData(), fNSkip, mean ) ) {
	 return false;
      }
   }
   else if ( fDVType == DVector::t_float ) {
      if ( !CalcMean<float> ( N, (const float*)fTSold.refData(), fNSkip, mean ) ) {
	 return false;
      }
   }
   else {
      return false;
   }
   
   fMean = mean;
   
   fMeanSum += fMean;
   ++fMeanCount;
   
   return true;
   
}

//______________________________________________________________________________
bool ChannelInfo::GetRMS () {
   int N = fTSold.getNSample ();
   float sdev = 0;
   
   if ( fDVType == DVector::t_short ) {
      if ( !CalcRMS<short> ( N, (const short*)fTSold.refData(), fNSkip, sdev ) ) {
	 return false;
      }
   }
   else if ( fDVType == DVector::t_float ) {
      if ( !CalcRMS<float> ( N, (const float*)fTSold.refData(), fNSkip, sdev ) ) {
	 return false;
      }
   }
   else {
      return false;
   }
   
   if ( sdev < kMinSDev ) 
      return false;
   
   float norm = 1 + kDecay * fLastNorm;
   fSDev = (sdev + fSDev * kDecay * fLastNorm) / norm;
   fLastNorm = norm;
   
   fSDevSum += fSDev;
   ++fSDevCount;
   
   return true;
}

//______________________________________________________________________________
bool ChannelInfo::Init ( const Time& t ) {
   fSFreq = fTSnew->getNSample();
   fNSkip = ( fSFreq > 256 ) ? fSFreq / 256 : 1;
   
   if ( !fNSkip ) {
      cerr << " fNSkip = 0 " << endl;
      return false;
   }
   
   for ( ConfigList::iterator itr = fList.begin();
	 itr != fList.end(); ++itr ) {
      itr->Init( fSFreq, t );
   }
   
   const TSeries* ts = fTSnew;
   fDVType = ts->refDVect()->getType();
   
   return true;
}

//______________________________________________________________________________
void ChannelInfo::ProcessChannel (const DaccAPI& dacc, EventList& elist, 
				  const LockStatus& lock ) {
   if ( fTSnew->isEmpty() ) 
      return;
	
   if ( fInit == 2 ) {
      if ( Init ( dacc.getCurrentTime() ) ) {
	 --fInit;
      }
   }
   else if ( lock.IsStable ( fCheckLock ) ) {
      if ( fTSold.Append ( *fTSnew ) == 0 ) {
	 if ( !GetMean () ) {
	    fTSold.Clear();
	    fInit = 1;
	    return;
	 }
				
	 fTSold -= fMean;
				
	 if ( !GetRMS () ) {
	    fTSold.Clear();
	    fInit = 1;
	    return;
	 }
			
	 if ( fInit == 1 ) {
	    --fInit;
	 }
	 else {
	    for ( ConfigList::iterator itr = fList.begin();
		  itr != fList.end(); ++itr) {
	       itr->ProcessEvent ( fTSold, fMean, elist );
	    }
	    ++fCount;
	 }
      }
      fTSold = *fTSnew;
   }
   else if ( lock.GetLockStatus( fCheckLock ) ) {
      GetOnGoingEvent ( elist ); // flush out existing events at loss of lock
      ClearConfigStat(); // reset stats in ConfigInfo
   }
}

//______________________________________________________________________________
void ChannelInfo::OutputLog ( html::table& results ) {
   float mean = (fMeanCount) ? fMeanSum/fMeanCount : 0;
   float sdev = (fSDevCount) ? fSDevSum/fSDevCount : 0;
   
   results.addRow();
   int row = results.getNRow () - 1;
   
   html::color clr (150, 0, 0);
   
   // channel
   html::text name ( fName );
   name.setColor (clr);
   
   // mean
   html::text m ( mean );
   m.setColor (clr);
   
   // std. dev.
   html::text s ( sdev );
   s.setColor (clr);
   
   // blank
   html::text none ( "-" );
   none.setColor (clr);
   
   // of events
   int n_low = 0, n_high = 0;
   for ( ConfigList::iterator itr = fList.begin();
	 itr != fList.end(); ++itr) {
      n_low += itr->fEventCountLow;
      n_high += itr->fEventCountHigh;
   }
   html::text e_low ( n_low );
   e_low.setColor (clr);
   
   html::text e_high ( n_high );
   e_high.setColor (clr);
   
   // processed data length
   html::text l ( fCount );
   l.setColor(clr);
   
   results.insertData (row, 0, name ); 
   results.refCell (row, 0).setAlign ( "left" );
   results.insertData (row, 1, m ); 
   results.refCell (row, 1).setAlign ( "right" ); 
   results.insertData (row, 2, s ); 
   results.refCell (row, 2).setAlign ( "right" );
   results.insertData (row, 3, none ); 
   results.refCell (row, 3).setAlign ( "center" );
   results.insertData (row, 4, e_low ); 
   results.refCell (row, 4).setAlign ( "right" );
   results.insertData (row, 5, e_high ); 
   results.refCell (row, 5).setAlign ( "right" );
   results.insertData (row, 6, l ); 
   results.refCell (row, 6).setAlign ( "right" );
   
   for ( ConfigList::iterator itr = fList.begin();
	 itr != fList.end(); ++itr) {
      itr->OutputLog ( results );
   }
   
   fCount = 0;
   fMeanCount = 0;
   fSDevCount = 0;
   fMeanSum = 0;
   fSDevSum = 0;
}

//______________________________________________________________________________
void ChannelInfo::Attention () {
   for ( ConfigList::iterator itr = fList.begin();
	 itr != fList.end(); ++itr) {
      itr->Attention ();
   }
}

//______________________________________________________________________________
void ChannelInfo::TSSave ( ofstream& out ) {
   for ( ConfigList::iterator itr = fList.begin();
	 itr != fList.end(); ++itr ) {
      itr->TSSave ( out );
   }
}

//______________________________________________________________________________   
void ChannelInfo::GetOnGoingEvent ( EventList& elist ) {
   for ( ConfigList::iterator itr = fList.begin();
	 itr != fList.end(); ++itr ) {
      if ( itr->fEvent.fEventStatus ) {
	 elist.push_back( itr->fEvent );
	 //             std::string suffix;
	 //             if ( itr->fEvent.fEventStatus == EventInfo::kEvent ) {
	 //                suffix = "_incmplt_event";
	 //             }
	 //             else {
	 //                suffix = "_incmplt_wait";
	 //             }
	 //             std::string comment = itr->fEvent.fComment + suffix;
	 //            elist.back().fComment = comment;
      }
   }
}
   
//______________________________________________________________________________   
void ChannelInfo::ClearConfigStat () {
   for ( ConfigList::iterator itr = fList.begin(); itr != fList.end(); ++itr ) {
      itr->ClearStat();
   }
}
	
	
//______________________________________________________________________________
ChannelInfo& ChannelInfo::operator= (const ChannelInfo& cinfo) {
   //----------------- configuration
   fName = cinfo.fName;
   fIfo = cinfo.fIfo;
   fCheckLock = cinfo.fCheckLock;
   
   //----------------- status
   fTSnew = cinfo.fTSnew;
   fDVType = cinfo.fDVType;
   fSFreq = cinfo.fSFreq;
   fNSkip = cinfo.fNSkip;
   fInit = cinfo.fInit;
   fMean = cinfo.fMean;
   fSDev = cinfo.fSDev;
   fLastNorm = cinfo.fLastNorm;
   fList = cinfo.fList;
   
   return *this;
}
