/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "DQ_Bit.hh"
#include "FilterDesign.hh"
#include "autopipe.hh"
#include <cmath>

using namespace std;

/**  The DQ_GenRms tool computes the rms of a single filtered channel and 
  *  sets a result bit if the measured rms is less than a specified maximum. 
  *  The rms values are calculated over each stride (typically 1/16th s) and
  *  may be exponentially averaged over multiple strides. 
  *  The tool accepts the following parameters:
  *  * rms-max: Is the maximum rms value to return true. Note that the sense
  *    of this comparison may beinverted by setting the inver parameter in 
  *    tool definition.
  *  * decay-time: The mean time for an exponential average.
  *  * filter-design: the design string in FilterDesign syntax used to 
  *    construct the filter applied before measuring the RMS.
  *  * settle: filter settling time before exponential averaging is started.
  */
class DQ_GenRms : public DQ_bit {
public:
   void
   init(void) {
      _decay_time = 0.0;
      _settle = 0;
      _print  = true;
      if (numeric_param_exists("rms-max")) {
	 _rms_limit  = numeric_param("rms-max");
      } else {
	 throw runtime_error("DQ_GenRms: Missing rms-max parameter");
      }
      if (numeric_param_exists("decay-time")) {
	 _decay_time = numeric_param("decay-time");
      }
      if (string_param_exists("filter-design")) {
	 _design     = string_param("filter-design");
      }
      if (numeric_param_exists("print")) {
	 _print = numeric_param("print") != 0;
	 cout << "set print to: " << _print << endl;
      }
      if (numeric_param_exists("settle")) {
	 _settle = numeric_param("settle");
      }
      _first        = true;
      _processed    = Interval(0.0);
      _current_time = Time(0);
      if (_print) {
	 cout << "DQ_GenRms Parameters: " << endl;
         cout << "rms-max:        " << _rms_limit  << endl;
         cout << "decay time:     " << _decay_time << endl;
         cout << "filter-design:  " << _design     << endl;
         cout << "settle:         " << _settle     << endl;
         cout << "print:          " << _print      << endl;
      }
   }
  
   bool
   bit_value(const tser_vect& data) {
      TSeries ts(data[0]);
      double stride = ts.getInterval();
      if (_current_time != Time(0) && _current_time != ts.getStartTime()) {
	 init();
      }
      if (_first) {
	 if (!_design.empty()) {
	    double rate = 1.0/double(data[0].getTStep());
	    FilterDesign fd(_design.c_str(), rate);
	    _filter.set(fd.release());
	 }
	 if (_decay_time > Interval(0.0)) {
	    _decay_fac = 1.0 - exp(-stride/_decay_time);
	 } else {
	    _decay_fac = 1.0;
	 }
	 _processed = 0.0;
      }
      if (!_filter.null()) ts = _filter(data[0]);
      double rmsi = (ts * ts) / ts.getNSample();
      if (_processed < _settle || _first) {
	 _rms = rmsi;
      } else {
	 _rms = _rms * (1 - _decay_fac) + rmsi * _decay_fac;
      }
      if (_print) {
	 cout.precision(4);
         cout << "DQ_GenRms: gps " << fixed << ts.getStartTime().totalS() 
	      << " rmsi " << sqrt(rmsi) << " rms_avg " << sqrt(_rms) << endl; 
      }
      _first = false;
      _processed += stride;
      _current_time = data[0].getEndTime();
      return sqrt(_rms) <= _rms_limit;
   }

private:
   //       parameters
   std::string _design;
   double      _rms_limit;
   Interval    _decay_time;
   Interval    _settle;
   bool        _print;

   //        state variables.
   bool        _first;
   double      _rms;
   double      _processed;
   auto_pipe   _filter;
   double      _decay_fac;
   Time        _current_time;
};

DQ_PLUGIN(DQ_GenRms)
