#ifndef ConfContainer_HH
#define ConfContainer_HH

#include <iosfwd>
#include <sstream>
#include "TSeries.hh"
#include "Histogram1.hh"
#include "Pipe.hh"
#include "Dacc.hh"

   class EventInfo {
   public:
      enum {
      // normal
      kNone = 0, 
      // during event
      kEvent = 1, 
      // waiting event end
      kWait = 2
      
      };
   
      int fEventStatus;
      Time fStartTime;
      Time fEndTime;
      Time fMaxTime;
      double fMaxData;
      double fMaxSDev;
      double fMaxMean;
      int fEventNPoint;
      std::string fChannel;
      std::string fIfo;
      std::string fComment;
   
      EventInfo () : fEventStatus (kNone), fMaxData (0.0), fMaxSDev (0.0), 
      fMaxMean (0.0), fEventNPoint (0) {
         fChannel = "";
         fIfo = "";
         fComment = "";
      }
   
      EventInfo ( const EventInfo& einfo ) : fEventStatus (kNone), fMaxData (0.0),
      fMaxSDev (0.0), fMaxMean (0.0), fEventNPoint (0) {
         *this = einfo;
      }
   
      ~EventInfo () {
      }
   
      void Dump ( std::ostream& out );
   
      Interval GetDuration () const {
         return (fEndTime < fStartTime) ? Interval(0) : fEndTime - fStartTime;
      };
   
      Interval GetPeakTime () const {
         return (fMaxTime < fStartTime) ? Interval(0) : fMaxTime - fStartTime;
      }
   
      void Clear () {
         fEventStatus = kNone;
         fStartTime = Time ( 0 );
         fEndTime = Time ( 0 );
         fMaxTime = Time ( 0 );
         fMaxData = 0.0;
         fMaxSDev = 0.0;
         fMaxMean = 0.0;
         fEventNPoint = 0;
      }
   
      EventInfo& operator= (const EventInfo& einfo);
   };

   class FilterInfo {
   public:
      std::string fName;
      std::string fFormula;
   
      int fLPEFLength;
      int fLPEFTrainPeriod;
      int fLPEFTrainLength;
   
   	/** Constructor.
          @memo Constructor
       ******************************************************************/
      FilterInfo() : fLPEFLength(0), fLPEFTrainPeriod(0), fLPEFTrainLength(0) {
         fFormula = "gain(1)";
      }
   
      FilterInfo( const FilterInfo& finfo ) : fLPEFLength(0), fLPEFTrainPeriod(0), fLPEFTrainLength(0) {
         *this = finfo;
      }
   
      ~FilterInfo () {
      }
   
      void Dump ();
   
      FilterInfo& operator= ( const FilterInfo& finfo ) {
      
         fName = finfo.fName;
         fFormula = finfo.fFormula;
      
         fLPEFLength = finfo.fLPEFLength;
         fLPEFTrainPeriod = finfo.fLPEFTrainPeriod;
         fLPEFTrainLength = finfo.fLPEFTrainLength;
      
         return *this;
      }

     bool operator == ( const FilterInfo& finfo ) const {
       return((fName == finfo.fName) &&
	      (fFormula == finfo.fFormula) &&
	      (fLPEFLength == finfo.fLPEFLength) &&
	      (fLPEFTrainPeriod == finfo.fLPEFTrainPeriod) &&
	      (fLPEFTrainLength == finfo.fLPEFTrainLength)); 
     }

   
      Pipe* GetFilter ( double fs );
   
      int GetLPEFLength() const { 
         return fLPEFLength; }
   
      int GetLPEFTrainPeriod() const { 
         return fLPEFTrainPeriod; }
   
      int GetLPEFTrainLength() const { 
         return fLPEFTrainLength; }
   
      const char* GetFilterFormula() const { 
         return fFormula.c_str(); }
   
     std::string GetFilterFormulaAll() const;
   };

   class ThresholdInfo {
   public:
      enum ThresholdType {
      kAbs = 0,
      kRel = 1
      };
   
   	/// threshold type (absolute or relative).
      ThresholdType fType;
   
   	/// threshold (low).
      float fLow;
   
   	/// thresholod (high).
      float fHigh;
   
   	/** Constructor.
          @memo Constructor
       ******************************************************************/
      ThresholdInfo () : fType (kAbs), fLow (0.0), fHigh (0.0) {
      }
   
      ThresholdInfo ( const ThresholdInfo& tinfo ) : fType (kAbs), fLow (0.0), fHigh (0.0) {
         *this = tinfo;
      }
   
      ~ThresholdInfo () {
      }
   
      void Dump ();
   
      ThresholdInfo& operator= ( const ThresholdInfo& tinfo ) {
         fType = tinfo.fType;
         fLow = tinfo.fLow;
         fHigh = tinfo.fHigh;
      
         return *this;
      }

      bool operator == ( const ThresholdInfo& tinfo ) const {
	return((fType == tinfo.fType) &&
	       (fLow == tinfo.fLow) &&
	       (fHigh == tinfo.fHigh));
      }
   

   };

   class Parameter {
   public:
   
   	/// minimum separation time between events.
      Interval fMinSeparation;
   
   	/// minimum event duration.
      Interval fMinDuration;
   
   	/// maximum suspension time (sec).
      Interval fMaxDuration;
   
   	/// minimum data point density.
      float fMinDensity;
   
   	/** Constructor.
          @memo Constructor
       ******************************************************************/
      Parameter () : fMinDensity (0.0) {
         fMinSeparation = Interval ( 0.0 );
         fMinDuration = Interval ( 0.0 );
         fMaxDuration = Interval ( 300.0 );
      }
      Parameter ( const Parameter& param ) : fMinDensity (0.0) {
         *this = param;
      }
   
      ~Parameter () {
      }
   
      void Dump ();
   
      Parameter& operator= ( const Parameter& param ) {
         fMinSeparation = param.fMinSeparation;
         fMinDuration = param.fMinDuration;
         fMaxDuration = param.fMaxDuration;
         fMinDensity = param.fMinDensity;
      
         return *this;
      }

      bool operator == ( const Parameter& param ) const {
	return((fMinSeparation == param.fMinSeparation) &&
	       (fMinDuration == param.fMinDuration) &&
	       (fMaxDuration == param.fMaxDuration) &&
	       (fMinDensity == param.fMinDensity));
      } 
     
   };

   class HistogramInfo {
   public:
      int fNBins;
      float fLow;
      float fHigh;
   
      HistogramInfo() : fNBins (500), fLow (0.0), fHigh(500.0) {
      }
   
      HistogramInfo( const HistogramInfo& h ) : fNBins (500), fLow (0.0), fHigh(500.0) {
         *this = h;
      }
   
      ~HistogramInfo() {
      }
   
      HistogramInfo& operator= ( const HistogramInfo& h ) {
         fNBins = h.fNBins;
         fLow = h.fLow;
         fHigh = h.fHigh;
      
         return *this;
      }
   
      bool operator == ( const HistogramInfo& h ) const {
	return((fNBins == h.fNBins) &&
	       (fLow == h.fLow) &&
	       (fHigh == h.fHigh));
      } 


      Histogram1 CreateHistogram ( const char* title = "", const char* xlabel = "", const char* nlabel = "" ) {
         return Histogram1 ( title, fNBins, fLow, fHigh, xlabel, nlabel );
      }
   };

#endif
