/* file burstMon.hh 
   author: Sergey Klimenko, Andrey Sazonov, University of Florida
*/


#ifndef BURSTMON_HH
#define BURSTMON_HH

#include "DatEnv.hh"
#include "MonServer.hh"
#include "OperStateCondList.hh"
#include "Trend.hh"
#include "wseries.hh"
#include "LineFilter.hh"
#include "FDEasyCalibrate.hh"
#include "PSD.hh"

typedef char * pchar;

class ostream;

namespace xsil {
  class Xwriter;
}


class BurstMon : public DatEnv, MonServer {

  public:
  
    BurstMon(int argc, const char *argv[]);     // Constructor
    ~BurstMon();                          	// Destructor

    /** data processing function
     *  @memo standard required function for monitors
     */
    void ProcessData(void);

    /** skeleton function needed for writing to the dmt viewer
     *  @memo standard required function for monitors
     */
    void Attention(void);

    /** print out BurstMon usage information
     *  @memo provides help information
     */
    void dumpHelp();

    /** read input parameters from configuration file
     *  @memo read configuration file
     *  @param input file name.
     *  @return true if .
     */
    bool readConfig(char *);

    /** read lock loss information from file
     *  @memo lock loss 
     *  @param offline	offline flag
     *  @return true if no errors
     */
    void readLockLoss(bool offline);

//    /** set selection cuts in WSeries object 
//     *  @memo set cuts on cluster size etc
//     *  @param wseries
//     *  @return return number of clusters passed the cuts
//     */
//    size_t setCuts(WSeries<double>&);

    /** print brief summary of parameters and current results to file
     *  @memo summary to file
     *  @param			file name
     */
    void dumpSummary(char*);

// procedures helping to read in parameters from files
    char *getToken(pchar &, bool = false);
    inline bool isParameter(const char *);
    inline bool getParameter(char *, const char *, int&);
    inline bool getParameter(char *, const char *, double&);

    /** check access permissions for given directory
     *  @memo access check
     *  @param name 		directory name
     *  @param permission	R_OK, W_OK, X_OK or bitwise combination
     *  @return 		exits with error if permission is not right
     */
    bool check_directory_access(char *name, int permission);

// procedures for injections

    /** read waveform from file
     *  @memo read waveform from file
     *  @param file name	file name with ASCII presentation of waveform
     *  @return waveform peak time, return waveform in wavearray
     */
    double read_waveform(wavearray<double> &wa, char* file_name);

    /** whiten signal by using noise estimates from WSeries::white
     *  @memo normalize signal by noise and scale to noise hrss
     *  @param signal	injection data
     *  @param noise	noise estimates
     *  @return noise hrss
     */
    double white(WSeries<double> &signal, WSeries<double> &noise, bool=true);

    /** return seed hrss corresponding to SNR 1
     *  @memo return seed hrss corresponding to SNR 1
     *  @param signal	injection data
     *  @param noise	noise estimates
     *  @return noise hrss
     */
    double getHRSS(WSeries<double> &signal, WSeries<double> &noise){
      return white(signal,noise,false);
    }

    /** fill gaps or reinitilize history time series for DMTViewer
     *  @memo synchronize history time series
     *  @param ts		history time series 
     *  @param value		fill history gaps with this value
     */
    void sync_history(TSeries *ts, float value = 0.);

    /** update all waveform trends and histories using current or dummy values
     *  @memo update waveform trends
     *  @param flag		set 'false' to use dummy values for updates
     */
    void update_trend_waveform(const bool flag=false);

    /** update trends and history using supplied values
     *  @memo update trend by data array
     *  @param data		pointer to data source
     *  @param size		data length
     */
    void update_trend_nvar(wavearray<float>&);

    /** update trend and history using current or dummy value
     *  @memo update trend by single number
     *  @param flag		set 'false' to use dummy values for update
     *  @param value		current value to put in trend or history
     */
    void update_trend_freq(bool flag=false, float value=0.);

    /** update trend and history using current or dummy value
     *  @memo update trend by single number
     *  @param flag		set 'false' to use dummy values for update
     *  @param value		current value to put in trend or history
     */
    void update_trend_SNR(bool flag=false, float value=0.);

    /** update trend and history using current or dummy value
     *  @memo update trend by single number
     *  @param flag		set 'false' to use dummy values for update
     *  @param value		current value to put in trend or history
     */
    void update_trend_time(bool flag=false, float value=0.);

    /** update trend and history using current or dummy value
     *  @memo update trend by single number
     *  @param flag		set 'false' to use dummy values for update
     *  @param value		current value to put in trend or history
     */
    void update_trend_rate(bool flag=false, float value=0.);

    /** update trend and history using current or dummy value
     *  @memo update bqf trend from wavearray 
     */
    void update_trend_mpf(wavearray<float>* = NULL);

    std::string getCalFileName(std::string);
  

  private:

    pchar pString;

    char masterChannelName[256];

// trends and histories for master channel
    char trendName_clusterRate[256]; 	// cluster rate trend name
    TSeries *history_clusterRate;	// cluster rate (per sec) history

    char trendName_mpf1[256];	// multiple pixel fraction 1
    TSeries *history_mpf1;	// history for bqf1
    char trendName_mpf2[256];	// multiple pixel fraction 2
    TSeries *history_mpf2;	// history for bqf2
    char trendName_mpf3[256];	// multiple pixel fraction 3
    TSeries *history_mpf3;	// history for bqf3

    char trendName_pxr1[256];	// pixel rate 1
    TSeries *history_pxr1;	// history
    char trendName_pxr2[256];	// pixel rate 2
    TSeries *history_pxr2;	// history
    char trendName_pxr3[256];	// pixel rate 3
    TSeries *history_pxr3;	// history

    char trendName_nvar[256];	// noise variability trend name
    TSeries *history_nvar;	// noise variability history

    char trendName_freq[256];   // frequency of the most significant cluster
    TSeries *history_freq;      // history

    char trendName_SNR[256];    // SNR of the most significant cluster
    TSeries *history_SNR;       // history

    char trendName_time[256];    // time of the most significant cluster
    TSeries *history_time;       // history

    char configFile[256];
    char indexFile[256];
    char summaryFile[256];
    char responseFile[256];
    char lockCondition[256];
    char suffix[256];		// additional suffix to monitor name
    char gpsTime[256];		// start GPS time, used to construct file names
      
    double   timeGate;		// coincidence time gate
    double   pFraction;		// pixel fraction selected for cluster analysis
    double   ifoLength;		// interferometer arm length
    double   timeStride;	// time stride for input data
    double   Threshold;   	// threshold on cluster significance
    double   targetFAR;   	// false alarm rate for setting rSF threshold
    double   samplingRate;	// data sampling rate for analysis
    double   resolution;	// frequency resolution for cluster analysis 
    double   injectionWindow;   // injection window
    double   wResolution;       // frequency resolution for data conditioning
    double   liveTime;		// processed data length
    double   deadTime;		// missed data length
    double   totalTime;		// run time of monitor
    double   offsetTime;	// length of data padding
    double   startTime;		// first data start time
    int      minClusterSize;	// parameter for cluster analysis
    bool     includeHalo;	// parameter for cluster analysis
    int      historyLength;   	// lengths of data for DMTviewer
    int      verbose;		// stdout verbosity level 0..10
    int      waveOrder;		// Symlet order for cluster analysis
    bool     firstCall;		// true at first data input
    bool     offLine;		// true if monitor works offline
    bool     ezCal;		// true if EZ calibration is used
    int      whiteLevel;	// DWT transform level for whitening
    int      analyLevel;	// DWT transform level for cluster analysis
    bool     approxLevel;	// flag for including approximation level 

    //  "Easy" calibration object for handling ASQ->strain(nm) conversion.
    FDEasyCalibrate *mCalibrate;
    FSpectrum modFResp;

    Time  currentTime;		// start time of data being processed
    float clusterRate;		// current cluster rate

    Trend mTrend;		// minute trend - cluster rate, h50, h50_err
    Trend sTrend;		// second trend - noise variability
    OperStateCondList mOSC;

    TSeries *masterTS;		// input buffer for data from master channel

// for waveforms 
    std::vector<char *> waveformFileName;
    std::vector<double> peakTime;		// waveforms peak times
    std::vector<wavearray<double> *> wf;	// waveforms
    std::vector<WSeries<double> *>  wfT;	// transformed waveforms
    std::vector<WSeries<double> *>  wfS;	// scaled waveforms
    std::vector<double> h50;		        // estimated amplitude
    std::vector<double> err_h50;		// estimated error, %
    std::vector<double> amp;	                // injection amplitudes
    std::vector<double> ADC;	                // seed value of hrss in ADC counts
    std::vector<char *> trendName_h50;		// for hrss_50 trend
    std::vector<char *> trendName_noise;	// for noise trend
    std::vector<TSeries *> history_h50;		// for DMTviewer - 50% HRSS
    std::vector<TSeries *> history_h50err;	// for DMTviewer - hrss error
    std::vector<TSeries *> history_noise;	// for DMTviewer - noise hrss

// injection parameters
    int Injections;		// # of injections per waveform

};

inline bool BurstMon::isParameter(const char *s){
   int n = strlen(s);
   bool isP = (n>0) ? true : false;
   for(int i=0; i<n; i++)
      if(!isdigit(s[i]) && s[i] != '-' && s[i] != '.')
	 isP = false;
   return isP;
}

inline bool BurstMon::getParameter(char *w, const char *c, int &x){
  if ( strcmp(w,c) == 0 ){
    if( isParameter(w = getToken(pString)) )
      sscanf(w,"%d",&x);
    return true;
  }
  return false;
}

inline bool BurstMon::getParameter(char *w, const char *c, double &x){
  if ( strcmp(w,c) == 0 ){
    if( isParameter(w = getToken(pString)) )
      sscanf(w,"%lf",&x);
    return true;
  }
  return false;
}

// survival probability for Gamma distribution
/*
double gammaCL(double x, int n){   
   double z,y,s;
   z = 0;

   if(x<=n){
     y = x; s = 1.;
     for(int k=n-1; k>0; k--){ 
       s += y; 
       y*=double(x)/(n-k+1); 
       if(y>1.e290) break;
     }
     return double(x-log(s));
   }
   else{
     y = n/x; s = n/x; z = log(x/n);
     for(int k=n-1; k>0; k--){ 
       y *= double(k/x); 
       s += y; 
       z += log(x/k); 
       if(y<1.e-290) break;
     }
     return float(x-z-log(s));
   }
}
*/
#endif     //  BurstMon_HH




