// This class stores the coefficients, history buffers, etc
// for the instance of a single IIR filter on a single data
// channel.
// Edward Daw, 4th September 2000
#ifndef IIRfilter_instance_HH
#define IIRfilter_instance_HH

/** This class contains all the filter coefficients and memory buffers
 *  for application of an IIRfilter to a specific data set. The filter
 *  coefficients are initiated from a filter bank, which is currently
 *  read from a text file as described in the documentation for
 *  the class IIRfilter_bank.
 *  @author Edward Daw
 *  @version 1.1; Last modified October 4th, 2000
 */

class IIRfilter_instance {

public:
  /** Default (Null) constructor. */
  IIRfilter_instance ( );

  /** Initialize a filter instance using a filter from some filter bank.
   * @memo Memory is allocated, the contents of the history buffers
   * are initialized to zero, an the filter coefficients are set.
   * @param filter_index 0 for 1st filter, 1 for 2nd, etc...
   * @param f_bank Pointer to filter bank containing filter to be used.
   */
  int init_filter_instance( int filter_index, IIRfilter_bank* f_bank );

  /** Dump contents of history buffers associated with current state
   * of filter for debugging purposes.
   */
  int dump_filter_contents( );

  /** Apply the filter to a float array. Output filtered data and (optionally)
   * output of filter.
   * @memo a second method, coming soon, will apply the filter to a 
   * time series.
   * @param data_length Length of float array.
   * @param input_data Pointer to input data array.
   * @param output_data Pointer to output data array. (must be allocated
   * same memory space as input array if get_output_data is set to 1.
   * @param pss Memory location where sum of the
   * squares of the filtered data points is written.
   * @param make_file If 1, a text file is made containing the input data
   * and the filtered data. If 0, no file is made.
   * @param test_file_name A character array containing the file name 
   * (max length 30 characters).
   * @param get_output_data If 1, the filtered data is written to 
   * output_data. If 0, no filtered data is written.
   */ 
  int apply_filter( int data_length, float* input_data, float* output_data,
		    float* pss, int make_file, char* test_file_name,
		    int get_output_data );

  /** Destructor. Frees up memory allocated to filter instance. */
  ~IIRfilter_instance( );

  int sampling_rate;
  char filter_name[30];

private:

  second_order_section* sos_set;
  double output;
  int no_of_sos_in_filter;
};

#endif // IIRfilter_instance_HH
