// functions for setting up application of digital
// filters to specific sets of channel data.
//
// Edward Daw, 4th September 2000

#include "logicals.h"
#include "IIRfilter_structures.hh"
#include "IIRfilter.hh"
#include "IIRfilter_bank.hh"
#include "IIRfilter_instance.hh"
#include <stdio.h>
#include <math.h>

void sos_apply( second_order_section* sec );

// default constructor just builds an empty filter instance
IIRfilter_instance::
IIRfilter_instance( )
{
  sos_set = NULL;
  output = 0.0;
  no_of_sos_in_filter = 0;
  sampling_rate = 0;
}

// initialize the second order sections for the filter instance
int
IIRfilter_instance::
init_filter_instance( int filter_index,
		      IIRfilter_bank* f_bank )
{
  // initialize filter order.
  f_bank->get_order( filter_index, &no_of_sos_in_filter);
 
  // initialize sampling rate.
  f_bank->get_sampling_rate( filter_index, &sampling_rate );

  // allocate memory for second order sections
  sos_set = new second_order_section [ no_of_sos_in_filter ];

  // set filter name
  sprintf(filter_name,"%s",f_bank->get_filter_name( filter_index) );

  // loop over second order sections initializing elements
  int sos_count;
  for(sos_count = 0;
      sos_count < (no_of_sos_in_filter);
      sos_count ++) {

    // hook the output of one second order section to the input of
    // the next, or the output of the chain for the last one

    (sos_set + sos_count)->in = 0.0;
    if(sos_count < (no_of_sos_in_filter - 1) ) {
      (sos_set + sos_count)->pout = &((sos_set + sos_count + 1)->in);
    }
    else {
      (sos_set + sos_count)->pout = &output;
    }

    // copy the filter coefficients into the second order section structures

    int i;
    for(i=0;i<3;++i) {
      (sos_set + sos_count)->shift_reg[i] = 0.0;
    }
    
    f_bank->get_b0( filter_index,
		    sos_count,
		    &((sos_set + sos_count)->b[0]) );
    f_bank->get_b1( filter_index,
		    sos_count,
		    &((sos_set + sos_count)->b[1]) );
    f_bank->get_b2( filter_index,
		    sos_count,
		    &((sos_set + sos_count)->b[2]) );
    f_bank->get_a1( filter_index,
		    sos_count,
		    &((sos_set + sos_count)->a[1]) );
    f_bank->get_a2( filter_index,
		    sos_count,
		    &((sos_set + sos_count)->a[2]) );

    // set the shift counter to zero

    (sos_set + sos_count)->shift_count = 0;
  }
  
  return 0;
}

int
IIRfilter_instance::dump_filter_contents ( )
{
  int sos_count;
  
  for(sos_count = 0;
      sos_count < no_of_sos_in_filter;
      sos_count++ ) {

    printf("history %d  %f : %f  %f  %f : %f  %d\n",
	   sos_count+1,
	   (sos_set+sos_count)->in,
	   (sos_set+sos_count)->shift_reg[0],
	   (sos_set+sos_count)->shift_reg[1],	   
	   (sos_set+sos_count)->shift_reg[2],
	   *((sos_set+sos_count)->pout),
	   (sos_set+sos_count)->shift_count );
    
  }
  return 0;
}

// pss is a pointer to the sum of the squares of the
// filtered data. It replaces prms which was the root mean
// square of the filtered data. 

int
IIRfilter_instance::apply_filter( int data_length,
				   float* input_data,
				   float* output_data,
				   float* pss,
				   int make_file,
				   char* file_name,
				   int get_output_data )
{
  int data_count;
  int sos_count;
  double data_sum_squared = 0;
  float* out_data(0x0);

  // if make_file is yes, setup output data array.
  if( make_file == YES ) {
    out_data = new float [data_length];
  }

  // loop over data elements
  for(data_count = 0;
      data_count < data_length;
      data_count++ ) {

    sos_set->in = (double)input_data[ data_count ];

    // loop over second order sections
    for( sos_count = 0;
	 sos_count < no_of_sos_in_filter;
	 sos_count ++ ) {
      
      sos_apply( sos_set + sos_count );

    }

    // if the output data is needed, send it to the output_data array
    if( get_output_data == YES ) {
      output_data[ data_count ] = (float)output;
    }

    if( make_file == YES ) {
      out_data[ data_count ] = (float)output;
    }

    data_sum_squared += (output*output);
  }

  // if a file is required of the input and output data, send the
  // input and output data to the test file.
  if( make_file == YES ) {
    FILE* out;
    out = fopen(file_name,"a");
    for( data_count = 0;
	 data_count < data_length;
	 ++data_count ) {
      fprintf(out,"%d\t%f\t%f\n", 
	      data_count, 
	      input_data[ data_count ],
	      out_data[ data_count ] );
    }
    fclose(out);
    delete [] out_data;
  }
  
  // this line modified on 2001_03_27 to just calculate the sum of
  // the squares
  //*prms = (float)sqrt(data_sum_squared / data_length);
  *pss = (float)data_sum_squared;
  return 0;
}

IIRfilter_instance::~IIRfilter_instance ( )
{
  // delete the second order section array
  delete [] sos_set;
}








