//
//   Class StochMon: Base class for Stochastic Background Sensitivity Monitor 
//   version: n/a ($Date: 2018-01-11 00:18:07 -0800 (Thu, 11 Jan 2018) $; $Id: StochMon.hh 7987 2018-01-11 08:18:07Z john.zweizig@LIGO.ORG $)
//   authors: Marc J. Cenac (mjcenac@loyno.edu)
//            Brian O'Reilly (irish@ligo-la.caltech.edu)
//            John T. Whelan (jtwhelan@loyno.edu)
//            Kevin C. Schlaufman (kcs149@psu.edu)
//            Patrick J. Sutton (psutton@ligo.caltech.edu)
//
///////////////////////////////////////////////////////////////////////////


#ifndef StochMon_HH
#define StochMon_HH

#ifndef __CINT__

#undef StochMon_debug
//-- Standard library stuff
#include <string>
#include <fstream>
#include <iostream>
#include <stdio.h> 
#include <signal.h>

//-- DMT-specific headers
#include "DatEnv.hh"
#include "Hamming.hh"
#include "Hanning.hh"
#include "FlatTop.hh"
#include "Blackman.hh"
#include "FSpectrum.hh"
#include "FSeries.hh"
#include "PSD.hh"
#include "MonServer.hh" 
//#include "LineFilter.hh"
#include "Trend.hh"
#include "OperStateCondList.hh"

#endif  //------ !def(__CINT__)

//-- Monitor-specific headers
#include "FDEasyCalibrate.hh"
#include "LLOLHO_Overlap.hh" 
#include "StochSensSummary.hh"
#include "PSD.hh"


/** Helper struct for StochMon class. 
  * Holds various parameters for the current 
  * invocation of StochMon.
  * @memo Holds parameters for the current invocation of StochMon.
  */
struct Parameter
{
    /** Number of intervals to divide time-series data into when calculating 
      * the noise PSD.  Must divide the length 'T' of the time 
      * series for the PSD class to be invoked. 
      */
    int num_ints;
    
    /// Length (sec) of data to use for each stochastic sensitivity estimate.
    double T; 

    /// Highest frequency used in stochastic sensitivity estimate.
    double h_freq;  

    /// Lowest frequency used in stochastic sensitivity estimate.
    double l_freq;  

    /// Number of strides in lock between each dump of a calibrated AS_Q PSD.
    int stride_dump;

    /// Name of uber xml file holding reference calibration data for first IFO.
    std::string xml_file_name_1; 

    /// Name of uber xml file holding reference calibration data for second IFO.
    std::string xml_file_name_2;

    /// Name of configuration file holding monitor parameters.
    std::string config_file_name;  

    ///  Interferometer 1 (one of H1, H2, or L1).
    std::string IFO_1;

    /// Interferometer 2 (one of H1, H2, or L1).  
    std::string IFO_2;

    /// Channel name.  Holds <IFO>:LSC-AS_Q, where <IFO> is one of H1, H2, or L1.
    std::string mChannel;
    std::string mChannel_2;

    /// Name of file holding open-loop-gain data for dynamic calibrations.
    std::string olg_file_name; 

    /** Name of configuration file holding operating state condition (OSC) 
      * definitions for the IFO.
      */
    std::string osc_file_name; 

    //-- Name of calculated psd text file
    std::string CalPSD_file_name;
    
  /** Operating state condition that must be satisfied for range estimates
   * to be computed.  Must be defined in osc_file_name.
   */
  std::string osc_cond;

    /// Name of window type to use in calculating the noise PSD.
    std::string window_name; 

};


/** Helper struct for StochMon class. 
  * Holds all information on where and what output files will be written.
  * @memo Holds information on output files to be written.
  */
struct Output_Files
{
    //  If true then the monitor will write error messages to file 
    //  <IFO>_StochMon_Error.html (default true).
    bool error_log; 

    //  If true then dump output to local directory
    //  instead of to directory specified by 
    //  environment variable DMTHTMLOUT. 
    bool local; 

    //  If true, output is echoed to the screen (default false).
    bool screen_output;  

    //  If true, StochMon writes to a trend file (default false).
    bool trend;

    //  If true then summary file <IFO>_StochMon_Log.html
    //  is written (default true).
    bool write_log; 

    //  If true, configuration file will be re-read 
    //  Output stream to file holding calibrated noise spectrum.
    std::ofstream calpsd_file;  
    std::ofstream calpsd_file_2;

    //  Name of file holding calibrated noise spectrum.
    //  Holds name of calibrated PSD file name.
    std::string calpsd_file_name;     
    std::string calpsd_file_name_2;

    /// Name (including path) for DMTViewer history file.
    std::string dmtviewer_file_name; 

    //----- Notes: 
    //      1. By default cout is redirected to the log file,
    //         and cerr to the error file. 
    //      2. dmtviewer_file is used as both input stream 
    //         (in the constructor) and output stream (in the 
    //         destructor) so the stream objects are declared 
    //         locally.
    //      3. Naming conventions:  
    //          file streams typically end with _file. 
    //          file names typically end with _file_name (with/without path). 
    //          file names ending with _file_link do not include a path. 

    /// Directory specified in DMTHTMLOUT, where output files go.
    char *html_dir; 
    
    /// Name (including path) for current log file.
    std::string log_file_name;       

    /// Name (excluding path) for current log file.
    std::string log_file_link;  

    /// Output stream for CumLog.txt file.
    std::ofstream cumlog_file;

    /// Name (excluding path) for cumulative log file.
    std::string cumlog_file_link;  

    /// Name (including path) for cumulative log file.
    std::string cumlog_file_name;  

    /// Name (including path) for error file.
    std::string error_file_name;     

    /// Name (excluding path) for error file
    std::string error_file_link;

    //  Name (including path) for html revolver file.
    std::string revolver_file_name;	

    //  Name (including path) for html summary file.
    std::string summary_file_name;   
};


/** Helper struct for StochMon class. 
  * Holds data related to recent stochastic sensitivity estimates. 
  * @memo Holds data related to recent stochastic sensitivity estimates. 
  */
struct StochSens_Data
{
    /// Current value of calibration parameter alpha.
    double alpha;

    /// Current value of calibration parameter beta.
    double beta; 

    /// Current value of calibration line amplitude.
    double ampl;

    /// Current value of gain.
    double gain;  

    /// Previous value of gain.
    double prev_gain;  

    /** Current value of effective stochastic sensitivity,
      * adjusted by calibration-line amplitude.
      */ 
    double invomegasq;
  double omega;

    /// Contains last 5 stochastic sensitivity estimates. (CAL or UNCAL?)
    double T5_array[5];

    /// Contains last 15 stochastic sensitivity estimates. (CAL or UNCAL?)
    double T15_array[15];   

    /// Average sensitivity over last 5 time steps.
    double T5_stochsens;

    /// Average sensitivity over last 15 time steps.
    double T15_stochsens;
};


/** The StochMon class is the base class for a DMT monitor of the same
  * name which produces estimates of one over the square of the Omega
  * parameter of the faintest constant-Omega stochastic background
  * which could be detected with a false alarm rate of 5% and a
  * false dismissal rate of 5% using one stride's worth of data.
  * @memo StochMon class for DMT stochastic sensitivity monitor.
  * @author Marc J. Cenac (mjcenac@loyno.edu) and John T. Whelan (jtwhelan@loyno.edu)
  * @version $Id: StochMon.hh 7987 2018-01-11 08:18:07Z john.zweizig@LIGO.ORG $: Last modified $Date: 2018-01-11 00:18:07 -0800 (Thu, 11 Jan 2018) $.
  */ 
class StochMon : public DatEnv, MonServer {

  public:

    /** Construct a StochMon object.
      * @memo Constructor.
      */ 
     StochMon(int argc, const char *argv[]); 

    /** Destroy a StochMon object.
      * @memo Destructor.
      */ 
    ~StochMon(void); 

    /** Process data statement.  The monitor reads a specified amount 
      * of time-series data for the LSC-AS_Q channel and computes the
      * 1/Omega^2 of the faintest detectable constant-Omega stochastic BG.
      * @memo Process data statement.  
      */
    void ProcessData(void);

    /** Write a brief html page summarizing the monitor and interferometer 
      * status.  The file is named <IFO>_StochMon_Summary.revolver.html, 
      * where <IFO> is one of L1, H1, or H2.  FILE IS WRITTEN TO WHAT DIRECTORY?
      * @memo Write an abbreviated html page summarizing the monitor and interferometer status.
      * @param t Start time of data for which summary is written. 
      * @param Run_Par Struct Parameter holding StochMon parameters.
      * @param R_Dat Struct StochSens_Data holding data on recent sensitivity and calibration.
      */ 
    virtual void RevolverSummary(const Time& t, Parameter Run_Par,
				 StochSens_Data R_Dat);

    /** Write an html page summarizing the monitor and interferometer 
      * status.  The file is named index.html.
      * FILE IS WRITTEN TO WHAT DIRECTORY?
      * @memo Write an html page summarizing the monitor and interferometer status.
      * @param t Start time of data for which summary is written. 
      * @param Run_Par Struct Parameter holding StochMon parameters.
      * @param R_Dat Struct StochSens_Data holding data on recent sensitivity and calibration.
      */ 
    virtual void Summary(const Time& t, Parameter Run_Par, StochSens_Data R_Dat);

    /** Read run parameters from specified configuration file.
      * See documentation for syntax.
      * @memo Read run parameters from specified configuration file.
      * @param filename Name of configuration file. 
      */ 
    virtual void Configure(std::string& filename);  

    /** Read run parameters from command-line arguments.  
      * See documentation for syntax.
      * @memo Read run parameters from command-line arguments.
      */ 
    virtual void Configure(int argc, const char *argv[]);

    /** Perform various checks on monitor parameters and set  
      * bool fatal_error to TRUE if any of the following hold:  
      * stride is not an integer multiple of number of averages;
      * low frequency is not less than high frequency;
      * calibration file(s) cannot be opened;
      * operating state condition (OSC) configuration file cannot be opened. 
      * Also prints warnings under various conditions.
      * @memo Perform various checks on monitor parameters.
      * @param Run_Par Struct Parameter holding StochMon parameters.  
      * @param fatal_error Set to TRUE if parameters in Run_Par are not valid.
      */
    virtual void VerifyParameters(Parameter& Run_Par, bool& fatal_error); 

    /** Dump run parameters to specified output stream. 
      * Dumps all members of specified Parameter except dyncal and 
      * config_file_name, plus MaxStride, 
      * frequency of the calibration line tracked (if any), 
      * frequency and amplitude of the simulated calibration line (if any), 
      * plus open-loop gain and sensing function data if dyncal is TRUE.  
      * Output is human-readable, and each line begins with  
      * the # character (except for calbration data).
      * @memo Dump run parameters to specified output stream.
      * @param file Ouput stream.   
      */ 
    virtual void DumpParameters(std::ostream& file); 

    /** Add a sine wave of specified frequency (Hz) and amplitude 
      * (AS_Q counts) to the specified times series.  
      * Used for testing calibration-line tracking.
      * @memo Add a sine wave of specified frequency and amplitude to the specified times series. 
      * @param ts TSeries* to which sine wave is to be added. 
      * @param frequency Frequency (Hz) of sine wave to be added.
      * @param amplitude Amplitude (AS_Q counts) of sine wave to be added.
      */
    //virtual void AddSimulatedLine(TSeries *ts, double frequency, double amplitude); 

    /** Write column headers for stochastic sensitivity data to specified output stream.  The precise 
      * header information depends on whether the calibration line is tracked.
      * @memo Write column headers for stochastic sensitivity data to specified output stream.
      * @param out Output stream to which to dump headers.
      */ 
    virtual void DumpReportHeader(std::ostream& out);

    /** Write column headers for stochastic sensitivity data to specified output stream.  The precise 
      * header information depends on whether the calibration line is tracked.
      * @memo Write column headers for stochastic sensitivity data to specified output stream.
      * @param out Output stream to which to dump headers.
      * @param frequency Frequency of calibration line to be tracked for calibration adjustments (use zero if no such line).
      */ 
    virtual void DumpReportHeader(std::ostream& out, double frequency);

    /** Write current range and calibration information to the specified 
      * output stream.
      * @memo Write current range and calibration information to the specified output stream.
      * @param out Output stream to which to write information. 
      */
    virtual void ReportResults(std::ostream& out, const Time& t, Parameter& Run_Par);

    /** Act on SIGUSR1 signal (reread configuration file).
      * @memo Act on SIGUSR1 signal.
      */
    //virtual void HandleUsr1();     

    /** Handle signals to monitor.
      * @memo Handle signals to monitor.
      */
    virtual void Attention(void) {
        //----- This call serves data to the DMTViewer. 
    //--- Replaced SIGPOLL with SIGIO Keith Thorne June, 2011
    // on Linux, Solaris SIGPOLL same as SIGIO, Mac OS X only has SIGIO 
	if (testAttnSignal(SIGIO)) MonServer::Attention();
        //----- This call triggers re-reading of calibration information.
	//if (testAttnSignal(SIGUSR1)) HandleUsr1();
    }

  private:

    //  If true then quit cleanly ASAP.
    bool fatal_error; 

    /// Live mode (1L or 2L)
    std::string LiveMode;

    //  Holds last 12 hrs of calibrated stochastic sensitivity values.
    double *invomegasq_values;
  double *omega_values;

    //  Holds last 12 hrs of cal-line-ampl values.
    double *amplvalues;	

    //  Holds last 12 hrs of gain values.
    double *gainvalues;	

    //  Holds last 12 hrs of alpha values.
    double *alphavalues;

    //  Holds last 12 hrs of beta values.
    double *betavalues;	

    //  Holds end time of last stride of data read.  Used to check for gaps in data.
    double prev_endtime;

    //  1000-sec average range reported to 2-week summary file.
    float running_average;

    //  Number of strides contributing to running_avg
    float running_count; 

    //  Total number of strides to be processed.
    int MaxStride; 

    //  Number of strides of data missing between last and current stride.
    int missing_strides;

    //  Number of strides already processed.
    int NStride;

    //  Time since last update of 2-week summary.
    int time_since_update;

    //  Number of time steps in DMTViewer history.
    int tserieslength;

    //  "Easy" calibration object for handling ASQ->strai(nm) conversion.
    FDEasyCalibrate *mCalibrate;
    FDEasyCalibrate *mCalibrate_2;

    //  FSpectrum object that holds the integrand of the stochastic background integral,  
    //  norm*gamma(f)*(f^6 * PSD_1 * PSD_2)^{-1}
    //  See Allen & Romano, PRD, 1999.
    FSpectrum stoch_igd; 

    //-- Holds the PSD of the AS_Q data.  It is filled by PSD::generate().
    FSpectrum PSDed_data_1;
    FSpectrum PSDed_data_2;  
   
    // Holds the calibrated amplitude spectrum of the detector noise (PSD^0.5).
    FSpectrum strain_noise_ampl;
    FSpectrum strain_noise_ampl_2;

    //--Holds the values for overlap reduction function
     FSpectrum GammaValues;

    /// GPS time (sec) at which the monitor is started.
    long int starttime; 
    
    //-- holds number of seconds per stride
    double sec_per_stride; 

    //  Monitors lock state of IFO.
    OperStateCondList *mOsclist;

    //  Struct holds metadata on output files.
    Output_Files mOutput; 

    //  Struct (defined above) that keeps all the run parameters in one place.
    Parameter Run_Par;		

    //  Object calculates the power-spectral density
    //  (stored in an FSpectrum object) of the
    //  TSeries data fed to it.  See PSD.hh for 
    //  further explanation.
    PSD *psd;                   

    //  Struct (defined above) that transports all
    //  of the calculated stochastic sensitivity data in one piece
    //  from the process data method to the summary
    //  method, where the webpage is created.
    StochSens_Data R_Dat;

    //  Range summary data for Szabi's status web page
    StochSensSummary mStochSensSumm;	

    /** Stores monitor arguments as read from command-line 
      * or configuration file; used in writing summary html files.
      */
    std::string command_line;

    /// Directory where monitor was started.
    std::string pwd_dir;  

    //  Trend object to which trend information is sent.
    Trend mTrend;

    //  History of alpha parameter sent to DMTViewer.
    TSeries *history_alpha; 

    //  History of beta parameter sent to DMTViewer.
    TSeries *history_beta; 

    //  History of calibration-line amplitude sent to DMTViewer.
    TSeries *history_ampl;  

    //  History of calibration-line gain sent to DMTViewer.
    TSeries *history_gain;  

    //  History of calibration-adjusted stochastic sensitivity estimates sent to DMTViewer.
    TSeries *history_invomegasq;
    TSeries *history_omega;

    //  History of non-calibration-adjusted stochastic sensitivity estimates sent to DMTViewer.
    TSeries *history_invomegasq_uncal; 

    //  Pointers to time series asq data.
    TSeries *ts_asq; 
    TSeries *ts_asq_2;

    //  Pointer to Hamming window.
    Hamming *ham;               

    //  Pointer to Hanning window.
    Hanning *han;               

    //  Pointer to FlatTop window.
    FlatTop *flat;              

    //  Pointer to Blackman window.
    Blackman *black;            

//:KLUDGE: New members for "truly dynamic" calibrations.
    //  Pointer to time series excitation channel data.
    TSeries *ts_exc;  

    //  Pointer to array of time series to hold DARM gain channel data.
    TSeries **ts_darm;  
//:END:KLUDGE:
};

#endif     //----------  StochMon_HH
