/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "SegGener.hh"
#include "ParseLine.hh"
#include "Segment.hh"
#include "html/color.hh"
#include "html/document.hh"
#include "html/Table2C.hh"
#include "html/writer.hh"
#include <unistd.h>
#include <sys/stat.h>
#include <fstream>

using namespace std;
using namespace html;

//======================================  Identify process
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/SegGener/SegGener.cc 6963 2014-01-16 10:02:10Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Segment generation from an OSC condition"
#include "ProcIdent.hh"

//======================================  Generate a main routine.
EXECDAT(SegGener)

//======================================  Construct an On/Off segment
SegCond::SegCond(void)
: mVersion(0), mLimit(0.0), mType(kUnknown), mDefined(false), mPrevious(false), 
    mInSegment(false), mStart(0), mStop(0), mLiveTime(0.0), mNSegment(0)
{}

//======================================  Construct an On/Off segment
SegCond::SegCond(const std::string& group, int v, const std::string& on)
  : mGroup(group), mVersion(v), mLimit(0.0), mType(kOnCond), mOnCond(on),
    mDefined(false), mPrevious(false), mInSegment(false), mStart(0), mStop(0), 
    mCurrent(0), mLiveTime(0.0), mNSegment(0)
{}
//======================================  Construct a stop/start segment
SegCond::SegCond(const std::string& group, int v, const std::string& on, 
	  const std::string& off)
  : mGroup(group), mVersion(v), mLimit(0.0), mType(kStartCond), mOnCond(on), 
    mOffCond(off), mDefined(false), mPrevious(false), mInSegment(false), 
    mStart(0), mStop(0), mCurrent(0), mLiveTime(0.0), mNSegment(0)
{}

//======================================  Destroy the segment definition.
SegCond::~SegCond(void) {
}

//======================================  Evaluate the segment state
bool
SegCond::evaluate(OperStateCondList& l, const Time& t, Interval dT) {
    mPrevious = mInSegment;

    //----------------------------------  Evaluate segment condition
    switch (mType) {
    case kOnCond:
      mInSegment = l.satisfied(mOnCond);
      break;

    case kStartCond:
      if (mPrevious && l.satisfied(mOffCond)) mInSegment = false;
      if (!mPrevious && l.satisfied(mOnCond)) mInSegment = true;
      break;

    default:
      mInSegment = false;
      break;
    }

    //------------------------------------  Test for complete
    bool segComplete = mPrevious != mInSegment;
    if (!mCurrent) {
        mStart = t;
	mStop  = t;
	segComplete = false;
    } else if (segComplete) {
        if (mPrevious) mStop  = t;
	else           mStart = t;
	if (mStart == mStop) segComplete = false;
    }

    //------------------------------------  Calculate time in segment
    mCurrent      = t + dT;
    Interval delt = mInSegment ? mCurrent-mStart : mCurrent-mStop;
    segComplete  |= (mLimit != Interval(0.0) && delt >= mLimit);
    if (mInSegment) mLiveTime += dT;
    return segComplete;
}

//======================================  Setup for final segment generation.
bool
SegCond::flush(void) {
    if (!mCurrent) return false;
    mPrevious = mInSegment;
    if (mPrevious) mStop  = mCurrent;
    else           mStart = mCurrent;
    return (mStart != mStop);
}

//======================================  Make a Segment
//
//   The segment actions should go as follows:
//
//   Prev  This  begin   end    action         comment
//    on    on   start  current start=current  Segment limit
//    on    off  start  stop    start=current  transition->off
//    off   on   stop   start   stop =current  transition->on
//    off   off  stop   current stop =current  Segment limit 
//
trig::Segment
SegCond::makeSegment(void) {
    Time tBegin, tEnd;
    if (mPrevious) {
        tBegin = mStart;
	tEnd   = mStop;
	mStart = mCurrent;
    } else {
        tBegin = mStop;
	tEnd   = mStart;
	mStop  = mCurrent;
    }
    if (mPrevious == mInSegment) tEnd = mCurrent; 

    trig::Segment s(mGroup, mVersion, tBegin, tEnd);
    if (!mPrevious) s.setActivity(0);
    s.setIfos(mIfo.c_str());
    s.setComment(mComment);
    mNSegment++;
    return s;
}

//======================================  Reset segment state.
void 
SegCond::reset(void) {
    mCurrent = Time(0);
    mDefined = false;
}

//======================================  Set the segment list file.
void 
SegCond::setComment(const std::string& comment) {
    mComment = comment;
}

//======================================  Set the segment list file.
void 
SegCond::setIfo(const std::string& ifos) {
    mIfo = ifos;
}

//======================================  Set the maximum segment time.
void 
SegCond::setLimit(Interval dT) {
    mLimit = dT;
}

//======================================  Set the segment list file.
void 
SegCond::setSegFile(const std::string& file) {
    const char* htmdir = getenv("DMTHTMLOUT");
    if (file.empty() || file[0] == '.' || file[0] == '/' || !htmdir) {
        mSegFile = file;
    } else {
        mSegFile  = htmdir;
	mSegFile += "/";
	mSegFile += file;
    }

    //----------------------------------  Complain if the file can't be written
    if (!mSegFile.empty()) {
        if (access(mSegFile.c_str(), W_OK) && !access(mSegFile.c_str(), F_OK)){
	    cout << "Can't write file: " << mSegFile 
		 << " disabling text file output for segment: " << mGroup 
		 << endl;
	    mSegFile.clear();
	}
    }
}

//======================================  Test the required OSC conditions.
bool
SegCond::valid(OperStateCondList& l) {
    bool isOK = l.defined(mOnCond);
    if (mType == kStartCond && !l.defined(mOffCond)) isOK = false;
    mDefined = isOK;
    return isOK;
}

//======================================  Write segment to the list file.
void 
SegCond::writeFile(const trig::Segment& s) const {
    if (!mSegFile.empty() && s.getActivity() > 0) {
	ofstream otxt(mSegFile.c_str(), ios::app);
	if (otxt.is_open()) {
	    otxt << mNSegment << " " << s.getStartTime().getS() 
		 << " " << s.getEndTime().getS() << " " 
		 << long(s.getEndTime() - s.getStartTime()) << endl;
	}
    }
}

//======================================  File and Date
#define MAXPATHL 2048

static std::string
FileAnDate(const std::string& file) {
    std::string out;
    struct stat statbuf;
    if (file[0] != '/') {
        char cwd[MAXPATHL];
	out  = getcwd(cwd, MAXPATHL);
	out += "/";
    }
    out += file;
    out += " [";
    stat(file.c_str(), &statbuf);
    Time tMod = fromUTC(statbuf.st_mtime);
    char datime[32];
    out += TimeStr(tMod, datime, "%3M %d, %Y %02H:%02N %Z");
    out += "]";
    return out;
}

//======================================  Write a segment message
static void
logSegment(const trig::Segment& s, std::ostream& out) {
    out << "Segment "     << (s.getActivity() ? "on" : "off")
	<< " group: "     << s.getGroup() 
	<< " version: "   << s.getVersion() 
	<< " Started: "   << s.getStartTime().getS() 
	<< " and ended: " << s.getEndTime().getS() 
	<< endl;
}

//======================================  Monitor constructor.
SegGener::SegGener(int argc, const char* argv[]) 
    : DatEnv(argc, argv), mConfig("SegGener.conf"), mOSC(getDacc()), 
      mRecord(false), mFirst(0), mLast(0), mSpan0(0), mSpan1(0)
{
    //----------------------------------  Parse the argument list
    bool syntax = false;

    for (int i=1 ; i<argc ; i++) {
        string argi(argv[i]);
	if (argi == "-conf") {
	    mConfig = argv[++i];
	} else if (argi == "-htfile") {
	    mHtFile = argv[++i];
	} else if (argi == "-osc") {
	    mOSCfile = argv[++i];
	} else if (argi == "+seg") {
	    mRecord = true;
	} else if (argi == "-span") {
	    const char* p=argv[++i];
	    Time::ulong_t t = strtoul(p, const_cast<char**>(&p), 0);
	    mSpan0 = Time(t);
	    if (*p++ == ':') {
		t = strtoul(p, const_cast<char**>(&p), 0);
		mSpan1 = Time(t);
	    }
	} else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    syntax = true;
	    break;
	}
    }

    //----------------------------------  Check syntax
    if (mOSCfile.empty()) {
        cerr << "OSC file not specified!" << endl;
	syntax = true;
    }

    if (syntax) {
        cerr << "Syntax error discovered in command. Syntax is:" << endl;
        cerr << "SegGener [-conf <config-file>] -osc <OSC-file> [+seg]" 
	     << endl;
	finish();
	return;
    }

    //----------------------------------  Configure the OperStateCondList
    cout << "Reading OSC file: " << FileAnDate(mOSCfile) << endl;
    mOSC.readConfig(mOSCfile);

    //----------------------------------  Read in the monitor configuration
    syntax = readConfig(mConfig.c_str());

    //----------------------------------  Invent an html file name
    if (mHtFile.empty()) {
        const char* d = getenv("DMTHTMLOUT");
	if (d) mHtFile = string(d) + "/";
	mHtFile += "Segment_List.html";
    }

    //----------------------------------  Bail out if there was an error.
    if (syntax) {
        cerr << "Syntax error found in file: " << mConfig 
	     << ". SegGener will terminate." << endl;
	finish();
	return;
    }
    getDacc().setIgnoreMissingChannel(true);
    TrigClient::setDebug(Debug());
}

//======================================  Read the configuration file
bool
SegGener::readConfig(const char* cfile) {
    static int depth = 10;

    //----------------------------------  Open the configuration file.
    ParseLine pl(cfile);
    if (!pl.isOpen()) {
        cerr << "Unable to open configuration file: " << cfile << endl;
	return true;
    }

    //----------------------------------  Record the configuration
    cout << "Reading SegGener configuration file: " << FileAnDate(cfile)
	 << endl;

    //----------------------------------  Read in the configuration.
    bool syntax = false;
    while (pl.getLine() >= 0) {
	int nArg = pl.getCount();
        if (nArg == 0) continue;

	//------------------------------  Get the groupname check for include
	string grpname(pl[0]);
	if (grpname == "include") {
	    if (nArg < 2) {
		cerr << "Syntax error in include command." << endl;
		syntax = true;
	    }
	    else if (! depth) {
		cerr << "Include nesting too deep. File: " << pl[1] 
		     << " not parsed." << endl;
		syntax = true;
	    }
	    else {
		depth--;
		syntax = readConfig(pl[1]);
		depth++;
	    }
	    continue;
	}

	//------------------------------  Get the ifo
	string ifo;
	string::size_type cinx = grpname.find(":");
	if (cinx != string::npos) {
	    ifo = grpname.substr(0, cinx);
	    grpname.erase(0,cinx+1);
	}

	//------------------------------  Build the SegCond
	SegCond sc;
	int iArg = 3;
	if (nArg < 3) {
	    cerr << "Error: Insufficient arguments on line." << endl;
	    syntax = true;
	    continue;
	} else if (nArg == 3 || *(pl[3]) == '-') {
	    sc = SegCond(grpname, pl.getInt(1), pl[2]);
	    if (!OSCexists(pl[2])) {
	        cerr << "On condition (" << pl[2] << ") for segment group "
		     << pl[0] << " doesn't exist or isn't valid." << endl;
		syntax = true;
		continue;
	    } else if (Debug()) {
	        cout << "Defined segment: " << grpname << " version: "
		     << pl.getInt(1) << " condition: " << pl[2] << endl;
	    }
	} else {
	    if (!OSCexists(pl[2])) {
	        cerr << "Start condition (" << pl[2] << ") for segment group "
		     << grpname << " doesn't exist or isn't valid." << endl;
		syntax = true;
		continue;
	    } else if (!OSCexists(pl[3])) {
	        cerr << "Stop condition (" << pl[3] << ") for segment group "
		     << grpname << " doesn't exist or isn't valid." << endl;
		syntax = true;
		continue;
	    } else if (Debug()) {
	        cout << "Defined segment: " << pl[0] << " version: "
		     << pl.getInt(1) << " start condition: " << pl[2] 
		     << " stop condition: " << pl[3] << endl;
	    }
	    sc = SegCond(grpname, pl.getInt(1), pl[2], pl[3]);
	    iArg = 4;
	}

	//------------------------------  Parse modifiers
	for (int i=iArg ; i<nArg ; i++) {
	    string argi(pl[i]);
	    if (argi == "-comment") {
	        sc.setComment(pl[++i]);
	    } else if (argi == "-file") {
	        sc.setSegFile(pl[++i]);
	    } else if (argi == "-ifo") {
	        ifo = pl[++i];
	    } else if (argi == "-limit") {
	        sc.setLimit(pl.getDouble(++i));
	    } else {
	        cerr << "Unrecognized argument in configuration file: " 
		     << argi << endl;
		syntax = true;
		continue;
	    }
	}
	if (ifo.empty()) {
	    if (sc.refStartCond()[2] == ':') {
		ifo = sc.refStartCond().substr(0,2);
	    } else {
		cerr << "No ifo specified for group: " << grpname << endl;
		syntax = true;
		continue;
	    }
	}
	sc.setIfo(ifo);
	mList.push_back(sc);
    }
    return syntax;
}

//======================================  SegGener monitor termination.
SegGener::~SegGener(void) {

    //----------------------------------  Write out any pending segments.
    for (list_iter i=mList.begin() ; i != mList.end() ; i++) {
	if (i->flush()) {
	    trig::Segment s = i->makeSegment();
	    if (mRecord) {
	        sendSegment(s);
	    } else {
		logSegment(s, cout);
	    }
	    i->reset();
	}
    }

    //----------------------------------  Write a final status report
    writeTable();

    //----------------------------------  Clear the segment definitions.
    mList.clear();
}

//======================================  Process a stride, generate segments
void 
SegGener::ProcessData(void) {
    //----------------------------------  Test that data are in time span
    Time     t  = getDacc().getFillTime();
    if (t < mSpan0) {
	return;
    } else if (mSpan1 != Time(0) && t >= mSpan1) {
	finish();
	return;
    }

    //----------------------------------  Do the run accounting
    if (!mFirst) mFirst = t;
    mLast = getDacc().getCurrentTime();
    Interval dT = mLast - t;

    //----------------------------------  Loop over segment types
    bool lostMsg = false;
    for (list_iter i=mList.begin() ; i != mList.end() ; i++) {

	//------------------------------  Flush segment after lost data.
	Time tLast = i->getCurrentTime();
	if (tLast != Time(0) && tLast != t) {
	    if (i->flush()) {
		trig::Segment s = i->makeSegment();
		if (mRecord) {
		    sendSegment(s);
		} else {
		    logSegment(s, cout);
		}
		i->writeFile(s);
	    }
	    i->reset();
	    if (!lostMsg) {
		cout << "Data lost, GPS " << tLast.getS() << "-" << t.getS() 
		     << endl;
		lostMsg = true;
	    }
	}

	//------------------------------  Test whether segment can be defined
	if (! i->valid(mOSC)) {
	    if (Debug()) {
		cout << "Input data not valid for segment: " << i->getGroup() 
		     << endl;
	    }
	    if (i->flush()) {
		trig::Segment s = i->makeSegment();
		if (mRecord) {
		    sendSegment(s);
		} else {
		    logSegment(s, cout);
		}
		i->writeFile(s);
	    }
	    i->reset();
	    continue;
	}

	//------------------------------  Evaluate the segment
        if (i->evaluate(mOSC, t, dT)) {
	    if (Debug()) {
	        if (i->testInSegment()) cout << "In " << i->getGroup() 
					     << " segment at: " << t << endl;
		else                    cout << "Out of " << i->getGroup() 
					     << " segment at: " << t << endl;
	    }
	    trig::Segment s = i->makeSegment();
	    if (mRecord) {
	        sendSegment(s);
	    } else {
		logSegment(s, cout);
	    }
	    i->writeFile(s);
	}
    }
    if (t.getS() % 60 == 13) writeTable();
}

//======================================  Build an html page
void
SegGener::writeTable(void) const {

    //----------------------------------  Define the document.
    document doc("Segment table");
    doc.setBackgroundColor(color("white"));

    //----------------------------------  Add a 2-column parameter table
    Table2C* pTab = new Table2C("SegGener Parameters", "", "");
    const char* dFmt = "%Y-%02m-%02d %H:%N:%S (GPS %s)";
    pTab->addRow("Report generated",     Now(),  dFmt);
    pTab->addRow("First data processed", mFirst, dFmt);
    pTab->addRow("Last data processed",  mLast,  dFmt);
    pTab->addRow("Segment Configuration file [date]", FileAnDate(mConfig));
    pTab->addRow("OSC condition file [date]", FileAnDate(mOSCfile));
    doc.addObject(pTab);

    table* tab = new table("Segment Types");
    tab->addColumn("Segment Group");
    tab->addColumn("In Segment");
    tab->addColumn("Start Condition");
    tab->addColumn("Stop Condition");
    tab->addColumn("Live (%)");
    tab->addColumn("Segments Generated");
    tab->addColumn("Limit (s)");
    tab->setBorder(true);
    doc.addObject(tab);

    //----------------------------------  Fill the table.
    double denom = double(mLast - mFirst);
    if (denom <= 0) denom = 1.0;
    for (const_iter i=mList.begin() ; i != mList.end() ; i++) {
        int j = tab->addRow();
	tab->insertData(j, 0, i->getGroup());
	if (!i->testValid())         tab->insertData(j, 1, "---");
	else if (i->testInSegment()) tab->insertData(j, 1, "Yes");
	else                         tab->insertData(j, 1, "No");
	tab->insertData(j, 2, i->getStartCond());
	tab->insertData(j, 3, i->getStopCond());
	tab->insertData(j, 4, 100*double(i->getLiveTime())/denom);
	tab->insertData(j, 5, long(i->getNSegment()));
	tab->insertData(j, 6, double(i->getLimit()));
    }

    //----------------------------------  Write the html page.
    ofstream s(mHtFile.c_str(), ios::out);
    writer w(s);
    doc.write(w);
    s.close();
}
