/* -*- mode: c++; c-basic-offset: 3; -*- */
//
//    Rayleigh Monitor, version 6.2 
//      author: Patrick J. Sutton
//        date: Oct 27, 2003  
//
//    Calculates Rayleigh statistic and averaged power spectrum for data 
//    from one or more channels.  Output consists of time-frequency plots  
//    in a created window, one-dimensional FSpectrum data to the dmtviewer, 
//    and updates and error messages to a web file.  
//
////////////////////////////////////////////////////////////////////////////


//====================================== Definitions and imported headers

//---------- Defined constants:
#define PLOT_TIMESTEPS 50     // Number of time strides to plot in output

//---------- Include monitor-specific headers
#include "RayleighMonitor.hh"
#include "THTFD.hh" 
#include "RayleighGUI.hh"
#include "RayleighConfig.hh"
#include "TSystem.h"

//---------- Include standard headers
#include <cstdlib>
#include <string>
#include <iostream>
#include <iomanip>
#include <fstream> 

//---------- Include LIGO-specific headers
#include "TSeries.hh"
#include "Hanning.hh"
#include "GaussNoise.hh"  
#include "Dacc.hh"
#include "DVecType.hh"
#include "fSeries/DFT.hh"
#include "html/writer.hh"
#include "html/Attrib.hh"
#include "html/align.hh"
#include "html/color.hh"
#include "html/document.hh"
#include "html/font.hh"
#include "html/hline.hh"
#include "html/image.hh"
#include "html/label.hh"
#include "html/link.hh"
#include "html/size.hh"
#include "html/style.hh"
#include "html/table.hh"
#include "html/text.hh"

//---------- Include headers for basic ROOT graphics
#include "TROOT.h"
#include "TApplication.h" 
#include "TCanvas.h"
#include "TGraph.h"
#include "TStyle.h"
#include "TF1.h" 
#include "TText.h" 
#include "TPaveText.h" 

using namespace std;
using containers::PSD;

//====================================== Usage information: 
const char *USAGE_INFO =
   "\n"
   "RayleighMonitor:  Gaussianity Monitor development version 6.2 (2003/10/27).\n"
   "         Author:  Patrick J. Sutton\n"
   "          email:  psutton@gravity.phys.psu.edu\n"
   "       homepage:  http://gravity.phys.psu.edu/~psutton\n"
   "\n"
   "Usage:\n"
   "\n"
   "  RayleighMonitor <optional arguments> \n"
   "\n"
   "Options (in any order):\n"
   "\n"
   "  -batch           Run in batch mode.  No GUI front-end and plots are \n"
   "                   non-interactive.  A properly formatted configuration \n"
   "                   file named RMconfig.txt must already exist in the \n"
   "                   current directory. \n"
   "  -canvas_height   Specify height of each plot in pixels.\n"
   "  -canvas_width    Specify width of each plot in pixels.\n"
   "  -fast            ON BY DEFAULT; disable with -slow option (which is not\n"
   "                   nearly as dumb an option as you might think). \n"
   "                   Use special code for fast-scrolling plot\n"
   "                   updates.  This option is useful when two-dimensional \n"
   "                   graphical output is slow, such as when running \n"
   "                   RayleighMonitor remotely or at the LIGO sites. \n"
   "                   RayleighMonitor remotely or at the LIGO sites. \n"
   "                   WARNING: Only stable when used with -batch option, \n"
   "                   in which case the plots are non-interactive.  Also, \n"
   "                   can plot one channel only, and -record, -skip \n"
   "                   options not available in this mode.  Recommend also \n"
   "                   using -fixedrange option in this mode. \n"
   "  -fixedrange      ON BY DEFAULT; disable with the -nofixedrange option.\n"
   "                   Fix the range of the Rayleigh and PSD plots instead \n"
   "                   of allowing the ranges to 'float' to follow the data. \n"
   "                   Rayleighgram ranges are fixed to [0.3,3.0], while \n"
   "                   spectrogram ranges are fixed to a factor of 3 below and \n"
   "                   above the minimum \n"
   "                   and maximum values in the first PSD calculated. \n"
   "                   Recommended for -fast mode, in which the monitor is \n"
   "                   unable to update the range scale. \n"
   "  -greyscale       Use greyscale rather than color for plots. \n"
   "  -h, --help       Print usage information and exit. \n"
   "  -local           Local mode:  All output files are placed in the \n"
   "                   the current directory.  By default, all output is \n"
   "                   diverted to the directory specified by the environment \n"
   "                   variable $DMTHTMLOUT (if set). \n"
   "  -logf            Set logscale in frequency for plots. \n"
   "  -nofixedrange    Range of plots floats with the range of the data displayed.\n"
   "                   Opposite of the -fixedrange option.\n"
   "  -record <label>  Save copies of the time-frequency plots covering the \n"
   "                   entire run of the program.  Files are named \n"
   "                   <label>_<GPS_End_Time>.gif.  Not available in -fast mode.\n"
   "  -screen          Divert update and error messages to \n"
   "                   the screen instead of to the standard files \n"
   "                   RayleighMonitor_Log.html, RayleighMonitor_Errors.html. \n"
   "                   Useful for debugging the code. \n"
   "  -seek <time>     Do not start analyzing data until the specified GPS <time>\n"
   "                   is reached in the data.  May be useful for offline running.\n"
   "  -simulation      Replace the data in each channel with simulated \n"
   "                   Gaussian noise of unit amplitude. \n"
   "  -skip            Update time-frequency plots only every tenth timestep \n"
   "                   instead of every timestep; may improve speed.  Not \n"
   "                   available in -fast mode. \n"
   "  -slow            Use plotting method which allows for plotting of arbitrary\n" 
   "                   numbers of channels with interactive plots.  Very nice if \n" 
   "                   you have speedy graphics hardware; terribly slow otherwise. \n" 
   "                   Opposite of the -fast option.\n"
   "  -version         Print usage information and exit. \n"
   "  --version        Print usage information and exit. \n"
   "  -whiten <N> <scale>  Approximate whitening by dividing PSD by average of \n"
   "                   the first <N> PSDs measured.  Reset scale of PSD plot to\n"
   "                   [1/<scale>,<scale>].  Only works with -fast option.  Note \n"
   "                   that first <N> PSD plots will not display properly.\n"
   "\n";



//====================================== Text for html output:
const char* const kHtmlTitle = "RayleighMonitor Status";
const char* const kHtmlAuthor = "Patrick J. Sutton";
const char* const kHtmlEmail = "mailto:psutton@gravity.phys.psu.edu";
const char* const kHtmlWeb = "http://gravity.phys.psu.edu/~psutton";
//const char* const kHtmlDescription = " ";
const char* const kHtmlLIGO =
   "Laser Interferometer Gravitational-wave Observatory ";
const char* const kHtmlLIGOLink = "http://www.ligo.caltech.edu";
//const char* const kHtmlFilename = "RayleighMonitor_Summary.html";



#ifndef __CINT__

//====================================== Generate the main routine.
// EXECDAT(RayleighMonitor)
// Note: If want to use EXEROOT then must "hardwire" input file name 
// into code, as TApplication.h wipes out any command-line arguments
// it recognizes.  Therefore -if option disabled; monitor reads file 
// named "RMconfig.txt" written by GUI.

EXEROOT(RayleighMonitor)

#endif               // !def(__CINT__)

//======================================  Construct Rayleigh monitor instance.
RayleighMonitor::RayleighMonitor(int argc, const char *argv[]) 
: DatEnv(argc, argv), MonServer(MonitorName), batch(false), detrend(true),
   errors(false), fast_update(true), fatal_error(false), fixedrange(true), 
   greyscale(false), initial(false), local(false), logf(false), 
   missing_data(false), record(false), seek(false), simulated_data(false), 
   slow_update(false), stderr_diverted(true), stdout_diverted(true),  
   whiten(false), window(true), whiten_scale(10.0), canvas_height(375),
   canvas_width(550), MaxNStep(0), NStride(0), Nwhiten(0)
{
   //---------------------------------- Assign default file names 
   sprintf(input_param_file_name,"RMconfig.txt");

   //---------------------------------- Check command-line options 
   for (int i=1 ; i<argc ; i++) {
      string argi = argv[i];
      if (argi == "-batch") {
	 cout << "RayleighMonitor: batch option: "
	      << "No GUI front-end, plots non-interactive.\n";
	 batch = true; 
      } else if (argi == "-canvas_height") {
	 canvas_height = (int) strtod(argv[++i], 0);
      } else if (argi == "-canvas_width") {
	 canvas_width = (int) strtod(argv[++i], 0);
      } else if (argi == "-fast") {
	 fast_update = true;
      } else if (argi == "-fixedrange") {
	 fixedrange = true;
      } else if (argi == "-greyscale") {
	 greyscale = true;
      } else if (argi == "-h" || argi == "--help") {
	 fatal_error = true;
      } else if(!strcmp("-local", argv[i])) {
	 local = true;
      } else if(!strcmp("-logf", argv[i])) {
	 logf = true;
      } else if (!strcmp("-nofixedrange", argv[i])) {
	 fixedrange = false;
      } else if (!strcmp("-record", argv[i])) {
	 if ( (i+1<argc) && (argv[i+1][0] != '-') ) {
	    record = true;
	    cout << "RayleighMonitor: record option: "
		 << "Save .gif copies of all plots." << endl;
	    record_file = ((string) argv[++i]);
	 } else {
	    fatal_error = true;
	 }
      } else if (argi == "-screen") {
	 cout << "RayleighMonitor: screen option: "
	      << "Update and error messages will be dumped to screen." << endl;
	 stdout_diverted = false;
	 stderr_diverted = false;
	 //:KLUDGE:SEEK
      } else if (argi == "-seek") {
	 seektime = Time(strtod(argv[++i], 0));
	 //seektime = 714829000;
	 seek = true;
	 //:END:KLUDGE:SEEK
      } else if (!strcmp("-simulation", argv[i])) {
	 cout << "RayleighMonitor: simulation option: " 
	      << "Using simulated Gaussian noise." << endl;
	 simulated_data = true; 
      } else if (!strcmp("-skip", argv[i])) {
	 slow_update = true;
      } else if (isDatEnvArg(argv[i])) {
	 i++;
      } else if (!strcmp("-slow", argv[i])) {
	 fast_update = false;
      } else if (!strcmp("-whiten", argv[i])) {
	 Nwhiten = (int) strtod(argv[++i], 0);
	 whiten_scale = (float) strtod(argv[++i], 0);
	 whiten = true;
      } else if (!strcmp("-version",argv[i]) || !strcmp("--version",argv[i])) {
	 fatal_error = true;
      } else {
	 cout << "RayleighMonitor: unrecognized option: " << argv[i] << "\n";
	 //      fatal_error = true;
	 cout << USAGE_INFO << endl;
      } 
   }
   if (fatal_error) { 
      cout << USAGE_INFO << endl;
      batch = true;
      stdout_diverted = false;
      stderr_diverted = false;
      finish();
   }

   //-------- Check to see if should dump html
   //         files to $DMTHTMLOUT or to local directory.
   if (!local) {
      const char* htmldir = getenv("DMTHTMLOUT");
      if (htmldir) {
	 summary_file = string(htmldir) + "/";
	 error_file = string(htmldir) + "/";
	 log_file = string(htmldir) + "/";
	 gif_file = string(htmldir) + "/";
	 record_file = string(htmldir) + "/" + record_file;
      }
   }
   summary_file += "RayleighMonitor_Summary.html";
   error_file += "RayleighMonitor_Errors.html";
   log_file += "RayleighMonitor_Log.html";
   gif_file += "RayleighMonitor.gif";

   //---------------------------------- Divert updates and error messages 
   //                                   to special files if desired (default).
   if (stdout_diverted && !freopen(log_file.c_str(), "w", stdout)) {
      cerr << "Unable to reopen stdout!" << endl;
   }
   if (stderr_diverted && !freopen(error_file.c_str(), "w", stderr)) {
      cerr << "Unable to reopen stderr!" << endl;
   }

   //---------------------------------- Launch GUI 
   // GUI activated by gApplication->Run(kTRUE);
   // Program stops here and execution passes to GUI until 
   // gApplication->Terminate(0) signal received (or ^C 
   // from keyboard), whereupon execution returns here.
   // (If use Run(kFALSE) then program will exit when 
   // Terminate(0) called).
   // :TODO: Invoking window causes core dump on exit.  Fix this.
   if (!batch) {
      mainWindow = new MainDialog(gClient->GetRoot(),400,220);
      gApplication->Run(kTRUE);
      if (mainWindow->IsExitRequested()) { 
	 fatal_error = true;
	 finish();
      }
   }

   //---------------------------------- Read run parameters from input file 
   //                                   (Stride, N, MaxStride, 
   //                                   number_of_channels, channel_names[]) 
   //----- Check for configuration file before trying to open:
   ifstream test(input_param_file_name);
   if (!test) {     //----- file not found
      test.close();
      fatal_error = true;
      cerr << "Run parameter file: " << input_param_file_name << " not found."
	   << endl;
      finish(); 
   } else {         //----- file found
      test.close(); 
      parameters = new RayleighConfig( input_param_file_name );
      f_lower = new double[parameters->number_of_channels];
      f_upper = new double[parameters->number_of_channels];
   }

   //---------------------------------- Verify and output parameters 
   if (!fatal_error) { 
      //---------- Verify that parameters are valid
      if (!parameters->CheckParameters()) {
	 fatal_error = true;
	 cerr << "CheckParameters failed" << endl;
	 finish();
      }

      //---------- Output parameters to log file via cout
      cout << "Run Parameters:  <br>" << endl;
      cout <<    "Stride = " << parameters->Stride << "  <br>\n"; 
      cout <<         "N = " << parameters->N << "  <br>\n"; 
      cout << "MaxStride = " << parameters->MaxStride << "  <br>\n";
      cout << "number_of_channels = " << parameters->number_of_channels;
      cout << "  <br>" << endl;
      cout << "Channels, Sampling Rates, and Frequency Ranges requested:  <br>"
	   << endl;
      for (int j=0; j<parameters->number_of_channels; ++j) {
	 f_lower[j] = parameters->f_lower[j];
	 f_upper[j] = parameters->f_upper[j];
	 cout << parameters->channel_names[j] << " (";
	 cout << parameters->channel_rates[j] << ")  [" << f_lower[j];
	 cout << "," << f_upper[j] << "] Hz   <br>" << endl;
      }        
   }
    
   //---------------------------------- If everything is okay then finish 
   //                                   setup.
   if (!fatal_error) { 
      //---------------------------------- Allocate memory for timeseries array
      rayleigh = new PSD [ parameters->number_of_channels ];
      psd = new PSD [ parameters->number_of_channels ];
      //psd_ref = new PSD [ parameters->number_of_channels ];
      psd_ref = new PSD [ 1 ];
   
      //---------------------------------- Allocate memory for sampling rates
      NStep = new size_t [ parameters->number_of_channels ];

      //---------------------------------- Specify channels to scan
      for(int j=0; j < (parameters->number_of_channels); ++j) {
	 int decimate = 1;
	 double f_Nyq = (parameters->channel_rates[j])/2;
	 while ( f_upper[j] <= f_Nyq/2.0 ) {
	    decimate *= 2; 
	    f_Nyq /= 2.0;
	 }
	 getDacc().addChannel( parameters->channel_names[j] , decimate );
      }

      //---------------------------------- Set the time stride.
      getDacc().setStride(Interval(parameters->Stride*double(parameters->N)));

      //:KLUDGE:SEEK
      //---------------------------------- Skip to desired time.
      if (seek) {
	 cout << "Seeking GPS time " << seektime.totalS() 
	      << ".  Successful if zero: ";
	 cout << getDacc().seek(seektime) << endl;
      }
      //:END:KLUDGE:SEEK

      //---------------------------------- Ready window if required.
      if (window) win = new Hanning();

      //---------------------------------- Connect to MonServer
      current_channel_name = new char[100];
      for(int j=0; j < (parameters->number_of_channels); ++j) {
	 strcpy( current_channel_name, parameters->channel_names[j] );
	 serveData( strcat(current_channel_name,"_ray"), (rayleigh+j) ); 
	 strcpy( current_channel_name, parameters->channel_names[j] );
	 //serveData( strcat(current_channel_name,"_psd"), (psd+j) ); 
	 //serveData( strcat(current_channel_name,"_psd_ref"), (psd_ref+j) ); 
      }
      serveData( strcat(current_channel_name,"_psd"), (psd) ); 
      serveData( strcat(current_channel_name,"_psd_ref"), (psd_ref) ); 

      //---------------------------------- Declare histograms, canvas for plots
      RAYhist = new THTFD*[parameters->number_of_channels];
      PSDhist = new THTFD*[parameters->number_of_channels];

      //---------- Initialize canvas(es). 
      //           Default: gROOT->SetStyle("Default"); other options are 
      //           "Bold" and "Video", both give bolder outlines and titles. 
      //----- Canvas for TFPainter fast plots:
      if (fast_update) {
	 Canvas2 = new TCanvas( "Canvas2", "Fast RayleighMonitor", 
				canvas_width, canvas_height );
	 Canvas3 = new TCanvas( "Canvas3", "Fast RayleighMonitor", 
				canvas_width, canvas_height );
      } else {
	 //-----------------------------  New canvas with correct width, height
	 Canvas = new TCanvas("Canvas", "Rayleigh Monitor", canvas_width*2,
			      canvas_height*parameters->number_of_channels);
	 Canvas->Divide(2,parameters->number_of_channels);
      }

      //--------------------------------   Sets color palette for plots.
      if (greyscale) {
	 gStyle->SetPalette(8,0);   // greyscale 
      } else {
	 gStyle->SetPalette(1,0);   // nice rainbow spectrum 
      }
	
      //--------------------------------  Set normalization for Rayleigh
      //                                  statistic using approximations: 
      expected_raymean = 1.0-0.6*pow(parameters->N,-0.9);
      expected_rayvar = pow(10.0,-0.35-0.2125*log10(double(parameters->N))
			    -0.0625*pow(log10(double(parameters->N)),2.0) 
			    )/expected_raymean;   //normalized for expected mean

   }  //--------- if(!fatal_error)

   //---------------------------------- Report            
   cout << "Constructor finished.   <br>" << endl;
   fflush(stdout);
   fflush(stderr);

}

//====================================== Destroy Rayleigh object.
RayleighMonitor::~RayleighMonitor() 
{
   //---------------------------------- Free any allocated memory 

   //---------- Assigned in constructor 
   if (!batch) delete mainWindow;
   //----- Check for configuration file before deleting these:
   ifstream test(input_param_file_name);
   if (!test) {     //----- file not found
      test.close();  
   } else {         //----- file found
      test.close(); 
      delete parameters;
      delete [] f_lower;
      delete [] f_upper;
   }
   if (!fatal_error) { 
      delete [] rayleigh; 
      delete [] psd;
      //delete [] psd_ref;
      delete [] psd_ref;
      delete [] NStep;
      delete win;
      delete [] current_channel_name;
      delete [] RAYhist;
      delete [] PSDhist;
      if (fast_update) {
	 delete Canvas2;
	 delete Canvas3;
      } else {
	 delete Canvas;
      }
   }

   //-----------------------------------  Assigned in other methods
   if (simulated_data) delete noise;

   cout << "Rayleigh Monitor is finished after " << NStride;
   cout << " ProcessData() calls.   <br>" << endl;

   //---------------------------------- Close update file and go home
   if (stdout_diverted) fclose(stdout);
   if (stderr_diverted) fclose(stderr);
}



//====================================== Frame processing function.
void
RayleighMonitor::ProcessData(void) {
   int     nAvg = parameters->N;

   //---------- Begin loop over channels "j".  
   for( int j=0; j < parameters->number_of_channels; ++j) {   
      DVectD mean, var;

      //--------------------------------  Fetch pointer to data.
      TSeries* ts = getDacc().refData(parameters->channel_names[j]);

      //--------------------------------  Check for time gaps. 
      //           Verified that this works for N*Stride != integer.
      if (!j) {
	 Time starttime = ts->getStartTime();
	 if (starttime != prev_endtime && NStride != 0 ) { 
	    cerr << "WARNING:  Missing data between GPS times ("
		 << fixed << setprecision(3) << prev_endtime.totalS() 
		 << ", " << starttime.totalS() <<")." << endl;
	    errors = true;
	    missing_data = true;
	 }
	 prev_endtime = ts->getEndTime();
      }

      //---------- Optional: Get simulated Gaussian  
      //           noise with the same number of samples
      //           and timestep as the original data.  
      if (simulated_data) { 
	 ts = SimulateData(ts->getNSample(),ts->getTStep());
      } 

      //----------------------------------  Split data into N segments "i".
      double f0=0, dF=1;
      for (int i = 0; i < nAvg; i++) {
	 Interval dT = parameters->Stride;
	 Time     t0 = ts->getStartTime() + dT * double(i);
	 TSeries ts_temp = ts->extract(t0, dT);
         ts_temp.Convert(DVector::t_double);

	 //-----------------------------  Detrend data if desired
	 if (detrend) ts_temp -= ts_temp.getAverage(); 

	 //-----------------------------  Window data if desired
	 if (window) ts_temp = win->apply(ts_temp); 

	 //-----------------------------  Construct frequency spectrum over
	 //                               desired frequency range for each i.
	 //                               Check that desired frequency range
	 //                               is valid and extract smallest range
	 //                               f_min, f_max containing desired  
	 //                               frequencies. [Consult notes on how 
	 //                               extract() selects frequency range. 
	 //                               Also, extract() method cannot get
	 //                               Nyquist frequency, so drop that
	 //                               point.]
	 PSD raw_inspec = PSD(containers::DFT(ts_temp));
	 dF = raw_inspec.getFStep();
	 if (!NStride && !i) {
	    double min_f_lower = raw_inspec.getLowFreq();
	    double max_f_upper = raw_inspec.getHighFreq() - dF;
	    if (f_lower[j] < min_f_lower || f_lower[j] >= max_f_upper){
	       f_lower[j] = min_f_lower;
	       cout << parameters->channel_names[j];
	       cout << "  Reset minimum bin frequency to lowest allowed ";
	       cout << "value of " << f_lower[j] << "Hz  <br>" << endl;
	    }
	    if (f_upper[j] <= min_f_lower || f_upper[j] > max_f_upper){
	       f_upper[j] = max_f_upper;
	       cout << parameters->channel_names[j];
	       cout << "  Reset maximum bin frequency to highest allowed ";
	       cout << "value of " << f_upper[j] << "Hz  <br>" << endl;
	    }
	 }

	 f0 = f_lower[j];
	 double bw = f_upper[j] - f0;
	 PSD inspec = raw_inspec.extract_psd(f0, bw);
	 if (!NStride && !i) {
	    f_lower[j] = inspec.getLowFreq();
	    f_upper[j] = inspec.getHighFreq();
	    f0 = f_lower[j];
	    bw = f_upper[j] - f0;
	    cout << parameters->channel_names[j];
	    cout << "  Lowest frequency bin centered at " << f_lower[j];
	    cout << "Hz  <br>" << endl;
	    cout << parameters->channel_names[j];
	    cout << "  Highest frequency bin centered at " << f_upper[j];
	    cout << "Hz  <br>" << endl;
	    cout << parameters->channel_names[j];
	    cout << "  Bin width " << dF << "Hz  <br>\n\n";
	    cout << "----------------------------------- Avg. over freq.: <br>\n";
	    cout << "GPS start time ----- Channel ------ Rayl. -- PSD --- <br>\n";
	 }

	 //---------------------------------- Initialize data storage arrays.  
	 //                                   (Do first time reading data from 
	 //                                   each channel.  If necessary, 
	 //                                   storage arrays resized to hold   
	 //                                   largest PSD to be handled).
	 if (!NStride && !i && ((NStep[j]=inspec.getNStep())>MaxNStep)) {
	    MaxNStep = NStep[j];
	    initial = true;                 
	 }

	 //---------------------------------- Error check
	 if (inspec.getNStep() != NStep[j]) {
	    cerr << "ERROR: Stride " << NStride << ", Channel " << j;  
	    cerr << ", Data Segment " << i << "      <br>" << endl;  
	    cerr << "       Number of data points has changed:       <br>";  
	    cerr << endl; 
	    cerr << "       previous = " << NStep[j];
	    cerr << ", current = " << inspec.getNStep();
	    cerr << "          <br>" << endl;  
	    errors = true;
	 }

	 //---------------------------------- Zero arrays and intialize psd, 
	 //                                   rayleigh.  Do once per channel 
	 //                                   per ProcessData() call.
	 if (i==0) {
	    rayleigh[j] = inspec;    // holds final rayleigh data (stdev/mean)
	    psd[j] = inspec;         // holds mean PSD 
	 }

	 //---------------------------------- Copy data into temporary storage 
	 DVectD meani(inspec.refDVect());
	 if (meani.size() != NStep[j]+1){ 
	    cerr << "ERROR: Stride " << NStride << ", Channel " << j;  
	    cerr << ", Data Segment " << i << "      <br>" << endl;  
	    cerr << "       Wrong number of data points received:    <br>";
	    cerr << endl; 
	    cerr << "       expected = " << NStep[j]+1;
	    cerr << ", received = " << meani.size();
	    cerr << "   <br>" << endl; 
	    errors = true;
	 }

	 //:KLUDGE: NUMERICAL RECIPES states that calculating variance using 
	 // moments of data (as we do here) typically results in roundoff 
	 // error problems and/or slower execution.  Unfortunately, this is 
	 // the only way I can think of that doesn't require having all of the 
	 // data in memory at once.   
	 DVectD vari = meani;
	 vari *= meani;
	 if  (!i) {
	    mean = meani;
	    var  = vari;
	 } else {
	    mean += meani;
	    var  += vari;
	 }
      }  // Loop over "i" for reading data from given channel finished 

      //---------------------------------- Calculate mean and standard 
      //                                   deviation of PSD
      DVectD meanSq = mean;
      mean   *= 1.0/double(nAvg);
      meanSq *= mean;
      var    -= meanSq;
      var    *= 1.0/double(nAvg -1);
      int nBin = var.size();

      for (int k = 0; k < nBin; k++) {
	 if(var[k] < 0){
	    if(var[k]/(mean[k]*mean[k]) < -0.001) {
	       cerr << "ERROR: Stride " << NStride << ", Channel " << j;  
	       cerr << "                      <br>" << endl;  
	       cerr << "       Variance large and negative for k = " << k; 
	       cerr << ":  <br>" << endl;  
	       cerr << "       var[k] = " << var[k];
	       cerr << ", var/mean^2 = " << var[k]/(mean[k]*mean[k]);
	       cerr << "   <br>" << endl;
	       errors = true;
	    }
	    var[k] = 0.0;
	 }
	 var[k]  = sqrt( var[k] );
      }
      var /= mean;

      //---------- Rescale Rayleigh statistic based on expected value 
      //           for Gaussian noise, given finite N.
      var /= expected_raymean;
      
      //:TODO:
      // Feature to be added: monitor skew and kurtosis, where:      
      //    skew = 1/n \sum_{i=1}^n (x_i-<x>)^3/\sigma^3
      //    kurt = 1/n \sum_{i=1}^n (x_i-<x>)^4/\sigma^4 - 3

      //---------------------------------- Assign PSD, Rayleigh statistics 
      psd[j].setData(f0, dF, mean);
      //psd[j].setCount(parameters->N); 

      rayleigh[j].setData(f0, dF, var);
      //rayleigh[j].setCount(parameters->N);

      //---------------------------------- Output some results to stdout
      //                                   This should be replaced by html 
      //                                   output.
      int nStep = rayleigh[j].size();
      cout << rayleigh[j].getStartTime();
      cout << "  " << parameters->channel_names[j];
      cout << "  " << rayleigh[j].refDVect().VSum(0, nStep)/double(nStep);
      cout << "  " << psd[j].refDVect().VSum(0, nStep)/double(nStep);
      cout << "  <br>" << endl;

      //---------------------------------- Check for GUI events and process 
      //                                   any requests.
      if (!batch) {
	 gSystem->ProcessEvents();
	 if ( mainWindow->IsExitRequested() )  finish();
      }
   } //----- Loop over channels 'j' finished.
   if (simulated_data) delete ts;

   //---------------------------------- Compute reference PSDs if whitening.
   if ( whiten && !NStride) {
      SetReferenceSpectra();
   } else if (whiten && NStride<Nwhiten) {
      AddToReferenceSpectra();
   }

   //---------------------------------- Output rayleigh data, psd to plots 
   if (fast_update) { 
      if (!NStride) RayleighMonitor::CreateFastPlot();
      else          RayleighMonitor::UpdateFastPlot();
   } else {
      if (!NStride) RayleighMonitor::CreatePlot();
      else          RayleighMonitor::UpdatePlot();
   }
   /*
   //:TIMETEST: Need to get non-integer part of times.
   time_t start_time2 = time(NULL);
   if (!NStride) RayleighMonitor::CreatePlot();
   if (NStride) RayleighMonitor::UpdatePlot();
   time_t end_time2 = time(NULL);
   cout << "Time to Create/Update Plot: " << end_time2 - start_time2 << endl;
   */  
 
   //---------------------------------- Output to html file 
   Summary(rayleigh[0].getStartTime()+(parameters->N)*(parameters->Stride)); 

   //---------------------------------- Check for GUI events and process 
   //                                   any requests.
   if (!batch) {
      gSystem->ProcessEvents();
      if ( mainWindow->IsExitRequested() )  finish();
   }
      
   //---------------------------------- Check to see if done
   if ( ++NStride >= (parameters->MaxStride) )  finish();

   //---------------------------------- Flush any remaining reports 
   fflush(stdout);
   fflush(stderr);
}

//====================================== Construct time-frequency plot of 
//                                       PSD data using TFPainter code.
void
RayleighMonitor::CreateFastPlot(void) 
{
   //---------- Hardwire number of time strides to plot
   int timesteps = PLOT_TIMESTEPS;

   //:KLUDGE:FAST:
   //---------- :KLUDGE:  Fast plotter can only handle one plot per 
   //           canvas, and we only make canvases for one channel.
   //           Therefore only Create/Update Fast plots for j=0 channel.
   //           (This really should get handled when parameters are set.)
   //for (int j=0; j<parameters->number_of_channels; ++j) {
   for (int j=0; j<1; ++j) {
      //:END:KLUDGE:FAST:
      //---------- Gather data about FSpectrum for plot parameters
      double deltat = double(parameters->N) * double(parameters->Stride);
      double flower = rayleigh[j].getLowFreq();
      double fupper = rayleigh[j].getHighFreq();
      double deltaf = rayleigh[j].getFStep();
      int nstep = rayleigh[j].getNStep();

      //---------- Initialize TLTimeFrequency histograms
      string histo_name = "Rayleigh Statistic:  ";
      histo_name += parameters->channel_names[j];
      RAY_TFHist = new THTFD("RAY_TFHist", histo_name.c_str(),
			     timesteps, -timesteps*deltat, 0.0,
			     nstep+1, flower-0.5*deltaf, 
			     fupper+0.5*deltaf
			     );
      histo_name = "Power Spectrum (1/Hz):  ";
      histo_name += parameters->channel_names[j];
      PSD_TFHist = new THTFD("PSD_TFHist", histo_name.c_str(),
			     timesteps, -timesteps*deltat, 0.0,
			     nstep+1, flower-0.5*deltaf, 
			     fupper+0.5*deltaf
			     );

      //---------- Fill histograms with data
      //----- Copy psd and rayleigh data to storage arrays.
      double RAYdata[nstep+1];
      double PSDdata[nstep+1];
      rayleigh[j].getData(size_t(nstep+1),RAYdata);
      psd[j].getData(size_t(nstep+1),PSDdata);

      //:KLUDGE:FAST
      //------ Get minimum of PSD data; use to fill out empty space on plot.
      double PSD_min = PSDdata[0];
      for (int i=1; i<nstep+1; ++i) {
	 if (PSDdata[i] < PSD_min) PSD_min = PSDdata[i];
      }

      //---------- Fill first line of histograms with data, rest with filler.
      int newbin;
      for (int m=1; m<timesteps; ++m) {  //-> x-dir: move data points -1 step
	 for (int k=1; k<nstep+2; ++k) {  //-> y-dir: do not alter.
	    newbin = (timesteps+2)*k+m;
	    RAY_TFHist->SetBinContent(newbin,1.0);
	    PSD_TFHist->SetBinContent(newbin,PSD_min);
	    //PSD_TFHist->SetBinContent(newbin,1.0);
	 }
      }
      //:END:KLUDGE:FAST
      //for (int k=1; k<nstep+2; ++k) {
      //	 newbin = (timesteps+2)*k+timesteps;
      //	 RAY_TFHist->SetBinContent(newbin,RAYdata[k-1]);
      //	 PSD_TFHist->SetBinContent(newbin,PSDdata[k-1]);
      //}
      RAY_TFHist->ShiftX(-1, RAYdata);
      PSD_TFHist->ShiftX(-1, PSDdata);

      //---------- Customize style of histograms and plots.
      //----- PSD plot:
      Canvas2->cd();
      gPad->SetLogz();
      if (logf) gPad->SetLogy();
      //    gPad->SetGridx();
      //    gPad->SetGridy();
      gPad->SetRightMargin(0.13);     // Default approx 0.1
      gPad->SetLeftMargin(0.13);       
      gPad->SetTicks(1,1);
      //----- Fix range of fast PSD plot, if desired or if whitening.
      //      RECOMMENDED, since unable to dynamically update scale. 
      if (fixedrange && !whiten) {
	 double PSD_min = PSDdata[0];
	 double PSD_max = PSDdata[0];
	 for (int i=1; i<nstep+1; ++i) {
	    if (PSDdata[i] < PSD_min) PSD_min = PSDdata[i];
	    if (PSDdata[i] > PSD_max) PSD_max = PSDdata[i];
	 }
	 PSD_TFHist->SetMinimum(0.3*PSD_min);
	 PSD_TFHist->SetMaximum(3.0*PSD_max);
      } else if (whiten) {
	 PSD_TFHist->SetMinimum(1.0/whiten_scale);
	 PSD_TFHist->SetMaximum(whiten_scale);
      }
      PSD_TFHist->SetXTitle ("Time before present (s)");
      PSD_TFHist->SetYTitle ("Frequency (Hz)");
      //    PSD_TFHist->GetXaxis()->SetTitle(xaxislabel); 
      PSD_TFHist->GetXaxis()->CenterTitle(kTRUE);
      PSD_TFHist->GetXaxis()->SetLabelOffset(-0.002);  
      PSD_TFHist->GetXaxis()->SetLabelFont(100);  
      PSD_TFHist->GetXaxis()->SetTitleFont(100);  
      //    PSD_TFHist->GetYaxis()->SetTitle("Frequency (Hz)");
      PSD_TFHist->GetYaxis()->CenterTitle(kTRUE);
      PSD_TFHist->GetYaxis()->SetTitleOffset(1.50);   //default about 1.0   
      PSD_TFHist->GetYaxis()->SetLabelOffset(0.005);  
      PSD_TFHist->GetYaxis()->SetLabelFont(100);  
      PSD_TFHist->GetZaxis()->SetLabelOffset(0.010);
      PSD_TFHist->GetZaxis()->SetLabelFont(100);  
      PSD_TFHist->SetStats(kFALSE);
      PSD_TFHist->SetOption("COLZ");
      PSD_TFHist->Draw();
      //----- Rayleigh plot:
      //----- Fix range of fast Rayleigh plot, if desired.
      //      RECOMMENDED, since unable to dynamically update scale. 
      if (fixedrange) {
	 RAY_TFHist->SetMinimum(0.3);
	 RAY_TFHist->SetMaximum(3.0);
      }
      Canvas3->cd();
      gPad->SetLogz();
      if (logf) gPad->SetLogy();
      gPad->SetRightMargin(0.13);     // Default approx 0.1
      gPad->SetLeftMargin(0.13);       
      gPad->SetTicks(1,1);
      RAY_TFHist->SetXTitle ("Time before present (s)");
      RAY_TFHist->SetYTitle ("Frequency (Hz)");
      //    RAY_TFHist->GetXaxis()->SetTitle(xaxislabel); 
      RAY_TFHist->GetXaxis()->CenterTitle(kTRUE);
      RAY_TFHist->GetXaxis()->SetLabelOffset(-0.002);  
      RAY_TFHist->GetXaxis()->SetLabelFont(100);  
      RAY_TFHist->GetXaxis()->SetTitleFont(100);  
      //    RAY_TFHist->GetYaxis()->SetTitle("Frequency (Hz)");
      RAY_TFHist->GetYaxis()->CenterTitle(kTRUE);
      RAY_TFHist->GetYaxis()->SetTitleOffset(1.50);   //default about 1.0   
      RAY_TFHist->GetYaxis()->SetLabelOffset(0.005);  
      RAY_TFHist->GetYaxis()->SetLabelFont(100);  
      RAY_TFHist->GetZaxis()->SetLabelOffset(0.010);
      RAY_TFHist->GetZaxis()->SetLabelFont(100);  
      RAY_TFHist->SetStats(kFALSE);
      RAY_TFHist->SetOption("COLZ");
      RAY_TFHist->Draw();
   }

   //---------- Add legend to plot.
   legend = new TPaveText(0.60,0.91,0.99,0.99,"NDC");
   char info[30];
   sprintf(info,"N = %d,  Stride = %f",parameters->N,parameters->Stride);
   legend->AddText(info);
   sprintf(info,"Rayleigh: 1 sigma = %0.2f",expected_rayvar);
   legend->AddText(info);
   legend->Draw();

   //---------- Draw plots.
   Canvas2->Update();
   Canvas3->Update(); 
}



//====================================== Construct time-frequency plot of 
//                                       FSpectrum data.  Initializes histogram 
//                                       and fills in first stride of data. 
//                                       Update with UpdatePlot() method. 
/** Construct time-frequency plots of the averaged PSD
 * and Rayleigh statistic for each channel.   
 * Uses first stride of data only; plots updated with UpdatePlot() method. 
 * @memo Construct time-frequency plot of FSpectrum data. 
 */
void
RayleighMonitor::CreatePlot(void) 
{
   //---------- Hardwire number of time strides to plot
   int timesteps = PLOT_TIMESTEPS;

   for (int j=0; j<parameters->number_of_channels; ++j) {
      //---------- Gather data about FSpectrum for plot parameters
      double deltat = 1.0*(parameters->N)*(parameters->Stride);
      double flower = rayleigh[j].getLowFreq();
      double fupper = rayleigh[j].getHighFreq();
      double deltaf = rayleigh[j].getFStep();
      int nstep = rayleigh[j].getNStep();  

      // :TODO:  Make title font sizes the same.

      //---------- Initialize histograms   
      char prefix_name[100];
      strcpy( prefix_name, "Rayleigh Statistic:  " );
      RAYhist[j] = new THTFD("RAYhist", 
			     strcat(prefix_name,parameters->channel_names[j]), 
			     timesteps, -timesteps*deltat, 0.0, 
			     nstep+1, flower-0.5*deltaf, fupper+0.5*deltaf 
			     );
      strcpy( prefix_name, "Power Spectrum (1/Hz):  " );
      PSDhist[j] = new THTFD("PSDhist", 
			     strcat(prefix_name,parameters->channel_names[j]), 
			     timesteps, -timesteps*deltat, 0.0, 
			     nstep+1, flower-0.5*deltaf, fupper+0.5*deltaf 
			     );

      //---------- Fill histograms with data
      //----- Copy psd and rayleigh data to storage arrays.
      double RAYdata[nstep+1];
      double PSDdata[nstep+1];
      rayleigh[j].getData(size_t(nstep+1),RAYdata); 
      psd[j].getData(size_t(nstep+1),PSDdata); 
      //:KLUDGE:FAST
      //------ Get minimum of PSD data; use to fill out empty space on plot.
      double PSD_min = PSDdata[0];
      for (int i=1; i<nstep+1; ++i) {
	 if (PSDdata[i] < PSD_min) PSD_min = PSDdata[i];
      }
      //---------- Fill first line of histograms with data, rest with ones
      int newbin;
      for (int m=1; m<=timesteps; ++m) {  //-> x-dir: move data points -1 step
	 for (int k=1; k<nstep+2; ++k) {  //-> y-dir: do not alter.
	    newbin = (timesteps+2)*k+m;
	    RAYhist[j]->SetBinContent(newbin,1.0); 
	    PSDhist[j]->SetBinContent(newbin,PSD_min); 
	 }
      }
      //:END:KLUDGE:FAST
      //for (int k=1; k<nstep+2; ++k) {   
      //   newbin = (timesteps+2)*k+timesteps;
      //   RAYhist[j]->SetBinContent(newbin,RAYdata[k-1]); 
      //   PSDhist[j]->SetBinContent(newbin,PSDdata[k-1]);
      //}
      RAYhist[j]->ShiftX(-1, RAYdata, false);
      RAYhist[j]->ShiftX(-1, PSDdata, false);
 
      //---------- Customize style of histograms and plot 
      Canvas->cd(2*j+1);
      gPad->SetLogz();
      if (logf) gPad->SetLogy();
      gPad->SetGridx();
      gPad->SetGridy();
      gPad->SetRightMargin(0.13);     // Default approx 0.1
      gPad->SetLeftMargin(0.13);       
      gPad->SetTicks(1,1);
      //----- Fix range of rayleighgram, if desired.
      if (fixedrange) {
	 RAYhist[j]->SetMinimum(0.3);
	 RAYhist[j]->SetMaximum(3.0);
      }
      //---------- Careful!  For N segments, endtime = starttime[0] + N*Stride 
      char xaxislabel[] = "Seconds before GPS ";
      double endtime = rayleigh[j].getStartTime().totalS();
      endtime += (parameters->N)*(parameters->Stride);
      //prev_endtime = endtime;
      char word[25];
      sprintf(word,"%f",endtime);
      strcat(xaxislabel,word);
      RAYhist[j]->GetXaxis()->SetTitle(xaxislabel); 
      RAYhist[j]->GetXaxis()->CenterTitle(kTRUE);
      RAYhist[j]->GetXaxis()->SetLabelOffset(-0.002);  //default about 0.005
      RAYhist[j]->GetXaxis()->SetLabelFont(100);  
      RAYhist[j]->GetXaxis()->SetTitleFont(100);  
      RAYhist[j]->GetYaxis()->SetTitle("Frequency (Hz)");
      RAYhist[j]->GetYaxis()->CenterTitle(kTRUE);
      RAYhist[j]->GetYaxis()->SetTitleOffset(1.50);   //default about 1.0   
      RAYhist[j]->GetYaxis()->SetLabelOffset(0.005);  
      //    RAYhist[j]->GetYaxis()->SetTitleFont(40);  // Makes label horizontal
      RAYhist[j]->GetYaxis()->SetLabelFont(100);  
      RAYhist[j]->GetZaxis()->SetLabelOffset(0.010);
      RAYhist[j]->GetZaxis()->SetLabelFont(100);  
      //    RAYhist[j]->GetYaxis()->SetLabelSize(0.04);
      RAYhist[j]->SetStats(kFALSE);  // No default statistics seem useful
      RAYhist[j]->SetOption("COLZ"); // Color scale with drawn palette
      RAYhist[j]->Draw();

      Canvas->cd(2*j+2);
      gPad->SetLogz();
      if (logf) gPad->SetLogy();
      gPad->SetGridx();
      gPad->SetGridy();
      gPad->SetRightMargin(0.13);
      gPad->SetLeftMargin(0.13);
      gPad->SetTicks(1,1);
      //----- Fix range of PSD plot, if desired.
      if (fixedrange) {
	 double PSD_min = PSDdata[0];
	 double PSD_max = PSDdata[0];
	 for (int i=1; i<nstep+1; ++i) {
	    if (PSDdata[i] < PSD_min) PSD_min = PSDdata[i];
	    if (PSDdata[i] > PSD_max) PSD_max = PSDdata[i];
	 }
	 PSDhist[j]->SetMinimum(0.3*PSD_min);
	 PSDhist[j]->SetMaximum(3.0*PSD_max);
      }
      PSDhist[j]->GetXaxis()->SetTitle(xaxislabel); 
      PSDhist[j]->GetXaxis()->CenterTitle(kTRUE);
      PSDhist[j]->GetXaxis()->SetLabelOffset(-0.002);  
      PSDhist[j]->GetXaxis()->SetLabelFont(100);  
      PSDhist[j]->GetXaxis()->SetTitleFont(100);  
      PSDhist[j]->GetYaxis()->SetTitle("Frequency (Hz)");
      PSDhist[j]->GetYaxis()->CenterTitle(kTRUE);
      PSDhist[j]->GetYaxis()->SetTitleOffset(1.50);   //default about 1.0   
      PSDhist[j]->GetYaxis()->SetLabelOffset(0.005);  
      PSDhist[j]->GetYaxis()->SetLabelFont(100);  
      PSDhist[j]->GetZaxis()->SetLabelOffset(0.010);
      PSDhist[j]->GetZaxis()->SetLabelFont(100);  
      PSDhist[j]->SetStats(kFALSE);
      PSDhist[j]->SetOption("COLZ");
      PSDhist[j]->Draw();
      Canvas->cd();
   }

   //---------- Add legend to plot.
   legend = new TPaveText(0.60,0.91,0.99,0.99,"NDC");
   char info[30];
   sprintf(info,"N = %d,  Stride = %f",parameters->N,parameters->Stride);
   legend->AddText(info); 
   sprintf(info,"Rayleigh: 1 sigma = %0.2f",expected_rayvar); 
   legend->AddText(info); 
   Canvas->cd(1);
   legend->Draw();
   Canvas->cd();

   //---------- Update plot and save copy to file.
   Canvas->Update();
   Canvas->Print(gif_file.c_str());
   cerr << " <br>" << endl;
}

 
 
//====================================== Update fast time-frequency plots
//                                       using TFPainter code.
void
RayleighMonitor::UpdateFastPlot(void) 
{
   //:KLUDGE:FAST:
   //---------- :KLUDGE:  Fast plotter can only handle one plot per 
   //           canvas, and we only make canvases for one channel.
   //           Therefore only Create/Update Fast plots for j=0 channel.
   //           (This really should get handled when parameters are set.)
   //for (int j=0; j<parameters->number_of_channels; ++j) {
   for (int j=0; j<1; ++j) {
      //:END:KLUDGE:FAST:
      //---------- Gather data about FSpectrum for plot parameters
      int nstep = rayleigh[j].getNStep();

      //---------- Move histogram data one time step and add new data
      double data[nstep+1];
      if (missing_data) {
	 for (int i=0; i<nstep+1; ++i) {
	    data[i] = 0.0;
	 }
	 Canvas2->cd();
	 PSD_TFHist->ShiftX(-1, data);
	 //PSD_TFHist->Update();
	 gPad->Update();
	 Canvas3->cd();
	 RAY_TFHist->ShiftX(-1, data);
	 gPad->Update();
	 //RAY_TFHist->Update();
	 missing_data = false;
      }
      //----- PSD plot:
      psd[j].getData(size_t(nstep+1),data);
      if (whiten && (NStride >= Nwhiten)) {
	 DVectD white_ref = dynamic_cast<DVectD&>(psd_ref[0].refDVect());
	 for (int i=0; i<nstep+1; ++i) {
	    data[i] /= white_ref[i];
	 }
      }
      Canvas2->cd();
      PSD_TFHist->ShiftX(-1, data);
      //----- Rayleigh plot:
      rayleigh[j].getData(size_t(nstep+1),data);
      Canvas3->cd();
      RAY_TFHist->ShiftX(-1, data);

      //----- Update plots:
      //PSD_TFHist->Update();
      //RAY_TFHist->Update();
      Canvas2->Update();
      Canvas3->Update();
      // gPad->SetEditable(kFALSE);  // Has no effect on crashes.
      // Canvas2->SetEditable(kFALSE);  // Has no effect on crashes.
      // cout << "Update: Is canvas editable: " << Canvas2->IsEditable()<< endl;
      // cout << "Update: Is gPad editable: " << gPad->IsEditable() << endl;
      // if (!Canvas2->IsEditable()) 
      //                cout << "Update: Canvas2->IsEditable() false." << endl;
      // if (!gPad->IsEditable()) cout << "Update: gPad->IsEditable() false."
      //                                 << endl;
      //    Canvas2->Update();  //updating canvas causes crash.
      //    Canvas2->Print("tf_test.gif");
   }
}



//====================================== Update time-frequency plot of 
//                                       FSpectrum data created by CreatePlot().
/** Update time-frequency plots of the averaged PSD 
 * and Rayleigh statistic for each channel.   
 * @memo Update time-frequency plot of FSpectrum data. 
 */
void
RayleighMonitor::UpdatePlot(void) 
{
   //---------- Hardwire number of time strides to plot
   int timesteps = PLOT_TIMESTEPS;

   for (int j=0; j<parameters->number_of_channels; ++j) {

      //---------- Update x (time) axis:
      char xaxislabel[] = "Seconds before GPS ";
      double endtime = rayleigh[j].getStartTime().totalS();
      endtime += (parameters->N)*(parameters->Stride);
      char word[25];
      sprintf(word,"%f",endtime);
      strcat(xaxislabel,word);
      RAYhist[j]->GetXaxis()->SetTitle(xaxislabel); 
      PSDhist[j]->GetXaxis()->SetTitle(xaxislabel); 

      //---------- Gather data about FSpectrum for plot parameters
      int nstep = rayleigh[j].getNStep();  

#if 0
      //---------- Move histogram data one time step and add new data
      double data[nstep+1];
      if (missing_data) {
	 for (int i=0; i<nstep+1; ++i) {
	    data[i] = 0.0;
	 }
	 Canvas2->cd();
	 PSD_TFHist->NewF(data);
	 PSD_TFHist->Update();
	 Canvas3->cd();
	 RAY_TFHist->NewF(data);
	 RAY_TFHist->Update();
      }
#endif
      missing_data = false;

      //---------- Move histogram data one time step and add new data
      double RAYdata[nstep+1];
      double PSDdata[nstep+1];
      rayleigh[j].getData(size_t(nstep+1),RAYdata); 
      psd[j].getData(size_t(nstep+1),PSDdata); 
      /*
	if (whiten) { 
        for (int i=0; i<nstep+1; ++i) {
	PSDdata[i] /= PSDdata_ref[i];
        }
	}
      */
      int newbin;
      int oldbin;
      for (int m=1; m<timesteps; ++m) { // time dir: move data points -1 step.
	 for (int k=1; k<nstep+2; ++k) { // freq dir: do not alter.
	    newbin = (timesteps+2)*k+m;
	    oldbin = (timesteps+2)*k+m+1;
	    RAYhist[j]->SetBinContent(newbin,RAYhist[j]->GetBinContent(oldbin));
	    PSDhist[j]->SetBinContent(newbin,PSDhist[j]->GetBinContent(oldbin));
	 }
      }
      //---------- If missing data, shift and add extra column of zero data as warning.
      if (missing_data) {
	 for (int k=1; k<nstep+2; ++k) {   
	    newbin = (timesteps+2)*k+timesteps;
	    RAYhist[j]->SetBinContent(newbin,0.0); 
	    PSDhist[j]->SetBinContent(newbin,0.0); 
	 }
	 for (int m=1; m<timesteps; ++m) { // time dir: move data points -1 step.
	    for (int k=1; k<nstep+2; ++k) { // freq dir: do not alter.
	       newbin = (timesteps+2)*k+m;
	       oldbin = (timesteps+2)*k+m+1;
	       RAYhist[j]->SetBinContent(newbin,RAYhist[j]->GetBinContent(oldbin)); 
	       PSDhist[j]->SetBinContent(newbin,PSDhist[j]->GetBinContent(oldbin)); 
	    }
	 }
	 if (j == parameters->number_of_channels - 1) missing_data = false;
      }
      for (int k=1; k<nstep+2; ++k) {   
	 newbin = (timesteps+2)*k+timesteps;
	 RAYhist[j]->SetBinContent(newbin,RAYdata[k-1]); 
	 PSDhist[j]->SetBinContent(newbin,PSDdata[k-1]); 
      }
   }

   //---------- Update plots if desired
   if ( !slow_update || (!(5*(NStride+1) % timesteps)) ) {
      for (int j=0; j<parameters->number_of_channels; ++j) {
	 //---------- Update histograms
	 Canvas->cd(2*j+1);
	 RAYhist[j]->Draw();
	 Canvas->cd(2*j+2);
	 PSDhist[j]->Draw();
	 Canvas->cd();
      }
     
      //---------- Add legend to plot.  Must be re-drawn every time; the
      //           problem seems to be that the first histogram is written 
      //           over it unless the legend is drawn after the histogram.
      //           Cleaner solution would be to set the legend to be "on top"
      //           of the histogram somehow and only draw it once.  
      legend = new TPaveText(0.60,0.89,0.99,0.99,"NDC");
      char info[30];
      sprintf(info,"N = %d,  Stride = %f",parameters->N,parameters->Stride);
      legend->AddText(info); 
      sprintf(info,"Rayleigh: 1 sigma = %0.2f",expected_rayvar); 
      legend->AddText(info); 
      Canvas->cd(1);
      legend->Draw();
      Canvas->cd();

      //---------- Save copy of latest plot to temporary .gif image file. 
      Canvas->Update();
      Canvas->Print(gif_file.c_str());
      cerr << " <br>" << endl;
   }

   //---------- Save permanent copy of plot once every 'timesteps' strides 
   //           if the -record option is used. 
   if ( record && !((NStride+1) % timesteps) ) {
      char name[25];
      double endtime = rayleigh[0].getStartTime().totalS();
      endtime += (parameters->N)*(parameters->Stride);
      int endtime_int = ((int) endtime);
      sprintf(name,"%s%s%d%s",record_file.c_str(),"_",endtime_int,".gif");
      Canvas->Print(name);
      cerr << " <br>" << endl;
   }
}



/** Add current PSD for first channel to reference spectra. 
 * Used to whiten PSD time-frequency plot. 
 * @memo Add current PSD to reference spectra.     
 */
void
RayleighMonitor::AddToReferenceSpectra(void) {
   PSDdata_ref += psd[0];

   //-----------------------------------  Normalize on the last entry
   if (NStride+1 == Nwhiten) {
      PSDdata_ref *= 1.0/double(Nwhiten);
      psd_ref[0]   = PSDdata_ref;
   }
}


/** Record current PSD for each channel as reference spectra. 
 * Used to whiten PSD time-frequency plots. 
 * @memo Record current PSDs as reference spectra.     
 */
void
RayleighMonitor::SetReferenceSpectra(void)
{
   //---------- Copy PSD to reference FSpectrum 
   //           (reference data will be updated later).
   psd_ref[0] = psd[0];

   //---------- Copy current PSD to reference data array. 
   PSDdata_ref = psd[0];
}



//====================================== Write summary information to a web  
//                                       page. 
/** Write summary information to a web page.
 * @memo Write summary information to a web page.
 * @param t Start time of data for which summary is written. 
 */
void
RayleighMonitor::Summary(const Time& t) 
{
   //---------- Construct html document object.
   html::document doc (kHtmlTitle);
   doc.setRefresh( (parameters->N)*(parameters->Stride) );

   //---------- Title section
   html::block titleblk ("center");
   titleblk.lineBreak();
   html::text title (kHtmlTitle);
   html::font fonttitle ("Helvetica");
   title.setFont (fonttitle);
   title.setColor (html::color (0, 0, 128));
   title.setSize (html::size (+4));
   titleblk.addObject (title);
   titleblk.lineBreak();
   titleblk.addObject (html::text ("&nbsp;"));
   doc.addObject (titleblk);
   doc.addObject (html::hline());


   // Link to documentation disabled; documentation available 
   // through DMT Observer page.
   /*   
   //---------- Look for documentation-type files in directory specified
   //           in $DMTDOC instead of in local directory.
   string file_name;
   const char* htmldir = getenv("DMTDOC");
   if (htmldir) {
   file_name = string(htmldir) + "/";
   }
   file_name += "RayleighMonitor.html";

   //---------- Short description
   html::block firstdescriptionblk ("div");
   html::text intro1 ("The "); 
   firstdescriptionblk.addObject (intro1);
   firstdescriptionblk.addObject (html::link (
   "RayleighMonitor ", file_name.c_str()
   ));
   html::text intro2 (" 
   provides a frequency-dependent measure 
   of the Gaussianity of the data in any given channel(s). 
   It works by monitoring the ratio of the standard deviation 
   in the power to the mean of the power at each frequency   
   (the `Rayleigh statistic'):
   ");
   */
   //---------- Short description
   html::block firstdescriptionblk ("div");
   html::text intro2;
   intro2 << "RayleighMonitor provides a frequency-dependent measure "
	  << "of the Gaussianity of the data in any given channel(s). " 
	  << "It works by monitoring the ratio of the standard deviation "
	  << "in the power to the mean of the power at each frequency "
	  << "(the `Rayleigh statistic'):";
   firstdescriptionblk.addObject (intro2);
   firstdescriptionblk.lineBreak();
   doc.addObject (firstdescriptionblk);
   html::block eqnblk ("center");
   html::image eqn;
   eqn.setSource ("eqn.png");
   eqn.setWidth ("129");
   eqnblk.addObject (eqn);
   eqnblk.lineBreak();
   doc.addObject (eqnblk);
   html::block seconddescriptionblk ("div");
   html::text intro3;
   intro3 << "Ratios significantly above unity indicate that the power of "
	  << "the dominant noise source at that frequency is more variable "
	  << "than for a Gaussian; ratios significantly below unity "
	  << "indicate the dominant noise source at that frequency "
	  << "is coherent. The RayleighMonitor produces time-frequency "
	  << "plots of both this ratio and the averaged power spectrum "
	  << "for each channel.";
   seconddescriptionblk.addObject (intro3);
   seconddescriptionblk.lineBreak();
   seconddescriptionblk.lineBreak();
   doc.addObject (seconddescriptionblk);

   //---------- Author
   doc.addObject (html::link ("Author:",kHtmlEmail));
   doc.addObject (html::text ("  "));
   doc.addObject (html::text (kHtmlAuthor));
   doc.lineBreak();
   doc.addObject (html::link (kHtmlWeb,kHtmlWeb));
   doc.addObject (html::hline());

   //---------- Plots 
   if (!fast_update) { 
      html::block plotblk ("div");
      plotblk.lineBreak();
      html::text plottitle ("Plots");
      plottitle.setFont (fonttitle);
      plottitle.setColor (html::color (0, 128, 0));
      plottitle.setSize (html::size (+2));
      plotblk.addObject (plottitle);
      plotblk.lineBreak();
      plotblk.lineBreak();
      html::text plotcaption;
      plotcaption << "Plots of the Rayleigh statistic (left) and "
		  << "averaged power spectrum (right) are shown below "
		  << "for each channel.  These plots should automatically "
		  << "update when the page is re-loaded. In netscape, click "
		  << "on the image with the right mouse button and select "
		  << "Save Image As ... to save as a .gif file. "
		  << "Time zero in these plots is approximately GPS:";
      char buf[128];
      plotcaption << TimeStr(t, buf, " %s or ");
      plotcaption << TimeStr(t, buf, "%M %d, %Y at %H:%N:%S.");
      plotblk.addObject (plotcaption);
      plotblk.lineBreak();
      plotblk.lineBreak();
      plotblk.addAttr ("align", html::align ("left"));
      html::image plot;
      plot.setSource (gif_file.c_str());
      plot.setWidth ("1400");
      plotblk.addObject (plot);
      doc.addObject (plotblk);
      doc.addObject (html::hline());
   }
    
   //---------- Current Status block
   html::block updateblk ("div");
   updateblk.lineBreak();
   html::text statustitle ("Current Status");
   statustitle.setFont (fonttitle);
   statustitle.setColor (html::color (0, 128, 0));
   statustitle.setSize (html::size (+2));
   updateblk.addObject (statustitle);
   updateblk.lineBreak();

   //---------- Last update time and update frequency
   html::text lasttime ("");
   char buf[128];
   lasttime << TimeStr(t, buf, "%M %d, %Y at %H:%N:%S");
   html::text frequency ( "(Updates every " );
   frequency << (parameters->N)*(parameters->Stride);
   frequency << " seconds.)";
    
   //---------- Current Status info table
   //           Create empty table
   html::table info;
   for (int j=0; j<3; ++j) {
      info.addColumn(" "); 
   }
   for (int j=0; j<parameters->number_of_channels+9; ++j) {
      info.addRow(); 
   }
    
   //---------- Insert first column
   info.insertData( 0, 0, "Last Updated: " );
   info.insertData( 2, 0, "Error Messages: " );
   info.insertData( 3, 0, "More Update Info: " );
   info.insertData( 4, 0, "Run Parameters: " );
   info.insertData( 9, 0, "Channels and Frequency Ranges (Hz): " );

   //---------- Insert update time and error message link
   info.insertData( 0, 1, lasttime );
   info.insertData( 1, 1, frequency );
   if (!errors) {
      html::text errormessage ("None");
      errormessage.setColor (html::color (0, 128, 0));
      info.insertData( 2, 1, errormessage );
   } else {
      //:TODO: Figure out how to put link in nasty red color.
      html::link errormessage ("Click Here", "RayleighMonitor_Errors.html");
      info.insertData( 2, 1, errormessage );
   }
   html::link logmessage ("Click Here", "RayleighMonitor_Log.html");
   info.insertData( 3, 1, logmessage );

   //---------- Insert Run Parameters
   info.insertData( 
		   4, 1, html::text ("Number of Samples Averaged (N): ")  
		    );
   info.insertData( 4, 2, html::text (parameters->N) );
   info.insertData( 
		   5, 1, html::text ("Length of Each Sample (Stride, sec.): ") 
		    );
   info.insertData( 5, 2, html::text (parameters->Stride) );
   info.insertData( 
		   6, 1, html::text ("Current Stride Number: ") 
		    );
   info.insertData( 6, 2, html::text (NStride + 1) );
   info.insertData( 
		   7, 1, html::text ("Maximum Stride Number: ")  
		    );
   info.insertData( 7, 2, html::text (parameters->MaxStride) );
   info.insertData( 
		   8, 1, html::text ("Number of Channels Monitored: ") 
		    );
   info.insertData( 
		   8, 2, html::text (parameters->number_of_channels) 
		    );

   //---------- Insert Channel Names and Freq. Ranges
   for (int j=0; j<parameters->number_of_channels; ++j) {
      info.insertData( 
		      j+9, 1, html::text (parameters->channel_names[j])
		       );
      html::text freq_range("[");
      freq_range << f_lower[j] << "," << f_upper[j] << "]";
      info.insertData( j+9, 2, freq_range );
   }
    
   //---------- Formatting 
   info.refCell(0, 0).setAlign("right");
   info.refCell(2, 0).setAlign("right");
   info.refCell(3, 0).setAlign("right");
   info.refCell(4, 0).setAlign("right");
   info.refCell(9, 0).setAlign("right");
    
   //---------- Attach status table to document
   updateblk.addObject (info);
   updateblk.addAttr ("align", html::align ("left"));
   doc.addObject (updateblk);
   doc.addObject (html::hline());
        
   //---------- Logo
   html::block logoblk ("div");
   logoblk.addAttr ("align", html::align ("right"));
   logoblk.addObject (html::link (kHtmlLIGO, kHtmlLIGOLink));
   html::image logo;
   logo.setSource ("ligo_logo.gif");
   logo.setWidth ("80");
   logoblk.addObject (logo);
   doc.addObject (logoblk);

   //---------- Write web page to file
   //  ofstream out2(kHtmlFilename);
   ofstream out2(summary_file.c_str());
   if (out2) {
      html::writer htmlout (out2);
      htmlout.meta("Pragma","no-cache");
      doc.write (htmlout);
   }
}



//====================================== Construct simulated data for testing 
//                                       the monitor. 
TSeries* 
RayleighMonitor::SimulateData(int nsample, double tstep) 
{
   //---------- Construct Gaussian noise template, use to initialize TSeries. 
   cout << "WARNING: Using simulated Gaussian data. <br>" << endl;
   GaussNoise gauss(10000.0,(parameters->N)*(parameters->Stride),Time(0));
   noise = new TSeries(Time(0), tstep, nsample, gauss); 
   //---------- Add sine wave at specific frequency to noise.
   //*noise += TSeries(Time(0), tstep, nsample, Sine(60.0)); 
   return noise;
}


