#include "wtypes.hh"
#include "param_list.hh"
#include "Time.hh"
#include <string>
#include <iosfwd>

// WSCAN Create spectrograms of significant channels for candidate events
//
// WSCAN Examines gravitational-wave channels, auxiliary interferometer channels,
// and environmental channels around times of interest.  Spectrograms are
// generated for those channels that exhibit statistically significant behavior.
//
// usage: wscan(eventTime, configurationFile, frameCacheFile, 
//              outputDirectory, generateReport, debugLevel);
//
//   eventTime             GPS time of candidate event
//   configurationFile     path name of channel configuration file
//   frameCacheFile        path name of frame file cache file
//   outputDirectory       directory to write results
//   generateReport        generate plots and html report
//   debugLevel            verboseness of debug level output
//
// To allow use as a stand-alone executable, the specified eventTime should be
// a string, not a number.
//
// The configuration file is an ASCII text file describing the parameters for
// each channel to be analyzed.  The entries should have the following syntax.
//
// {
//   channelName:                 "H1:LSC-AS_Q"
//   frameType:                   "RDS_R_L1"
//   sampleFrequency:             4096
//   searchTimeRange:             16
//   searchFrequencyRange:        [64 1024]
//   searchQRange:                [4 64]
//   searchMaximumEnergyLoss:     0.2
//   adjustLimits:                false
//   whiteNoiseFalseRate:         1e-2
//   searchWindowDuration:        0.1
//   plotTimeRanges:              [0.1 1.0 10.0]
//   plotFrequencyRange:          [64 1024]
//   plotNormalizedEnergyRange:   [0 25.5]
//   alwaysPlotFlag:              0
// }
//
// Groups of related channels may optionally be separated by a section title
// specifier with the following form.
//
// [index_entry:section_title]
//
// This will be used to generate a index entry and section title in the 
// resulting web page.
//
// The WCONFIGURE.SH script can be used to automatically generate a reasonable
// configuration file for sample frame files.
//
// If no configuration file is specified, WSCAN looks for the file
// configuration.txt in the current working directory.  Similarly, if (not frame
// cache file is specified, WSCAN looks for the file framecache.txt in the
// current working directory.
//
// For information on the frameCacheFile, see READFRAMEDATA.
//
// The resulting spectrograms for statistically significant channels are written
// to an event sub-directory that is created in the specified output directory
// and is labelled by the GPS time of the event under analysis.  If no output
// directory is specified, a default output directory called wscans is created
// in the current working directory.  A web page named index.html is also 
// created in each event sub-directory and presents the resulting spectrograms 
// in table form.
//
// The specified debugLevel controls the amount of detail in the output log.
// A debugLevel of unity is assumed by default.
//
// See also WREADDATA, WRESAMPLE, WTILE, WCONDITION, WTRANSFORM, WTHRESHOLD,
// WSELECT, WSPECTROGRAM, WEVENTGRAM, WTIMESERIES, WCONFIGURE.SH and WSCAN.SH.

// Shourov K. Chatterji <shourov@ligo.mit.edu>

// $Id: wscan.m 2945 2010-04-28 21:17:26Z jrollins $

namespace wpipe {

  //==================================== Scan parameters for a single channel
  struct search_chan : public param_list {
    search_chan(void);
    search_chan(const search_chan& x);
    void init_map(void);
    void validate(int channelNumber, const search_chan& def);
    str_vect  channelName;
    dble_vect timeShifts;
    str_vect  frameType;
    double    sampleFrequency;
    double    searchTimeRange;
    dble_vect searchFrequencyRange;
    dble_vect searchQRange;
    double    searchMaximumEnergyLoss;
    bool      adjustLimits;
    double    whiteNoiseFalseRate;
    double    eventThreshold;
    double    searchWindowDuration;
    double    whiteningDuration;
    dble_vect plotTimeRanges;
    dble_vect plotFrequencyRange;
    dble_vect plotNormalizedEnergyRange;
    bool      alwaysPlotFlag;
    int       plotHorizontalResolution;
    int       plotVerticalResolution;
    std::string colorMap;
    std::string zUnits;
  };

  //====================================  Main program
  void
  wscan(const std::string& eventTimeString, 
	const std::string& configurationFile,                      
	const std::string& frameCacheFile,
	const std::string& outputDirectory, 
	bool generateReport, int debugLevel,
	const search_chan& def);
  
  typedef std::vector<search_chan> config_vect;

  struct section {
    std::string Name;
    std::string Index;
    int         Start;
    std::string Checked;
    std::string Display;
  };

  typedef std::vector<section> section_vect;

  void
  scan_config(const std::string& file, const search_chan& defval, 
	      config_vect& chans, section_vect& sections);

} // namespace wpipe
