/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "wtypes.hh"
#include "param_list.hh"
#include "Time.hh"
#include <string>
#include <iosfwd>

// WPLOT Create spectrograms of significant channels for candidate events
//
// WPLOT Examines gravitational-wave channels, auxiliary interferometer channels,
// and environmental channels around times of interest.  Spectrograms are
// generated for those channels that exhibit statistically significant behavior.
//
// usage: wplot(eventTime, configurationFile, frameCacheFile, 
//              outputDirectory, generateReport, debugLevel);
//
//   eventTime             GPS time of candidate event
//   configurationFile     path name of channel configuration file
//   frameCacheFile        path name of frame file cache file
//   outputDirectory       directory to write results
//   generateReport        generate plots and html report
//   debugLevel            verboseness of debug level output
//
// To allow use as a stand-alone executable, the specified eventTime should be
// a string, not a number.
//
// The configuration file is an ASCII text file describing the parameters for
// each channel to be analyzed.  The entries should have the following syntax.
//
// {
//   channelName:                 "H1:LSC-AS_Q"
//   frameType:                   "RDS_R_L1"
//   sampleFrequency:             4096
//   searchTimeRange:             16
//   searchFrequencyRange:        [64 1024]
//   searchQRange:                [4 64]
//   searchMaximumEnergyLoss:     0.2
//   adjustLimits                 false
//   whiteNoiseFalseRate:         1e-2
//   searchWindowDuration:        0.1
//   plotTimeRanges:              [0.1 1.0 10.0]
//   plotFrequencyRange:          [64 1024]
//   plotNormalizedEnergyRange:   [0 25.5]
//   alwaysPlotFlag:              0
// }
//
// Groups of related channels may optionally be separated by a section title
// specifier with the following form.
//
// [index_entry:section_title]
//
// This will be used to generate a index entry and section title in the 
// resulting web page.
//
// The WCONFIGURE.SH script can be used to automatically generate a reasonable
// configuration file for sample frame files.
//
// If no configuration file is specified, WPLOT looks for the file
// configuration.txt in the current working directory.  Similarly, if (not frame
// cache file is specified, WPLOT looks for the file framecache.txt in the
// current working directory.
//
// For information on the frameCacheFile, see READFRAMEDATA.
//
// The resulting spectrograms for statistically significant channels are written
// to an event sub-directory that is created in the specified output directory
// and is labelled by the GPS time of the event under analysis.  If no output
// directory is specified, a default output directory called wplots is created
// in the current working directory.  A web page named index.html is also 
// created in each event sub-directory and presents the resulting spectrograms 
// in table form.
//
// The specified debugLevel controls the amount of detail in the output log.
// A debugLevel of unity is assumed by default.
//
// See also WREADDATA, WRESAMPLE, WTILE, WCONDITION, WTRANSFORM, WTHRESHOLD,
// WSELECT, WSPECTROGRAM, WEVENTGRAM, WTIMESERIES, WCONFIGURE.SH and WPLOT.SH.

// Shourov K. Chatterji <shourov@ligo.mit.edu>

// $Id: wplot.m 2945 2010-04-28 21:17:26Z jrollins $

namespace wpipe {

  struct plot_chan : public param_list {
    plot_chan(void);
    void clear(void);
    void def_params(void);
    void display(std::ostream& out) const;
    void reset(void);
    void validate(void);
    str_vect    channelName;
    str_vect    frameType;
    dble_vect   timeShifts;
    std::string frameCacheFile;
    std::string ndsServer;
    std::string outDir;
    std::string plotFile;
    Time        eventTime;
    double      sampleFrequency;
    double      searchTimeRange;
    dble_vect   searchFrequencyRange;
    dble_vect   searchQRange;
    double      searchMaximumEnergyLoss;
    bool        adjustLimits;
    double      whiteNoiseFalseRate;
    double      eventThreshold;
    double      searchWindowDuration;
    dble_vect   plotTimeRanges;
    dble_vect   plotFrequencyRange;
    dble_vect   plotNormalizedEnergyRange;
    str_vect    plotType;
    double      transientFactor;
    double      outlierFactor;
    int         plotHorizontalResolution;
    int         plotVerticalResolution;
    double      plotDurationInflation;
    double      plotBandwidthInflation;
    double      plotThreshold;
    int         maximumSignificants;
    int         maximumMosaics;
    bool        plotThumbnail;
    int         nSmooth;
    std::string smoothOpt;
    dble_vect   plotSize;
    std::string plotFormat;
    std::string zUnits;
    std::string colorMap;
    std::string colorMapFile;
    int         debugLevel;
  };

  void
  wplot(const plot_chan& pars);

  void
  plot_config(int argc, const char* argv[], plot_chan& pars);


} // namespace wpipe
