/* -*- mode: c++; c-basic-offset: 3; -*- */
//======================================  W-pipe headers
#include "wplot.hh"
#include "wcondition.hh"
#include "weventlist.hh" 
#include "wfigure.hh"
#include "wframecache.hh"
#include "wmeasure.hh"
#include "wreaddata.hh"
#include "wresample.hh"
#include "wtile.hh"
#include "wtransform.hh"
#include "matlab_fcs.hh"
#include "param_list.hh"

//======================================  DMT headers
#include "TSeries.hh"
#include "fSeries/DFT.hh"
#include "Interval.hh"
#include "frame_name.hh"

//======================================  C++ and system headers.
#include <fstream>
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <cstdio>
#include <cmath>
#include <iomanip>
#include <stdexcept>
#include <algorithm>
#include <unistd.h>

using namespace std;
using namespace wpipe;

//======================================  Syntax output function
void
wplot_syntax(void) {
   cout << "The dmt_wplot command syntax is:" << endl;
   cout << "  wplot <argument-1> [ ... <argument-n>]" << endl;
   cout << "arguments are in the form <keyword>=<value>" << endl;
}


//======================================  Plot names
const std::string evnt_raw   = "eventgram_raw";
const std::string evnt_white = "eventgram_whitened";
const std::string evnt_auto  = "eventgram_autoscaled";
const std::string spec_asd   = "asd_spectrogram";
const std::string spec_raw   = "spectrogram_raw";
const std::string spec_white = "spectrogram_whitened";
const std::string spec_auto  = "spectrogram_autoscaled";
const std::string tser_raw   = "timeseries_raw";
const std::string tser_white = "timeseries_whitened";
const std::string tser_hipas = "timeseries_highpassed";


//======================================  Main function.
int 
main(int argc, const char* argv[]) {
   plot_chan par;

   try {
      plot_config(argc, argv, par);
   } catch (std::exception& e) {
      cerr << "Error reading configuration: " << e.what() << endl;
      return 1;
   }

   // cout << "wplot parameters: " <<endl;
   // par.display(cout);

   try {
      wplot(par);
   }
   catch (exception& e) {
      cerr << "caught exception: " << e.what() << endl;
   }
}

//======================================  plot function.
void
wpipe::wplot(const plot_chan& pchan) {

   /////////////////////////////////////////////////////////////////////////////
   //                        process command line arguments                   //
   /////////////////////////////////////////////////////////////////////////////
   str_vect channelName = pchan.channelName;
   size_t   nChan       = channelName.size();
   str_vect frameType   = pchan.frameType;
   dble_vect timeShifts = pchan.timeShifts;
   int      debugLevel  = pchan.debugLevel;
   
   // convert string event time and debug level to numbers
   ostringstream outstr;
   outstr << fixed << setprecision(3) << pchan.eventTime.totalS();
   string eventTimeString = outstr.str();

   // report status
   if (debugLevel >= 1) {
      time_t clock=0;
      cout << "OmegaScan of event at " << eventTimeString << endl;
      cout << "created by " << getenv("USER") << " on " << datestr(clock, 29)
	   <<" at " << datestr(clock, 13) << endl;
   }

   /////////////////////////////////////////////////////////////////////////////
   //                           load frame cache file                         //
   /////////////////////////////////////////////////////////////////////////////

   // report status
   if (debugLevel >= 1 && !pchan.frameCacheFile.empty()) {
      cout << "Building frame cache from: " << pchan.frameCacheFile << endl;
   }

   // load frame file cache
   wframecache frameCache(pchan.frameCacheFile);
   if (frameCache.empty() && (frameType.empty() || frameType[0] == "NDS2")) {
      frameType.resize(nChan, "NDS2");
      frameCache.add_group("H-NDS2", 0, 1999999999, 0, pchan.ndsServer);
      frameCache.add_group("G-NDS2", 0, 1999999999, 0, pchan.ndsServer);
      frameCache.add_group("L-NDS2", 0, 1999999999, 0, pchan.ndsServer);
      frameCache.add_group("V-NDS2", 0, 1999999999, 0, pchan.ndsServer);
   }

   if (debugLevel > 2) frameCache.display();

   /////////////////////////////////////////////////////////////////////////////
   //                         create output directory                         //
   /////////////////////////////////////////////////////////////////////////////

   // if (outputDirectory not specified, make one based on center time
   frame_name outFile("plots/%g", pchan.plotFile, 0, "");
   if (!pchan.outDir.empty()) outFile.set_directory(pchan.outDir);

   string outputDirectory = outFile.dir_name(pchan.eventTime);
   if (debugLevel >= 1) {
      cout << "creating event directory" << endl;
      cout << "  outputDirectory:         " << outputDirectory << endl;
   }
   //-----------------------------------  The directory name must be reset
   //                                     in outFile because the final image
   //                                     path will use time=0 to suppress
   //                                     the start time and dt fields.
   outFile.set_directory(outputDirectory);
   outFile.make_dir(outputDirectory, 2);

   ////////////////////////////////////////////////////////////////////////////
   //                       loop over configuration channels                 //
   ////////////////////////////////////////////////////////////////////////////
  
   resampler resample;
   resample.reset();
   dble_vect emptyRange;

   if (debugLevel > 1) {
      cout << "wplot parameters" << endl;
      pchan.display(cout);
      cout << endl;
   }

   /////////////////////////////////////////////////////////////////////////
   //               identify statistically significant channels           //
   /////////////////////////////////////////////////////////////////////////
   double    sampleFrequency =      pchan.sampleFrequency;
   double    timeRange =            pchan.searchTimeRange;
   dble_vect frequencyRange =       pchan.searchFrequencyRange;
   dble_vect qRange =               pchan.searchQRange;
   double    maximumEnergyLoss =    pchan.searchMaximumEnergyLoss;
   bool      adjustLimits =         pchan.adjustLimits;
   double    searchWindowDuration = pchan.searchWindowDuration;

   // standardize channel names
   string channelNameString = strrep(channelName[0], ";", ":");

   // Plot parameters
   dble_vect plotTimeRanges            = pchan.plotTimeRanges;
   dble_vect plotFrequencyRange        = pchan.plotFrequencyRange;
   dble_vect plotNormalizedEnergyRange = pchan.plotNormalizedEnergyRange;

   // find closest sample time to event time
   Time centerTime = Time(pchan.eventTime.getS());
   centerTime += round(double(pchan.eventTime-centerTime)*sampleFrequency)
               / sampleFrequency;

   // determine segment start and stop times
   Time startTime = centerTime - Interval(timeRange / 2.);
   if (startTime-Time(startTime.getS()) < Interval(0.5)) {
      startTime = Time(startTime.getS());
   } else {
      startTime = Time(startTime.getS()+1);
   }
   Time stopTime = startTime + timeRange;

   // generate search tiling
   wlog(debugLevel, 2, "  tiling for search");
   double highPassCutoff = -1;
   double lowPassCutoff = -1;
   double whiteningDuration = -1;
   int minAllowableIndep = 50;

   wtile tiling(timeRange, qRange, frequencyRange, sampleFrequency, 
		maximumEnergyLoss, highPassCutoff, lowPassCutoff, 
		whiteningDuration, pchan.transientFactor, 
		minAllowableIndep, adjustLimits, debugLevel);


   //  The threshold is set to yield the specified false event rate when 
   //  applied to all available frequencies and Qs of a white noise 
   //  signal. It is not modified to account for restricted f, Q ranges. 
   //  It is also only a rough estimate, and the result false event rate
   //  may vary significantly depending on the quality of the data.
   //
   double whiteNoiseFalseRate = pchan.whiteNoiseFalseRate;
   double eventThreshold = pchan.eventThreshold;
   if (eventThreshold == 0) {
      eventThreshold = tiling.threshold_from_rate(whiteNoiseFalseRate);
   }

   //========================================================================
   //  This has to be fixed for reasonable data I/O
   // read data from frame file
   wlog(debugLevel, 2, "  reading data");
   tser_vect rawData;
   Interval primetime = 1.0;
   wreaddata(frameCache, channelName, frameType, startTime-primetime,
	     stopTime, timeShifts, debugLevel, rawData);

   // check for read error
   if (rawData.empty()) {
      wlog(debugLevel, 1, "  ERROR: cannot load frame data");
      return;
   }

   // test multiple channels for consistency
   if (rawData.size() > 1) {
      wlog(debugLevel, 2, "  subtracting data");
      rawData[0] -= rawData[1];
      rawData.erase(rawData.begin()+1, rawData.end());
   }

   //--------------------------------  Check for all zeros
   if (rawData[0].getMinimum() == rawData[0].getMaximum()) {
      wlog(debugLevel, 1, "  WARNING: channel contains no information");
      return;
   }
   
   //--------------------------------  prime the resampler
   resample.prime_from_data(rawData, primetime, sampleFrequency);

   //--------------------------------  Resample data
   wlog(debugLevel, 2, "  resampling data");
   tser_vect rsamp = resample.wresample(rawData, sampleFrequency);
   
   // high pass filter and whiten data
   wlog(debugLevel, 2, "  conditioning data");

   wcondition conditioner(rsamp, tiling);
   containers::DFT whitenedData  = conditioner.whitenedDFT();
   containers::DFT whitenedCoefs = conditioner.coefficientDFT();

   // q transform whitened data
   wlog(debugLevel, 2, "  transforming whitened data");
   wtransform whitenedTransform(whitenedData, tiling, whitenedCoefs,
				pchan.outlierFactor, channelNameString);
   //whitenedTransform.dump(cout);

   // identify most significant whitened transform tile
   wlog(debugLevel, 2, "  measuring peak significance");
   Time thresholdReferenceTime = centerTime;
   dble_vect thresholdTimeRange(2);
   thresholdTimeRange[0] = -0.5*searchWindowDuration;
   thresholdTimeRange[1] =  0.5*searchWindowDuration;
   dble_vect thresholdQRange;
   dble_vect thresholdFrequencyRange;
   wmeasure whitenedProperties(whitenedTransform, tiling, startTime, 
			       thresholdReferenceTime, thresholdTimeRange, 
			       thresholdFrequencyRange, thresholdQRange, 
			       debugLevel);
   //whitenedProperties.dump(cout);

   //--------------------------------  If channel not significant
   if (whitenedProperties[0].peak.NormalizedEnergy < pchan.plotThreshold) {
      cerr << "Channel: " << channelNameString 
	   << " failed event thresold test, peak = " 
	   << whitenedProperties[0].peak.NormalizedEnergy
	   << " threshold = " << eventThreshold << endl;
      return;
   } // end test of channel significance

   /////////////////////////////////////////////////////////////////////////
   //               analyze statistically significant channels            //
   /////////////////////////////////////////////////////////////////////////
 
   //--------------------------------  Q transform raw data
   wlog(debugLevel, 2, "  transforming raw data");
   wtransform rawTransform(conditioner.rawDFT(), tiling, whitenedCoefs,
			   pchan.outlierFactor, channelNameString);
   //rawTransform.dump(cout);

   weventstack rawEvents, whitenedEvents;

   // identify significant whitened q transform tiles
   double maxPlotTRange = *(max_element(plotTimeRanges.begin(), 
					plotTimeRanges.end()));
   thresholdTimeRange[0] = -0.5*(maxPlotTRange 
				 + tiling.end_plane().row(0).duration 
				 * pchan.plotDurationInflation);
   thresholdTimeRange[1] = -thresholdTimeRange[0];
   thresholdFrequencyRange = plotFrequencyRange;
   thresholdQRange = emptyRange;

   //----------------  Identify significant tiles for whitened eventgram
   const_strv_iter begPlot = pchan.plotType.begin();
   const_strv_iter endPlot = pchan.plotType.end();

   if (find(begPlot, endPlot, evnt_white) != endPlot ||
       find(begPlot, endPlot, evnt_auto)  != endPlot) {

      wlog(debugLevel, 2, "  thresholding whitened transform");
      weventstack whitenedSignificants;
      whitenedSignificants.wthreshold(
			whitenedTransform, tiling, eventThreshold, 
			thresholdReferenceTime, thresholdTimeRange, 
			thresholdFrequencyRange, thresholdQRange, 
			pchan.maximumSignificants, "independent", 0, 0, 0, 
			debugLevel);
      //whitenedSignificants.dump(cout);

      // select non-overlapping whitened significant tiles
      wlog(debugLevel, 2, "  selecting whitened events");
      whitenedEvents.wselect(whitenedSignificants, pchan.plotDurationInflation, 
			  pchan.plotBandwidthInflation,  pchan.maximumMosaics, 
			  debugLevel);
   }

   //---------------- Identify significant raw q transform tiles
   if (find(begPlot, endPlot, evnt_raw) != endPlot) {
      wlog(debugLevel, 2, "  thresholding raw transform");
      weventstack rawSignificants;
      rawSignificants.wthreshold(rawTransform, tiling,  
				 eventThreshold, thresholdReferenceTime,
				 thresholdTimeRange, thresholdFrequencyRange,
				 thresholdQRange, pchan.maximumSignificants, 
				 "independent", 0, 0, 0, debugLevel);
      //rawSignificants.dump(cout);

      // select non-overlapping raw significant tiles
      wlog(debugLevel, 2, "  selecting raw events");
      rawEvents.wselect(rawSignificants, pchan.plotDurationInflation, 
			pchan.plotBandwidthInflation, pchan.maximumMosaics, 
			debugLevel);
   }

   //////////////////////////////////////////////////////////////////////////
   //                     add channel to summary report                    //
   //////////////////////////////////////////////////////////////////////////

   // identify most significant raw transform tile
   wlog(debugLevel, 2, "  measuring peak properties");
   thresholdReferenceTime = whitenedProperties[0].peak.centerTime();
   thresholdTimeRange[0] = -whitenedProperties[0].peak.Duration;
   thresholdTimeRange[1] = -thresholdTimeRange[0];
   thresholdFrequencyRange.resize(2);
   thresholdFrequencyRange[0] = whitenedProperties[0].peak.Frequency 
      - whitenedProperties[0].peak.Bandwidth;
   thresholdFrequencyRange[1] = whitenedProperties[0].peak.Frequency 
      + whitenedProperties[0].peak.Bandwidth;
   thresholdQRange = emptyRange;
   wmeasure rawProperties(rawTransform, tiling, startTime, 
			  thresholdReferenceTime, thresholdTimeRange, 
			  thresholdFrequencyRange, thresholdQRange, 
			  debugLevel);
   //rawProperties.dump(cout);
 
   // most significant tile properties
   Time mostSignificantTime =  whitenedProperties[0].peak.centerTime();
   double mostSignificantFrequency = whitenedProperties[0].peak.Frequency;
   double mostSignificantQ =  whitenedProperties[0].peak.Q;
   double mostSignificantNormalizedEnergy =
      whitenedProperties[0].peak.NormalizedEnergy;

   //----------------------------------------------------------------------
   //   This was changed to use signalAmplitude[0]
   // double mostSignificantAmplitude = rawProperties{1}.peakAmplitude;
   // double mostSignificantAmplitude = rawProperties[0].signalAmplitude;
   double mostSignificantAmplitude = rawProperties[0].signal.Amplitude;

   // write most significant tile properties to text summary file
   wlog(debugLevel, 2, "  writing text summary");
   cerr.precision(3);
   cerr << "Channel name:                        " << channelNameString << endl;
   cerr << "Most significant tile time:          " 
	<< fixed << mostSignificantTime.totalS() << endl;
   cerr << "Most significant tile Frequency:     " << mostSignificantFrequency 
	<< endl;
   cerr << "Most significant tile Q:             " << mostSignificantQ 
	<< endl; 
   cerr << "Most significant tile normalized E:  " 
	<< mostSignificantNormalizedEnergy << endl;
   cerr.unsetf(ios_base::floatfield);
   cerr << "Most significant tile amplitude:     " << mostSignificantAmplitude 
	<< endl;

   //===================================  begin loop over time ranges
   for (size_t iPlTiR=0; iPlTiR < plotTimeRanges.size(); iPlTiR++) {
      double plotTimeRange =  plotTimeRanges[iPlTiR];
      dble_vect autoNormalizedEnergyRange;

      // report status
      if (debugLevel >= 2) {
	 cout << "  processing " << plotTimeRange << " second time range" 
	      << endl;
      }

      dble_vect tRange(2);
      tRange[0] = -plotTimeRange/2;
      tRange[1] =  plotTimeRange/2;

      dble_vect qRange(1, mostSignificantQ);
	 
      //--------------------------  Loop over plots;
      int nPlot = pchan.plotType.size();
      for (int iplot=0; iplot< nPlot; iplot++) {
	 wfigure figureHandle;

	 if (!pchan.plotFormat.empty()) {
	    figureHandle.set_format(pchan.plotFormat);
	 }
	 
	 figureHandle.open();

	 if (!pchan.plotSize.empty()) {
	    int nx = int(pchan.plotSize[0]);
	    int ny = int(pchan.plotSize[0] * 5.0 / 7.0);
	    if (pchan.plotSize.size() > 1) ny = int(pchan.plotSize[1]);
	    figureHandle.set_size(nx, ny);
	 }

	 if (!pchan.colorMapFile.empty()) {
	    figureHandle.read_palette(pchan.colorMapFile);
	 } else if (!pchan.colorMap.empty()) {
	    figureHandle.set_palette(pchan.colorMap);
	 }
	 figureHandle.set_zunits(pchan.zUnits);
	 
	 string pltType = pchan.plotType[iplot];
	 //-----------------------------  Raw time-series
	 if (pltType == tser_raw) {
	    figureHandle.wtimeseries(rawData[0], 
				     centerTime, 
				     -plotTimeRange/2,
				     plotTimeRange/2, 
				     channelNameString);
	 }

	 //-----------------------------  High-passed time-series
	 else if (pltType == tser_hipas) {
	    figureHandle.wtimeseries(conditioner.highPassedData(), 
				     centerTime, 
				     -plotTimeRange/2, plotTimeRange/2, 
				     channelNameString);
	 }

	 //-----------------------------  Whitened time-series
	 else if (pltType == tser_white) {
	    figureHandle.wtimeseries(conditioner.whitenedData(), 
				     centerTime, 
				     -plotTimeRange/2, plotTimeRange/2, 
				     channelNameString);
	 }

	 //-----------------------------  ASD spectrogram
	 else if (pltType == spec_asd) {
	    Interval dT = rawData[0].getInterval()
	                / double(pchan.plotHorizontalResolution);
	    figureHandle.asdspectrogram(rawData[0], dT,
					plotFrequencyRange, 
					channelNameString);
	 }

	 //-----------------------------  Raw spectrogram
	 else if (pltType == spec_raw) {
	    figureHandle.set_smooth(pchan.nSmooth, pchan.smoothOpt);
	    figureHandle.wspectrogram(rawTransform[0], tiling, 
				      centerTime, 
				      tRange, plotFrequencyRange, 
				      qRange, plotNormalizedEnergyRange, 
				      pchan.plotHorizontalResolution, 
				      pchan.plotVerticalResolution);
	 }

	 //-----------------------------  Whitened spectrogram
	 else if (pltType == spec_white) {
	    figureHandle.set_smooth(pchan.nSmooth, pchan.smoothOpt);
	    figureHandle.wspectrogram(whitenedTransform[0], tiling,
				      centerTime, tRange, 
				      plotFrequencyRange, 
				      qRange, plotNormalizedEnergyRange, 
				      pchan.plotHorizontalResolution, 
				      pchan.plotVerticalResolution);
	 }

	 //-----------------------------  Auto-scaled spectrogram
	 else if (pltType == spec_auto) {
	    figureHandle.set_smooth(pchan.nSmooth, pchan.smoothOpt);
	    figureHandle.wspectrogram(whitenedTransform[0], tiling, 
				      centerTime, tRange,
				      plotFrequencyRange, qRange, 
				      autoNormalizedEnergyRange, 
				      pchan.plotHorizontalResolution, 
				      pchan.plotVerticalResolution);
	 }

	 //-----------------------------  Raw eventgram
	 else if (pltType == evnt_raw) {
	    figureHandle.weventgram(rawEvents[0], tiling, centerTime, 
				    tRange, plotFrequencyRange, 
				    pchan.plotDurationInflation,
				    pchan.plotBandwidthInflation, 
				    plotNormalizedEnergyRange);
	 }

	 //-----------------------------  Whitened eventgram
	 else if (pltType == evnt_white) {
	    figureHandle.weventgram(whitenedEvents[0], tiling, centerTime,
				    tRange, plotFrequencyRange, 
				    pchan.plotDurationInflation, 
				    pchan.plotBandwidthInflation, 
				    plotNormalizedEnergyRange);
	 }

	 //-----------------------------  Auto-scaled eventgram
	 else if (pltType == evnt_auto) {
	    figureHandle.weventgram(whitenedEvents[0], tiling, centerTime, 
				    tRange, plotFrequencyRange, 
				    pchan.plotDurationInflation, 
				    pchan.plotBandwidthInflation, 
				    autoNormalizedEnergyRange);
	 }
	 else {
	    cerr << "Unrecognized plot type: " << pltType << endl;
	    continue;
	 }

	 //----------------------------- Generate the file path.
	 ostringstream figName;
	 if (pchan.plotFile.empty()) {
	    figName << eventTimeString << "_" << channelNameString << "_" 
		    << fixed << setprecision(2) << plotTimeRange 
		    << "_" << pltType;
	    outFile.set_prefix(figName.str());
	 } else if (nPlot > 1) {
	    figName << pchan.plotFile << "_" << pltType;
	    outFile.set_prefix(figName.str());
	 }
	 figureHandle.wprintfig(outFile.file_path(Time(0)), 
				pchan.plotThumbnail);
	 figureHandle.clear();
      }
   }   // end loop time ranges


   //-----------------------------------  Exit
   if (debugLevel >= 1) {
      cout << "finished on " << datestr(0, 29) 
	   << " at " << datestr(0, 13) << endl;
   }
   //figureHandle.close();
}

//======================================  Search channel constructor.
plot_chan::plot_chan(void) {
   def_params();
   reset();
}

//======================================  Plot type list
static const char* plot_types[] = {
      "spectrogram_whitened", "spectrogram_raw", "spectrogram_autoscaled",
      "timeseries_raw", "timeseries_whitened", "timeseries_highpassed", 
      "eventgram_whitened", "eventgram_raw", "eventgram_autoscaled", 0};

//======================================  Set up parameter definitions.
void
plot_chan::def_params(void) {
   add_param("channelName", channelName);
   add_param("frameType", frameType);
   add_param("timeShifts", timeShifts);
   add_param("frameCacheFile", frameCacheFile);
   add_param("ndsServer", ndsServer);
   add_param("outDir", outDir);
   add_param("outputDirectory", outDir); // backward compatible alias
   add_param("plotFile", plotFile);
   add_param("eventTime", eventTime);
   add_param("sampleFrequency", sampleFrequency);
   add_param("searchTimeRange", searchTimeRange);
   add_param("searchFrequencyRange", searchFrequencyRange);
   add_param("searchQRange", searchQRange);
   add_param("searchMaximumEnergyLoss", searchMaximumEnergyLoss);
   add_param("adjustLimits", adjustLimits);
   add_param("whiteNoiseFalseRate", whiteNoiseFalseRate);
   add_param("eventThreshold", eventThreshold);
   add_param("searchWindowDuration", searchWindowDuration);
   add_param("plotTimeRanges", plotTimeRanges);
   add_param("plotFrequencyRange", plotFrequencyRange);
   add_param("plotNormalizedEnergyRange", plotNormalizedEnergyRange);
   add_param("plotType", plotType);
   add_param("transientFactor", transientFactor);
   add_param("outlierFactor", outlierFactor);
   add_param("plotHorizontalResolution", plotHorizontalResolution);
   add_param("plotVerticalResolution", plotVerticalResolution);
   add_param("plotDurationInflation", plotDurationInflation);
   add_param("plotBandwidthInflation", plotBandwidthInflation);
   add_param("plotThreshold", plotThreshold);
   add_param("maximumSignificants", maximumSignificants);
   add_param("maximumMosaics", maximumMosaics);
   add_param("plotThumbnail", plotThumbnail);
   add_param("nSmooth", nSmooth);
   add_param("smoothOption", smoothOpt);
   add_param("plotSize", plotSize);
   add_param("colorMap", colorMap);
   add_param("colorMapFile", colorMapFile);
   add_param("plotFormat", plotFormat);
   add_param("zUnits", zUnits);
   add_param("debugLevel", debugLevel);
}

//======================================  Set configuration struct to defaults
void
plot_chan::reset(void) {
   clear_all();
   const double dInf(1.0/0.0);
   frameType.resize(1, "NDS2");
   timeShifts.resize(1, 0.0);
   ndsServer = "nds.ligo.caltech.edu:31200";
   sampleFrequency = 4096;
   searchTimeRange = 64;
   double fRange[] = {0, dInf};
   searchFrequencyRange = dble_vect(fRange, fRange+2);
   double qRange[] = {4, 64};
   searchQRange = dble_vect(qRange, qRange+2);
   searchMaximumEnergyLoss = 0.2;
   adjustLimits = false;
   whiteNoiseFalseRate = 1;
   searchWindowDuration = 0.1;
   double plotRange[] = {1, 4, 16};
   plotTimeRanges = dble_vect(plotRange, plotRange+1);
   double plotFRange[]  = {0, dInf};
   plotFrequencyRange = dble_vect(plotFRange, plotFRange+2);
   //notused double defaultPlotMaximumEnergyLoss = 0.2;
   double plotNormER[] = {0, 25.5};
   plotNormalizedEnergyRange = dble_vect(plotNormER, plotNormER+2);

   // plot types
   plotType = str_vect(plot_types, plot_types+1);
   plotThreshold = 0.0;

   // search parameters
   transientFactor = 2.0;
   outlierFactor = 2.0;

   // display parameters
   plotHorizontalResolution = 512;
   plotVerticalResolution = 256;
   plotDurationInflation = 0.5;
   plotBandwidthInflation = 0.5;
   plotFormat = "png";
   plotThumbnail = false;
   zUnits = "NormE";

   // limits on number of significant tiles
   maximumSignificants = 10000;
   maximumMosaics = 10000;
}

//======================================  Scan the configuration.
void
wpipe::plot_config(int argc, const char* argv[], plot_chan& p) {
  
   //-----------------------------------  Get init file parameters
   string parfile;
   const char* homep = getenv("HOME");
   if (homep) {
      parfile = homep;
      parfile += "/";
   }
   parfile += ".wplotrc";
   if (exist(parfile, "file")) {
      cout << "Opening file: " << parfile << endl;
      ifstream in(parfile.c_str());
      while (in.good()) {
	 string line;
	 getline(in, line);
	 line = deblank(line);
	 size_t inx = line.find("#");
	 if (inx != string::npos) line.erase(inx);
	 if (!line.empty()) {
	    p.assign_param(line, ":=");
	 }
      }
   }

   //----------------------------------- Loop over command line arguments
   for (int iarg=1; iarg<argc; iarg++) {
      string cmdstr = argv[iarg];
      if (cmdstr == "--help") {
	 wplot_syntax();
	 cout << "The following keywords are known: " << endl;
	 cout << "  Keyword                    Default value(s)" << endl;
	 cout << "  -------                    ----------------" << endl;
	 p.display(cout);
	 error("Help requested");
      }

      // parse configuration line
      p.assign_param(cmdstr, ":=");

   }      // end loop over command line arguments.
   p.validate();
}

//======================================  Display channel parameters.
void
plot_chan::display(std::ostream& out) const {
   param_list::display(out, "  ");
}

//======================================  Check parameter validity
void
plot_chan::validate(void) {
   //-----------------------------------  See if channel name was specified.
   size_t nChan = channelName.size();
   if (nChan == 0) {
      throw runtime_error("channelName not specified");
   }
   //-----------------------------------  Pad out time shifts and frame types
   if (timeShifts.size() < nChan) timeShifts.resize(nChan, 0.0);
   if (!frameType.empty() && frameType.size() < nChan) {
      frameType.resize(nChan, frameType[0]);
   }
   //-----------------------------------  Enumerate "all" plot types
   if (plotType.size() == 1 && plotType[0] == "all") {
      plotType.clear();
      for (size_t i=0; plot_types[i] != 0; i++) {
	 plotType.push_back(plot_types[i]);
      }
   }
}
