/* -*- mode: c++; c-basic-offset: 3; -*- */
// $Id:$
#ifndef WPIPE_WPARAMETERS_HH
#define WPIPE_WPARAMETERS_HH

// WPARAMETERS Read WSEARCH parameter file
//
// WPARAMETERS reads the parameters from the specifiec parameter file for
// use by the WSEARCH, WEVENT, and WPROPERTIES functions.
//
// usage: parameters = wparameters(parameterFile, debugLevel)
//
//   parameterFile       path name of parameter file to read
//   debugLevel          verboseness level of debug output
//
//   parameters          parameter structure
//
// By default, WPARAMETERS assumes the parameterFile name './parameters.txt'
// and a debugLevel of 1.
//
// The parameter file syntax conists of a field name and value separate by a
// colon.  The value should be an expression evaluable by Matlab, and can
// include scalars, vectors, matrices, or cell arrays.
//
//   fieldName:             fieldValue
//
// Comments are delineated by '#', '//', or '//', and extend to the end of
// the current line.  Blank lines are allowed, and are simply ignored.
//
// The following fields are required:
//
//   channelNames:
//   frameTypes:
//
// The following fields are optional and have the listed default value:
//
//   analysisMode:          'independent'
//   sampleFrequency:       4096
//   qRange:                [sqrt(11) 100]
//   frequencyRange:        [48 Inf]
//   adjustLimits:          change frequency range as appropriate for Q plane
//   maximumMismatch:       0.2
//   falseEventRate:        1.0 (White noise false event rate in Hz)
//   eventThreshold:        0 (Normalized energy threshold = min-snr^2 + 1)
//   blockDuration:         64 (seconds)
//   conditionDuration:     (value of blockDuration)
//   dataFactors:           1
//   timeShifts:            0
//   injectionNames:        'NONE'
//   injectionTypes:        'NONE'
//   injectionFactors:      0
//   injectionTimeShifts:   0
//   highPassCutoff:        (determined from tiling, disabled if highPassCutoff=0)
//   lowPassCutoff:         (determined from tiling)
//   whiteningDuration:     (determined from tiling)
//   transientFactor:       4
//   doubleWhiten:          true (enable double whitening in wcondition)
//   extraBlockOverlap:     0
//   minimumIndependents:   50
//   outlierFactor:         2.0
//   minAllowableIndependents: 50
//   maximumSignificants:   100000 
//   maximumTriggers:       1000
//   durationInflation:     1.0
//   bandwidthInflation:    1.0
//   coincidenceNumber:     0
//   maximumCoincidents:    Inf
//   minTrriggerSNR:        0
//   triggerFields:         (depends on analysis mode and clustering)
//   triggerFiles:          (depends on analysis mode and clustering)
//   triggerFormat:         'txt'
//   randomSeed:            sum(1e6 * clock)
//
// The following optional fields are used to read detector state data:
//
//   stateNames:            
//   stateTypes:            
//   stateMasks:            
//   errorOnStateError:     false
//
// The following optional fields related to clustering:
//
//   applyClustering:       0
//   clusterMethod:         'density'
//   clusterRadius:         4.0                   (for density clustering)
//   clusterDensity:        3.0                   (for density clustering)
//   clusterSingles:        1                     (for density clustering)
//   clusterLinkage:        'single'              (for hierarchical clustering)
//   clusterCriterion:      'distance'            (for hierarchical clustering)
//   clusterThreshold:      4.0                   (for hierarchical clustering)
//   distanceMetric:        'integratedMismatch'
//   writeClusters:         0
//
// The following optional fields are for targeted searches:
//
//   skyPosition:           []
//   skyCoordinateSystem:   'equatorial'
//
// The following fields apply only to 'coherent' anlaysis mode:
//
//   applyVeto:             1
//   falseVetoRate:         0.0
//   uncertaintyFactor:     0.0
//   correlationFactor:     0.0
//   vetoDurationFactor:    0.5
//   vetoBandwidthFactor:   0.5
//   maximumConsistents:    1e3
//
//
// The following are for followup analyses.  The first parameter is what turns
// the followup on or off:
//
// 'bayesian' position reconstruction:
//
//   bayesian:              true
//   maxFollowTriggers:     5
//   snrRatioCut:           50
//   writeSkymap:           false
//   gzipSkymap:            false
//   prcInjectionFile:      '/path/to/injection/file'
//
// xCoherentCheck null energy statistic:
//
//   xCoherentCheck:        false
//
// MaxEnt waveform reconstruction:
//
//   maxEnt:                false
//
// Combined event statistic:
//
//   combinedStat:          false
//   combinedParameters:    [1.1 1]
//
// The returned parameters structure also includes the following derived fields:
//
//   numberOfChannels
//   numberOfSites
//   injectionChannels
//
// See also WSEARCH, WEVENT, and WPROPERTIES.

// Shourov K. Chatterji <shourov@ligo.caltech.edu>
// John Zweizig <john.zweizig@ligo.org>

#include "wtypes.hh"
#include "param_list.hh"
#include <iosfwd>

namespace wpipe {

   struct wparameters : public param_list {
      /**  Initialize parameter set with null values. All parameters are
       *  set either to empty or invalid state.
       *  \brief Null constructor.
       */
      wparameters(void);
      wparameters(const wparameters& x);
      wparameters(const std::string& parameterFile,
		  const wparameters& default_pars,
		  int debugLevel);

      /**  Build a list of unique sites from a channel name list.
        *  \brief Build network list
	*  \param chans Channel name list
	*  \return Network string
	*/
      static std::string buildNetworkString(const str_vect& chans);

      void display(std::ostream& out) const;


      /** Initialize the parameter list
       */
      void init_list(void);
      
      /**  Number of channel names in this group.
       *  \brief Number of Channels
       *  \return number of channel names in this group.
       */
      size_t numberOfChannels(void) const;

      /**  Number of unique sites represented by data channels in this group.
       *  \brief Number of sites
       *  \return Number of unique Sites in this group.
       */
      size_t numberOfSites(void) const;

      /**  Number of state channels defined for channels this group.
       *  \brief Number of state channels
       *  \return Number of state channels
       */
      size_t numberOfStates(void) const;
      bool   inject(size_t inx) const;
      size_t injectionChannels(void) const;

      void merge_defaults(const wparameters& d);

      /**  Read a single line from an open parameter file. Commments and leading
       *  and trailing spaces are removed.
       *  \brief Read a line
       *  \param parameterFileID Open parameter file.
       *  \return Input character string.
       */
      static std::string read_line(std::istream& parameterFileID);

      /**  Read parameters from the specified istream. The current parameter 
       *  set is updated with the parameter values specified in the file.
       *  Reading stops and an appropriate status code (rc) is returned if 
       *  a close brace (rc=0), end-of-file (rc=1) or error condition (rc=-1)
       *  is encountered. 
       *  \brief Read a parameter set.
       *  \param parFile    File to be read from.
       *  \param debugLevel debugging level.
       *  \return Status code indicating a close brace (rc=0), end-of-file 
       *          (rc=1) or error condition (rc=-1) was encountered.
       */
      int read_params(std::istream& parFile, int debugLevel);

      /**  Set all parameters to their default values.
       */
      void set_defaults(void);

      /** Validate the contents of the parameters set.
       */
      void validate(void) const;

      //--------------------------------  Useful info
      static str_vect validFileTypes;
      
      //--------------------- group name
      std::string groupName;        //  none

      //---------------------  Input data.
      str_vect channelNames;
      str_vect frameTypes;

      //--------------------- control parameters.
      std::string analysisMode;        //  'independent'
      double sampleFrequency;          // 4096
      dble_vect qRange;                // [sqrt(11) 100]
      dble_vect frequencyRange;        // [48 Inf]
      wtrit   adjustLimits;            // false
      double maximumMismatch;          // 0.2
      double falseEventRate;           // 1
      double eventThreshold;           // 0
      double blockDuration;            // 64 (seconds)
      double conditionDuration;        // (value of blockDuration)
      dble_vect dataFactors;           // 1
      dble_vect timeShifts;            // 0
      str_vect injectionNames;         // 'NONE'
      str_vect injectionTypes;         // 'NONE'
      dble_vect injectionFactors;      // 0
      dble_vect injectionTimeShifts;   // 0
      double highPassCutoff;           // (determined from tiling)
      double lowPassCutoff;            // (determined from tiling)
      double whiteningDuration;        // (determined from tiling)
      double transientFactor;          // 4
      wtrit  doubleWhiten;             // 1
      double extraBlockOverlap;        // 0
      double outlierFactor;            // 2.0
      int    minAllowableIndependents; // 50
      int    maximumSignificants;      // 1e5
      int    maximumTriggers;          // 1e3
      double durationInflation;        // 1.0
      double bandwidthInflation;       // 1.0
      int    coincidenceNumber;        // 0
      int    maximumCoincidents;       // Inf
      double minTriggerSNR;            // 0
      str_vect triggerFields;          // (depends on analysis mode, clustering)
      str_vect triggerTypes;           // (depends on analysis mode, clustering)
      str_vect triggerFiles;           // "XX-OMEGA_TRIGGER_<type>"
      std::string triggerFormat;       // 'txt'
      double randomSeed;               // sum(1e6 * clock)

      //--------------------------------  State information parameters
      str_vect stateNames;            
      str_vect stateTypes;            
      dble_vect stateMasks;            
      wtrit errorOnStateError;          // false

      //--------------------------------  condition tests
      std::string oscFile;             // osc condition definition file name
      std::string oscFrameType;        // frame-type used to test osc conditions
      str_vect oscConditions;          // osc condition for the i'th channel
      double   oscStride;              // osc stride
      
      //--------------------------------  Clustering parameters.
      wtrit applyClustering;           // 0
      std::string clusterMethod;       // 'density'
      double clusterRadius;            // 4.0 (density clustering)
      double clusterDensity;           // 3.0 (density clustering)
      int clusterSingles;              // 1   (density clustering)
      std::string clusterLinkage;      // 'single' (hierarchical clustering)
      std::string clusterCriterion;    // 'distance' (hierarchical clustering)
      double clusterThreshold;         // 4.0      (hierarchical clustering)
      std::string distanceMetric;      // 'integratedMismatch'
      wtrit writeClusters;             // 0

#ifndef NO_TARGETED
      //-------------------------------  Optional fields for targeted searches:
      dble_vect   skyPosition;         // []
      std::string skyCoordinateSystem; // 'equatorial'
#endif // !defined (NO_TARGETED)

      //--------------------------------  Fields for 'coherent' analysis
      wtrit applyVeto;                 // true
      double falseVetoRate;            // 0.0
      double vetoThreshold;            // 0
      double uncertaintyFactor;        // 0.0
      double correlationFactor;        // 0.0
      double vetoDurationFactor;       // 0.5
      double vetoBandwidthFactor;      // 0.5
      int maximumConsistents;          // 1e3

      // 'bayesian' position reconstruction:
      wtrit bayesian;                   // true
#ifndef NO_FOLLOWUP
      //---------------------------------  Parameters for followup analyses.
      int maxFollowTriggers;           // 5
      double snrRatioCut;              // 50
      wtrit writeSkymap;               // false
      wtrit gzipSkymap;                // false
      std::string prcInjectionFile;    // '/path/to/injection/file'

      // xCoherentCheck null energy statistic:
      wtrit xCoherentCheck;            // false

      // MaxEnt waveform reconstruction:
      wtrit maxEnt;                    // false

      // Combined event statistic:
      wtrit combinedStat;              // false
      dble_vect combinedParameters;    // [1.1 1]
#endif // !defined (NO_FOLLOWUP)

      //=============================  Extra first character of each channel
      std::string sites;
   };

   //===================================  Inline methods
   inline size_t
   wparameters::numberOfChannels(void) const {
      return channelNames.size();
   }

   inline size_t
   wparameters::numberOfSites(void) const {
      return sites.size();
   }

   inline size_t
   wparameters::numberOfStates(void) const {
      return stateNames.size();
   }

} // namespace wpipe

#endif //!defined(WPIPE_WPARAMETERS_HH)
