/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "wparameters.hh"
#include "matlab_fcs.hh"
#include <iostream>
#include <fstream>
#include <cstdlib>

using namespace wpipe;
using namespace std;

const char* trigTypeList[] = {"DOWNSELECT", "CLUSTER", "COINCIDE", "EVENTS"};
str_vect wparameters::validFileTypes(trigTypeList, trigTypeList+4);

//======================================  Construct a Null parameter list.
wparameters::wparameters(void) {
   init_list();
}

//======================================  Parameter list copy constructor
wparameters::wparameters(const wparameters& p) {
   init_list();
   copy(p);
}

//======================================  Construct parameter list from file.
wparameters::wparameters(const std::string& parFile,
			 const wparameters& default_pars,
			 int debugLevel) {
   init_list();
   
   //-----------------------------------  Get default parameter list file.
   string parameterFile = parFile;
   if (parameterFile.empty()) parameterFile = "./parameters.txt";

   // validate command line arguments
   if (!exist(parameterFile, "file")) {
      error(string("could not find parameter file: ") + parameterFile);
   }

   // open parameter file for reading
   ifstream parameterFileID(parameterFile.c_str());

   try  {
      read_params(parameterFileID, debugLevel);
      merge_defaults(default_pars);
   } 
   catch (std::exception& e) {
      parameterFileID.close();
      throw;
   }
   // close parameter file
   parameterFileID.close();

   // validate
   validate();
}

//======================================  Build network string from channel list
std::string
wparameters::buildNetworkString(const str_vect& chan) {
   string net;
   size_t N = chan.size();
   for (size_t i=0; i<N; ++i) {
      string site = chan[i].substr(0,1);
      if (net.empty()) {
	 net  = site;
      } else if (net.find(site) == string::npos) {
	 net += site;
      }
   }
   return net;
}

//======================================  Initialize the parameter list and
//                                        preset the empty values.
void
wparameters::init_list(void) {
   //-----------------------------------  Initialize all parameters
   add_param("channelNames", channelNames);
   add_param("frameTypes", frameTypes);
   add_param("analysisMode", analysisMode, "independent");
   add_param("sampleFrequency", sampleFrequency, 0);
   add_param("qRange", qRange);
   add_param("frequencyRange", frequencyRange);
   add_param("adjustLimits", adjustLimits);
   add_param("maximumMismatch", maximumMismatch, 0);
   add_param("falseEventRate",falseEventRate, 0);
   add_param("eventThreshold", eventThreshold, 0);
   add_param("blockDuration", blockDuration, 0);
   add_param("conditionDuration", conditionDuration, 0);
   add_param("dataFactors", dataFactors);
   add_param("timeShifts", timeShifts);
   add_param("stateNames", stateNames);
   add_param("stateTypes", stateTypes);
   add_param("stateMasks", stateMasks);
   add_param("errorOnStateError", errorOnStateError, false);
   add_param("highPassCutoff", highPassCutoff, -1.0);
   add_param("lowPassCutoff", lowPassCutoff, -1.0);
   add_param("whiteningDuration", whiteningDuration, 0);
   add_param("transientFactor", transientFactor, 4.0);
   add_param("doubleWhiten", doubleWhiten);
   add_param("extraBlockOverlap", extraBlockOverlap, 0);
   add_param("outlierFactor", outlierFactor, 2.0);
   add_param("minAllowableIndependents",minAllowableIndependents, 0);
   add_param("maximumSignificants", maximumSignificants, 0);
   add_param("maximumTriggers", maximumTriggers, 0);
   add_param("minTriggerSNR", minTriggerSNR, 0);
   add_param("durationInflation", durationInflation, 1.0);
   add_param("bandwidthInflation", bandwidthInflation, 1.0);
   add_param("triggerFields", triggerFields);
   add_param("triggerFiles", triggerFiles);
   add_param("triggerTypes", triggerTypes);
   add_param("triggerFormat", triggerFormat);
   add_param("randomSeed", randomSeed, 0.0);

   add_param("applyClustering", applyClustering);
   add_param("clusterMethod", clusterMethod);
   add_param("clusterRadius", clusterRadius, 0.0);
   add_param("clusterDensity", clusterDensity, 0.0);
   add_param("clusterSingles", clusterSingles, 0);
   add_param("clusterThreshold", clusterThreshold, 0.0);
   add_param("distanceMetric", distanceMetric);
   add_param("writeClusters", writeClusters, -1);

   add_param("coincidenceNumber", coincidenceNumber, 0);
   add_param("maximumCoincidents", maximumCoincidents, 999999999);

#ifndef NO_COHERENT
   //-----------------------------------  "coherent" mode parameters
   add_param("skyPosition", skyPosition);
   add_param("skyCoordinateSystem", skyCoordinateSystem);
#endif
  
   add_param("applyVeto", applyVeto, 1);
   add_param("falseVetoRate", falseVetoRate, 0);
   add_param("vetoThreshold", vetoThreshold, 0);
   add_param("uncertaintyFactor", uncertaintyFactor, 0);
   add_param("correlationFactor", correlationFactor, 0);
   add_param("vetoDurationFactor", vetoDurationFactor, 0.5);
   add_param("vetoBandwidthFactor", vetoBandwidthFactor, 0.5);
   add_param("maximumConsistents", maximumConsistents, 1000);

   //-----------------------------------  OSC parameters
   add_param("oscFile", oscFile);
   add_param("oscFrameType", oscFrameType);
   add_param("oscConditions", oscConditions);
   add_param("oscStride", oscStride, 1.0);
   
   //-----------------------------------  Injection parameters
   add_param("injectionNames", injectionNames);
   add_param("injectionTypes", injectionTypes);
   add_param("injectionFactors", injectionFactors);
   add_param("injectionTimeShifts", injectionTimeShifts);

   //-----------------------------------  Follow-up parameters
#ifndef NO_FOLLOWUP
   add_param("bayesian", bayesian, true);
   add_param("maxFollowTriggers", maxFollowTriggers, 5);
   add_param("writeSkymap", writeSkymap, false);
   add_param("gzipSkymap", gzipSkymap, false);
   add_param("prcInjectionFile", prcInjectionFile);
   add_param("snrRatioCut", snrRatioCut, 50.0);

   add_param("xCoherentCheck", xCoherentCheck, false);
   add_param("maxEnt", maxEnt, false);

   add_param("combinedStat", combinedStat, false);
   add_param("combinedParameters", combinedParameters);
#else
   add_param("bayesian", bayesian, false);
#endif
}

//====================================== Read a single line of a parameter file
std::string
wparameters::read_line(std::istream& parameterFileID) {
   std::string parameterLine;
   bool continuation = true;
   while (continuation && !parameterFileID.fail()) {
      string line;
      getline(parameterFileID, line);
      continuation = false;
      if (!line.empty()) {
	 if (line[line.size()-1] == '\\') {
	    line.erase(line.size()-1,1);
	    continuation=true;
	 }
	 deblank(line);
	 if (!parameterLine.empty()) parameterLine += " ";
	 parameterLine += line;
      }
   }

   if (!parameterLine.empty()) {
      // remove any comments
      string::size_type commentIndex = parameterLine.find_first_of("#%");
      string::size_type altIndex     = parameterLine.find("//");
      if (altIndex < commentIndex) commentIndex = altIndex;
      if (commentIndex != string::npos) parameterLine.erase(commentIndex);

      // remove leading and trailing blanks
      parameterLine = deblank(parameterLine);
   }
   return parameterLine;
}

//======================================  Construct parameter list from file.
int
wparameters::read_params(std::istream& parameterFileID, int debugLevel) {
   // begin loop over parameter file
   int rc = 1;
   while (!parameterFileID.eof()) {
      string parameterLine = read_line(parameterFileID);
      // if (empty line, skip to the next line
      if (parameterLine.empty()) continue;

      string::size_type braceIndex = parameterLine.find("{");
      if (braceIndex == parameterLine.size()-1) {
	 parameterLine.erase(braceIndex);
	 groupName = deblank(parameterLine);
	 continue;
      }

      if (parameterLine == "}") {
	 rc = 0;
	 break;
      }

      // locate field separator
      string::size_type colonIndex = parameterLine.find(":");

      // if (field separator not located, report syntax error
      if (colonIndex == string::npos) {
	 error(string("syntax error in parameter file line: ") + parameterLine);
      }

      // parse parameter line
      string parameterName  = parameterLine.substr(0, colonIndex);
      string parameterValue = parameterLine.substr(colonIndex + 1);
      parameterName = deblank(parameterName);
      parameterValue = deblank(parameterValue);

      // report parameter settings
      wlog(debugLevel, 3, parameterName + ": " + parameterValue);

      // assign parameters based on name
      // required parameters
      try {
	 set_param(parameterName, parameterValue);
      } catch (exception& e) {
	 // handle unknown parameters
	 wlog(debugLevel, 1, string("unknown parameter ") + parameterName
	                     + ".  skipping.");
      }  // end assign parameters based on name
    
   } // end loop over parameter file entries

   //------------------------------------  Build site list.
   sites = buildNetworkString(channelNames);
   return rc;
}

//======================================  Merge default values
void
wparameters::merge_defaults(const wparameters& d) {
   size_t nChan = numberOfChannels();
   // default values
   if (analysisMode.empty())   analysisMode    = d.analysisMode;
   if (sampleFrequency == 0)   sampleFrequency = d.sampleFrequency;
   if (qRange.empty())         qRange          = d.qRange;
   if (frequencyRange.empty()) frequencyRange  = d.frequencyRange;
   if (maximumMismatch == 0)   maximumMismatch = d.maximumMismatch;
   if (!adjustLimits.is_set()) adjustLimits    = d.adjustLimits;
   if (falseEventRate == 0)    falseEventRate  = d.falseEventRate;
   if (eventThreshold == 0)    eventThreshold  = d.eventThreshold;
   if (blockDuration == 0)     blockDuration   = d.blockDuration;
   if (conditionDuration == 0) conditionDuration = blockDuration;
   if (!minAllowableIndependents) minAllowableIndependents = d.minAllowableIndependents;
   if (!maximumSignificants)   maximumSignificants = d.maximumSignificants;
   if (!maximumTriggers)       maximumTriggers = d.maximumTriggers;
   if (minTriggerSNR == 0)     minTriggerSNR = d.minTriggerSNR;
   if (dataFactors.size() < nChan)    dataFactors.resize(nChan, 1.0);
   if (timeShifts.size() < nChan)     timeShifts.resize(nChan, 0.0);
   if (injectionNames.size() < nChan) injectionNames.resize(nChan);
   if (injectionTypes.size() < nChan) injectionTypes.resize(nChan, "NONE");
   if (injectionFactors.size() < nChan) injectionFactors.resize(nChan, 0.0);
   if (injectionTimeShifts.size() < nChan) injectionTimeShifts.resize(nChan, 0.0);
   if (highPassCutoff < 0)     highPassCutoff  = d.highPassCutoff;
   if (lowPassCutoff < 0)      lowPassCutoff   = d.lowPassCutoff;
   if (!doubleWhiten.is_set()) doubleWhiten    = d.doubleWhiten;
   if (triggerFormat.empty())  triggerFormat   = d.triggerFormat;
   if (oscFrameType.empty())   oscFrameType    = d.oscFrameType;
   if (oscFile.empty())        oscFile         = d.oscFile;

   //if (randomSeed == 0) {
   //  randomSeed = sum(1e6 * clock);
   //}

   //-----------------------------------  Defaults for clustering
   if (!applyClustering.is_set()) applyClustering  = d.applyClustering;
   if (!writeClusters.is_set())   writeClusters    = d.writeClusters;
   if (clusterMethod.empty())     clusterMethod    = d.clusterMethod;
   if (clusterRadius  == 0)       clusterRadius    = d.clusterRadius;
   if (clusterDensity == 0)       clusterDensity   = d.clusterDensity;
   if (clusterSingles == 0)       clusterSingles   = d.clusterSingles;
   if (clusterLinkage.empty())    clusterLinkage   = d.clusterLinkage;
   if (clusterCriterion.empty())  clusterCriterion = d.clusterCriterion;
   if (clusterThreshold == 0.0)   clusterThreshold = d.clusterThreshold;
   if (distanceMetric.empty())    distanceMetric   = d.distanceMetric;

#ifndef NO_COHERENT
   // defaults for coherent analysis
   if (skyCoordinateSystem.empty()) skyCoordinateSystem = d.skyCoordinateSystem;
   if (combinedParameters.empty())  combinedParameters  = d.combinedParameters;
#endif
  
   // default trigger fields
   if (triggerFields.empty()) {
      triggerFields = d.triggerFields;

      if (analysisMode == "coherent") {
	 triggerFields.push_back("incoherentEnergy");
      }

      if (applyClustering && writeClusters) {
	 triggerFields.push_back("clusterSize");
	 triggerFields.push_back("clusterNormalizedEnergy");
      }
   }

   // default trigger files
   if (triggerFiles.empty()) triggerTypes = d.triggerTypes;
   if (triggerTypes.empty() && !triggerFiles.empty()) {
      triggerTypes.swap(triggerFiles);
   }
   if (triggerTypes.empty()) {
      if (applyClustering) {
	 triggerTypes.push_back("CLUSTER");
      } else {
	 triggerTypes.push_back("DOWNSELECT");
      }
   }

   for (size_t i=0; i<nChan; ++i) {
      timeShifts[i] = long(timeShifts[i]*sampleFrequency + 0.5) / sampleFrequency;
   }
}

//======================================  Merge default values
void
wparameters::set_defaults(void) {
 
   // default values
   analysisMode    ="independent";
   sampleFrequency = 4096;

   doubleWhiten    = true;

   double qDefault[2] = {sqrt(11), 100.0};
   qRange = dble_vect(qDefault, qDefault+2);

   double fDefault[2] = {32.0, 1.0/0.0};
   frequencyRange     = dble_vect(fDefault, fDefault+2);
   adjustLimits       = false;

   maximumMismatch = 0.2;
   falseEventRate  = 1;
   blockDuration   = 64;
   conditionDuration = blockDuration;
   minAllowableIndependents = 50;
   maximumSignificants = 100000;
   maximumTriggers = 1000;
   dataFactors     = dble_vect(1, 1.0);
   timeShifts      = dble_vect(1, 0.0);
   injectionNames.resize(1);
   injectionTypes.resize(1, "NONE");
   injectionFactors.resize(1, 0.0);
   injectionTimeShifts = dble_vect(1, 0.0);
   minTriggerSNR = 0;
   triggerFormat   = "xml";

   // defaults for clustering
   applyClustering  = false;
   writeClusters    = false;
   clusterMethod    = "density";
   clusterRadius    = 4.0;
   clusterDensity   = 3.0;
   clusterSingles   = 1;
   clusterLinkage   = "single";
   clusterCriterion = "distance";
   clusterThreshold = 4.0;
   distanceMetric   = "integratedMismatch";

#ifndef NO_COHERENT
   // defaults for coherent analysis
   skyCoordinateSystem = "equatorial";

   double defCPar[2] = {1.1, 1.0};
   combinedParameters = dble_vect(defCPar, defCPar+2);
#endif
  
   // default trigger fields
   const char* defFields[5] = {"time", "frequency", "duration", 
			       "bandwidth", "normalizedEnergy"};
   triggerFields = str_vect(defFields, defFields+5);
}

//======================================  Validate parameters
void
wparameters::validate(void ) const {

   size_t nChan = numberOfChannels();
   if (!nChan) {
      error("channelNames not specified");
   }
   
   if (analysisMode == "coherent") {
      if (nChan < 2) {
	 error("coherent analysis mode requires at least two channels");
      }
   } else if (analysisMode != "independent") {
      error(string("unknown analysis mode ") + analysisMode);
   }

   // validate number of frame types
   if (frameTypes.size() != nChan) {
      error("number of frameTypes and channelNames are inconsistent");
   }

   // validate number of data factors
   if (dataFactors.size() != nChan && !dataFactors.empty()) {
      error("number of dataFactors and channelNames are inconsistent");
   }

   // validate number of time shifts
   if (timeShifts.size() != nChan && !timeShifts.empty()) {
      error("number of timeShifts and channelNames are inconsistent");
   }

   // validate number of state names
   size_t nState = stateNames.size();
   if (nState > nChan) {
      error("number of stateNames and channelNames are inconsistent");
   }

   // validate number of state types
   if (stateTypes.size() != nState) {
      error("number of stateTypes and stateNames are inconsistent");
   }

   // validate number of state masks
   if (stateMasks.size() != nState) {
      error("number of stateMasks and stateNames are inconsistent");
   }

   // validate number of injection names
   size_t nInject = injectionNames.size();
   if (nInject > nChan) {
      error("number of injectionNames and channelNames are inconsistent");
   }

   // validate number of injection types
   if (injectionTypes.size() != nInject) {
      error("number of injectionTypes and injections are inconsistent");
   }

   // validate number of injection factors
   if (injectionFactors.size() != nInject && !injectionFactors.empty()) {
      error("number of injectionFactors and injections are inconsistent");
   }

   //-----------------------------------  Warn of incomplete injection spec.
   for (size_t i=0; i<nInject; i++) {
      if (!injectionNames[i].empty() && injectionNames[i] != "NONE") {
	 if (injectionTypes[i].empty() || injectionTypes[i] == "NONE") 
	    error("Injection channel specified with invalid frame type");
	 if (injectionFactors[i] == 0.0) {
	    cout << "warning: zero factor on valid injection channel: "
		 << injectionNames[i] << endl;
	 }
      }
   }

   //-----------------------------------  Validate injection time shift count
   if (injectionTimeShifts.size() != nInject && !injectionTimeShifts.empty()) {
      error("number of injectionTimeShifts and injections are inconsistent");
   }

   // validate block duration
   if (blockDuration <= 0) {
      error("block duration must be non-zero and positive");
   } else if (blockDuration != double(int(blockDuration))) {
      error("block duration must be an integer");
   }

   if (!applyClustering.is_set()) {
      error("applyClustering parameter not set");
   }

   if (writeClusters < 0) {
      error("writeClusters parameter not set");
   }

   if (triggerFormat != "txt" && triggerFormat != "xml") {
      error(string("Unrecognized trigger format: ") + triggerFormat);
   }


#ifndef NO_TARGETED
   // validate sky coordinate system and position
   if (skyCoordinateSystem != "equatorial" &&
       skyCoordinateSystem != "geocentric" &&
       skyCoordinateSystem != "galactic") {
      error("unknown sky coordinate system");
   }
#endif

#ifndef NO_FOLLOWUP
   // check prc injection file location
   if (bayesian) {
      if (nChan < 2) {
	 error("bayesian followup requires at least two channels");
      }
      if (coincidenceNumber < 2) {
	 error("bayesian followup requires coincidenceNumber > 1");
      }
      if (!prcInjectionFile.empty() && !exist(prcInjectionFile,"file") ) {
	 error(string("PRC injection file ")+prcInjectionFile+" not found.");
      }
   }
#endif
}

//======================================  Inline function to extend a string
//
//   Return a string with at least the specified length. If the initial text
//   is longer than the specified length, the returned string will be identical
//   to the initiali string.
inline std::string
straw(const std::string& in, size_t wid) {
   string col(in);
   if (wid > col.size()) col.resize(wid, ' ');
   return col;
}

static std::ostream&
disp_svalue(std::ostream& out, const std::string& name, const std::string& s) {
   out << straw(name + ":", 32) << "'" << s << "'" << endl;
   return out;
}

//======================================  Displat all parameters.
void
wparameters::display(std::ostream& out) const {
   param_list::display(out, "");
   
   //-----------------------------------  Extra first character of each channel
   disp_svalue(out, "sites",                sites);
}

//======================================  Count injections.
bool
wparameters::inject(size_t inx) const {
   return injectionNames[inx] != "NONE" &&
      injectionTypes[inx] != "NONE" &&
      injectionFactors[inx] != 0.0;
}

//======================================  Count injections.
size_t
wparameters::injectionChannels(void) const {
   size_t nInj = 0;
   size_t N = numberOfChannels();
   for (size_t i=0; i<N; i++) {
      if (inject(i)) nInj++;
   }
   return nInj;
}
