/***************************************************************************
    File        : Fourier.cpp
    Description : Implements class Fourier
 ---------------------------------------------------------------------------
    Begin       : Fri Jul 27 2001
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#include "Fourier.h"



// Default constructor
gwd::Fourier::Fourier ()
{
  mSize        = 0;
  mRealData    = 0;
  mComplexData = 0;
  mPlanC2C     = 0;
  mPlanR2C     = 0;
  mPlanC2R     = 0;
}
// Initialize a plan for a complex DFT in-place
// Create a fftw-plan to improve performance for repeated ffts
gwd::Fourier::Fourier (const size_t NN,const int isign,std::vector<Complex>& data)
{
  Init(NN,isign,data);
}

// Initialize a plan for a real to complex DFT
// Create a fftw-plan to improve performance for repeated ffts
gwd::Fourier::Fourier (const size_t NN,const Vector& inData,std::vector<Complex>& outData)
{
  Init(NN,inData,outData);
}
// Constructor: initialize a plan for a complex to real DFT
// Create a fftw-plan to improve performance for repeated ffts
gwd::Fourier::Fourier (const size_t NN,const std::vector<Complex>& inData,Vector& outData)
{
  Init(NN,inData,outData);
}

// Desctructor
gwd::Fourier::~Fourier()
{
  // Data arrays are destroyed by
  // the main application
  mRealData    = 0;
  mComplexData = 0;
  if (mPlanC2C)
    fftw_destroy_plan(mPlanC2C);
  if (mPlanR2C)
    fftw_destroy_plan(mPlanR2C);
  if (mPlanC2R)
    fftw_destroy_plan(mPlanC2R);
}

// This function creates the fftw-plans
// Complex DFT in-place
bool
gwd::Fourier::Init(const size_t NN,const int isign,std::vector<Complex>& data)
{
  mSize = NN;

  // Check size of data array
  if (mSize != data.size())
    data.resize(mSize);
  // Set data pointer
  mRealData    = 0;
  mComplexData = (fftw_complex*)&data[0];

  // Create Plan
  mPlanC2C = fftw_plan_dft_1d((int)mSize,mComplexData,mComplexData,isign,FFTW_ESTIMATE);
  mPlanR2C = 0;
  mPlanC2R = 0;

  // End
  return true;
}

// This function creates the fftw-plans
// Real to Complex DFT
bool
gwd::Fourier::Init(const size_t NN,const Vector& inData,std::vector<Complex>& outData)
{
  mSize = NN;
  size_t cSize = NN/2 + 1;
  // Check size of data array
  if (mSize != inData.size())
  {
    Singleton* single = Singleton::exemplar();
    single->AppendMessage("gwd::Fourier::Init(): Input array has wrong dimension for real to complex fft");
    return false;
  }
  if (cSize != outData.size())
    outData.resize(cSize);
  // Set data pointer
  mRealData    = (double*)&inData[0];
  mComplexData = (fftw_complex*)&outData[0];

  // Create Plan
  mPlanC2C = 0;
  mPlanR2C = fftw_plan_dft_r2c_1d((int)mSize,mRealData,mComplexData,FFTW_ESTIMATE);
  mPlanC2R = 0;

  // End
  return true;
}

// This function creates the fftw-plans
// Real to Complex DFT
bool
gwd::Fourier::Init(const size_t NN,const std::vector<Complex>& inData,Vector& outData)
{
  mSize = NN;
  size_t cSize = NN/2 + 1;
  // Check size of data array
  if (mSize != outData.size())
    outData.resize(mSize);
  if (cSize != inData.size())
  {
    Singleton* single = Singleton::exemplar();
    single->AppendMessage("gwd::Fourier::Init(): Input array has wrong dimension for real to complex fft");
    return false;
  }
  // Set data pointer
  mComplexData = (fftw_complex*)&inData[0];
  mRealData    = (double*)&outData[0];

  // Create Plan
  mPlanC2C = 0;
  mPlanR2C = 0;
  mPlanC2R = fftw_plan_dft_c2r_1d((int)mSize,mComplexData,mRealData,FFTW_ESTIMATE);

  // End
  return true;
}

// Computes the dft.
// A fftw-plan has to be created previously
bool
gwd::Fourier::dft()
{
  if (mPlanC2C)
  {
    fftw_execute(mPlanC2C);
  }
  else if (mPlanR2C)
  {
    fftw_execute(mPlanR2C);
  }
  else if (mPlanC2R)
  {
    fftw_execute(mPlanC2R);
  }
  else
  {
    Singleton* single = Singleton::exemplar();
    single->AppendMessage("gwd::Fourier::dft(): no fftw3 plan was initialized");
    return false;
  }

  // end
  return true;
}

// Creates a fftw-plan and computes complex the dft in-place
bool
gwd::Fourier::dft(const int isign,std::vector<Complex>& inData)
{
  size_t cSize = inData.size();
  // Get pointer to data
  fftw_complex* _cData = (fftw_complex*)&inData[0];
  // Create Plan
  fftw_plan _c2cPlan = fftw_plan_dft_1d((int)cSize,_cData,_cData,isign,FFTW_ESTIMATE);
  // Execute
  fftw_execute(_c2cPlan);
  // Remove data
  fftw_destroy_plan(_c2cPlan);

  return true;
}


// Creates a fftw-plan and computes real to complex the dft
bool
gwd::Fourier::dft(const Vector& inData,std::vector<Complex>& outData)
{
  size_t rSize = inData.size();
  size_t cSize = rSize/2 + 1;
  if (cSize != outData.size())
    outData.resize(cSize);
  // Get pointer to data
  double*       _rData = (double*)&inData[0];
  fftw_complex* _cData = (fftw_complex*)&outData[0];
  // Create Plan
  fftw_plan _r2cPlan = fftw_plan_dft_r2c_1d((int)rSize,_rData,_cData,FFTW_ESTIMATE);

  // Execute
  fftw_execute(_r2cPlan);
  // Remove data
  fftw_destroy_plan(_r2cPlan);

  return true;
}


// Creates a fftw-plan and computes complex to realthe dft in-place
bool
gwd::Fourier::dft(const std::vector<Complex>& inData,Vector& outData)
{
  const size_t cSize = inData.size();
  size_t rSize = 2*(cSize - 1);
//  if (rSize != outData.size())
  if (outData.size() == (rSize+1))
    rSize++;

  if (outData.size() != rSize)
    outData.resize(rSize);

  // Get pointer to data
  fftw_complex* _cData = (fftw_complex*)&inData[0];//static_cast<fftw_complex*>(inData.begin());
  double*       _rData = (double*)&outData[0];
  // Create Plan
  fftw_plan _c2rPlan = fftw_plan_dft_c2r_1d((int)rSize,_cData,_rData,FFTW_ESTIMATE);
  // Execute
  fftw_execute(_c2rPlan);
  // Remove data
  fftw_destroy_plan(_c2rPlan);

  return true;
}


// Sort frequencies
bool
gwd::Fourier::Sort(std::vector<Complex>& inData)
{
  size_t NN = inData.size();
  size_t N2;
  if (IsOdd((unsigned int)NN))
  {
    N2 = NN / 2;
    size_t counter = 0;
    size_t index;
    size_t nn = 0;
    Complex val1 = inData[nn];
    Complex val2;
    while(counter < NN)
    {
      index = (nn+N2)%NN;
      val2 = inData[index];
      inData[index] = val1;
      val1 = val2;
      nn = index;
      counter++;
    }
  }
  else
  {
    N2 = NN / 2;
    for (size_t nn = 0; nn < N2; nn++)
    {
      size_t index = (nn+N2)%NN;
      std::swap<Complex>(inData[index],inData[nn]);
    }
  }

  // ending
  return true;
}
// Sort frequencies
bool
gwd::Fourier::Sort(const std::vector<Complex>& inData,std::vector<Complex>& sortedData)
{
  sortedData = inData;
  Sort(sortedData);
  return true;
}
// Sort frequencies
bool
gwd::Fourier::Sort(Vector& inData)
{
  size_t NN = inData.size();
  size_t N2;
  if (IsOdd((unsigned int)NN))
  {
    N2 = NN / 2;
    size_t counter = 0;
    size_t index;
    size_t nn = 0;
    double val1 = inData[nn];
    double val2;
    while(counter < NN)
    {
      index = (nn+N2)%NN;
      val2 = inData[index];
      inData[index] = val1;
      val1 = val2;
      nn = index;
      counter++;
    }
  }
  else
  {
    N2 = NN / 2;
    for (size_t nn = 0; nn < N2; nn++)
    {
      size_t index = (nn + N2)%NN;
      std::swap<double>(inData[index],inData[nn]);
    }
  }
  return true;
}
// Sort frequencies
bool
gwd::Fourier::Sort(const Vector& inData,Vector& sortedData)
{
  sortedData = inData;
  Sort(sortedData);
  return true;
}

