/*  File LineTable.cc      
    Sergey Klimenko
    produce table of significant lines and post it as a html file.
*/

#ifndef _USE_DMT
#define _USE_DMT
#endif

#include <fstream>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream>
#include "LineFilter.hh"
#include "LineTable.hh"
#include "html/color.hh"
#include "html/document.hh"
#include "html/table.hh"
#include "html/size.hh"
#include "html/text.hh"
#include "html/writer.hh"

using namespace std;

//--- LineTable constructor -------------

LineTable::LineTable(double res, size_t nS, double snr) 
  : wavearray<double>(), 
    nSub(1), limSNR(2.), fftRes(1.),  currentTime(0.), startTime(0.),
    liveTime(0.)
{ 
   fftRes = res;
   nSub = nS;
   limSNR = snr;
   lineTable.clear();
   lineList.clear();
   lmonList.clear();
}

//--------------------------------------  destructor.

LineTable::~LineTable() 
{}

void LineTable::lineFinder(TSeries &ts, 
		  float f1, 
		  float f2)
{
  wavearray<double> td;
  td = ts;
  lineFinder(td,f1,f2);
  return;
}

void LineTable::lineFinder(wavearray<double> &td, 
		  float f1, 
		  float f2)
{
  wavearray<double> sp(1);
  wavearray<double> psd(1);
  double* p;

  if(!startTime) startTime = td.start();
  currentTime = td.start();

  size_t i,j, i1,i2;
  double x, y;
  double f3=0., f5;
  size_t n = td.size();
  size_t nn = n/nSub;
  size_t nn2 = nn/2;

  double dfft = td.rate() / nn;
  double dphi=2.*PI/nn;          // needed for Hann window
  double w=sqrt(2./3.);          // needed for Hann window

  if (td.rate() <= 0.) {
    cout <<" LineTable::lineFinder() error: invalid sample rate ="<< td.rate() <<"\n'";
    return;
  }

  strideTime = td.size()/td.rate();
  liveTime += strideTime;

  i1 = (f1 > dfft && f1 < nn2*dfft)? int(f1/dfft) : 1;
  if (f2 <= i1*dfft) i2 = nn2;
  else if (f2 > nn2*dfft) i2 = nn2;
       else i2 = int (f2/dfft);

  sp.resize(nn);
  sp.rate(td.rate());
  psd.resize(nn2);
  psd = 0.;
  p = psd.data;

  for (j=0;  j <= (n - nn) ; j+=nn) {
    sp.cpf(td, nn, j);

// do Hann window

    if(size() != nn){
      resize(nn);
      for (i=0; i < nn; i++) data[i] =w*(1.-cos(i*dphi));
    }

    for (i=0; i < nn; i++) sp.data[i] *= data[i];
    sp.FFT();

// calculate power spectrum from Fourier coefficients excluding f=0;

   for (i = i1; i < i2; i++) {
      x = sp.data[2*i];
      y = sp.data[2*i + 1];
      p[i] += (x*x + y*y);
    }
  }

// find lines

  double sNoise, sLine, snr1, snr3, snr5;
  double delta, d1, d2;
  //  double scale = double(nn)/td.rate()/nSub;
  double scale = 3./nSub;  // Hann normalization (9/3/02)
  
  lineList.clear();
  lineEntry z;
  z.SNR = 0.;
  z.event = 0;
  z.frequency = 0.;
  z.amplitude = 0;

  if(i1<8) i1 = 8;
  if(i2>nn2-8) i2 = nn2-8;

  for(i=i1; i<i2; i++) {

    d1 = p[i-4]+p[i+4]+p[i-5]+p[i+5];
    d2 = p[i-6]+p[i+6]+p[i-7]+p[i+7];
    sNoise = d1+d2;
    sLine  = p[i]+p[i-1]+p[i+1]+p[i-2]+p[i+2];
    f5  = (p[i+1]-p[i-1]+2*p[i+2]-2*p[i-2])/sLine;
    snr5 = 1.6*sLine/sNoise - 1.;

    delta = p[i-2]+p[i+2];
    sNoise += delta-d2;
    sLine  -= p[i-2]+p[i+2];
    f5  = (p[i+1]-p[i-1])/sLine;
    snr3 = 3.*sLine/sNoise - 1.;

    delta = p[i-1]+p[i+1];
    sNoise += delta-d1;
    sLine  -= delta;
    snr1 = 7.*sLine/sNoise - 1.;

    if(snr1 < limSNR && snr3 < limSNR && snr5 < limSNR){
      if(z.event){
	z.width = z.event*dfft;
	z.event = 1;
	lineList.push_back(z);
      }
      z.SNR   = 0.;
      z.event = 0;
    }

    else {
      if(snr1 > z.SNR) { 
	z.SNR = snr1;
	z.frequency = i*dfft;
	z.amplitude = sqrt(sLine*scale);
      }
      if(snr3 > z.SNR) {
	z.SNR = snr3;
	z.frequency = (i+f3)*dfft;
	z.amplitude = sqrt(sLine*scale);
      }
      if(snr5 > z.SNR) {
	z.SNR = snr5;
	z.frequency = (i+f5)*dfft;
	z.amplitude = sqrt(sLine*scale);
      }

      z.event    += 1;

    }
  }  
}

// update Table

void LineTable::update(vector<LineFilter *> &Filter, vector<int> &Index)
{
  size_t i,j;
  lineEntry al;
  wavearray<float> wf;
  lmonList.clear();

  if(Filter.size()){
    for(i=0; i<Filter.size(); i++){
      al.frequency = Filter[i]->Frequency * Filter[i]->nFirst;
      al.width     = Filter[i]->SeedFrequency * Filter[i]->nFirst;

      wf=Filter[i]->getTrend(0,'a');
      al.amplitude = float(wf.mean());

      wf = Filter[i]->getTrend(0,'F');
      wf *= Filter[i]->nFirst;
      for(j=0; j<wf.size(); j++)
	if(wf.data[j]<0.) wf.data[j]*=-1.;
      al.SNR = float(wf.mean());
      al.event = wf.size();
      al.al_m  = false;
      al.index = Index[i];
      lmonList.insert(lmonList.end(),al);
    }
  }
  
// update Line Table 

  if(!lineList.size()) return;

  list<lineEntry>::iterator vT = lineTable.begin();
  list<lineEntry>::iterator vL = lineList.begin();
  list<lineEntry>::iterator vX = lineList.begin();


// update lineTable

  double dF, sW;

  if(lineList.empty()) return;

  for(; vL != lineList.end(); vL++){
    vX = vL;
    vX++;
    if(vX != lineList.end()){
      dF = vX->frequency - vL->frequency;
      sW = (vL->width+vX->width)/2.;
      if(dF < sW){
	vX->frequency -= dF/2.;
	vX->width += (vL->width - vX->width)/2.;
	vX->SNR += (vL->SNR - vX->SNR)/2.;
	vL = vX;
      }
    }

    while(vT != lineTable.end()){
      dF = vT->frequency - vL->frequency;
      sW = (vT->width + vL->width)/2.;
//      cout<<"fT="<<vT->frequency<<" snrT="<<vT->SNR<<" wT="<<vT->width<<endl;
      
      if(dF > sW){                      // insert new entry
	lineTable.insert(vT, *vL);
	vL->event = 0;
 //	cout<<"fi="<<vT->frequency<<" snri="<<vT->SNR<<" wi="<<vT->width<<endl;
	break;
      }

      else if(fabs(dF) < sW){    // update entry
	vT->event++;
	vT->SNR += (vL->SNR - vT->SNR)/double(vT->event);  
	vT->width += (vL->width - vT->width)/double(vT->event);  
	vT->frequency += (vL->frequency - vT->frequency)/double(vT->event);  
	vT->amplitude += (vL->amplitude - vT->amplitude)/double(vT->event);  
	vL->event = 0;
//	cout<<"fa="<<vT->frequency<<" snra="<<vT->SNR<<" wa="<<vT->event<<endl;
	break;
      }

      else vT++;
    }
    
    if(vL->event && vT == lineTable.end()){           // append new entry
      lineTable.push_back(*vL);
      vT = lineTable.end();
    }
  }

// merge entries in lineTable
  
  double ratio = 0.;

  for(vT = lineTable.begin(); vT != lineTable.end(); vT++){
    vX = vT;
    vX++;
    if(vX != lineTable.end()){
      dF = vX->frequency - vT->frequency;
      if(fabs(dF) < (vT->width+vX->width)/2. || fabs(dF)<fftRes){
//      cout<<"fL="<<vT->frequency<<" snrL="<<vT->SNR<<" wL="<<vT->width<<endl;
//      cout<<"fX="<<vX->frequency<<" snrX="<<vX->SNR<<" wX="<<vX->width<<endl;
	vX->event += vT->event;
        ratio = double(vT->event)/double(vX->event);
	vX->frequency -= dF*ratio;
	vX->width = (vT->width - vX->width)*ratio;
	vX->SNR += (vT->SNR - vX->SNR)*ratio;
//	cout<<"before: "<<lineTable.size();
	lineTable.erase(vT);
//	cout<<"   after: "<<lineTable.size()<<endl;
	vT = vX;
      }
    }
  }
  
}


void LineTable::update(vector<alarmEntry *> al_po)
{
    
    size_t y=0,i=0;
    list<lineEntry>::iterator it;
    for (it=lmonList.begin() ; it != lmonList.end() ; it++) 
    {
	it->al_m=false;
	if(y<al_po.size())
	    if(al_po[y]->fid==i)
	    {
		it->al_m=al_po[y]->b;
		y+=1;
	    }
	i+=1;
    }

}
void
LineTable::DumpHTML(vector<char*> &channel, char* ofile, int full) {

//----------------------------------  Build an html document

    html::document doc("LineMonitor statistics");
    char cbuf[256];
    list<lineEntry>::iterator it;
    int row;

    doc.setBackgroundColor(html::color("wheat"));
    LocalStr(Now(), cbuf, "%M %d, %Y %02H:%02N");
    html::text hdr(string("LineMonitor statistics at: ") + cbuf);
    hdr.setSize(html::size(2));
    html::block cb("center");
    cb.addObject(hdr);
    doc.addObject(cb);

    html::table parTbl(string("master channel:  ")+channel[0]);
    parTbl.addColumn(" ");
    parTbl.addColumn(" ");
    row = parTbl.addRow();
    row = parTbl.addRow();
    parTbl.insertData(row, 0, string("start GPS time:  "));
    parTbl.insertData(row, 1, long(startTime));
    row = parTbl.addRow();
    parTbl.insertData(row, 0, string("current GPS time:&nbsp;&nbsp;"));
    parTbl.insertData(row, 1, long(currentTime));
    row = parTbl.addRow();
    parTbl.insertData(row, 0, string("live time, sec:  "));
    parTbl.insertData(row, 1, long(liveTime));

    doc.addObject(parTbl);

//----------------------------------  Add activ lines table.
    int nn, mm;
    double duty = 0.;
 
    html::table lmonTbl("Monitored Lines");
    
    lmonTbl.addColumn("");
    lmonTbl.addColumn("frequency");
    lmonTbl.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;frequency");
    lmonTbl.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;frequency");
    lmonTbl.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;intensity");
    lmonTbl.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;entries");
    lmonTbl.addColumn("");
    lmonTbl.addColumn("");
    
    row = lmonTbl.addRow();
    lmonTbl.insertData(row, 0, "");
    lmonTbl.insertData(row, 1, "&nbsp;&nbsp;seed, Hz");
    lmonTbl.insertData(row, 2, "&nbsp;&nbsp;&nbsp;&nbsp;current, Hz");
    lmonTbl.insertData(row, 3, "&nbsp;&nbsp;&nbsp;&nbsp;average, Hz");
    lmonTbl.insertData(row, 4, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;average");
    lmonTbl.insertData(row, 5, "");
    lmonTbl.insertData(row, 6, "");

    for (it=lmonList.begin() ; it != lmonList.end() ; it++) {
        row = lmonTbl.addRow();

	if(it->al_m)lmonTbl.insertData(row, 0,"|-|/td> |-|tr bgcolor=#FF0026> |-|td>");
	else lmonTbl.insertData(row, 0," ");
	mm = int(it->width);
	nn = int(10000.*(it->width - mm));
	if(!(nn-(nn/10)*10)) nn++;
	duty = mm + double(nn)/double(10000.);
        lmonTbl.insertData(row, 1, duty);

	mm = int(it->frequency);
	nn = int(10000.*(it->frequency - mm));
	if(!(nn-(nn/10)*10)) nn++;
	duty = mm + double(nn)/double(10000.);
        lmonTbl.insertData(row, 2, duty);

	mm = int(it->SNR);
	nn = int(10000.*(it->SNR - mm));
	if(!(nn-(nn/10)*10)) nn++;
	duty = mm + double(nn)/double(10000.);
        lmonTbl.insertData(row, 3, duty);
	
	sprintf(cbuf,"&nbsp;&nbsp;&nbsp;&nbsp;%13.2e",it->amplitude);
        lmonTbl.insertData(row, 4, cbuf);
        lmonTbl.insertData(row, 5, long(it->event));
        if(it->index) lmonTbl.insertData(row, 6, channel[it->index]);
	else lmonTbl.insertData(row, 6,""); 
	
	if(it->al_m)lmonTbl.insertData(row, 7,"|-|/tr>");
	else lmonTbl.insertData(row, 7,"");
    }

    doc.addObject(lmonTbl);


//----------------------------------  Add detected line table.

    if(full){

    html::table lineTbl("Detected Lines");

    lineTbl.addColumn("frequency");
    lineTbl.addColumn("&nbsp;&nbsp;&nbsp;intensity");
    lineTbl.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;width");
    lineTbl.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;SNR");
    lineTbl.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;entries");
    lineTbl.addColumn("&nbsp;&nbsp;duty cycle");

    row = lineTbl.addRow();
    lineTbl.insertData(row, 0, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Hz");
    lineTbl.insertData(row, 1, "&nbsp;&nbsp;&nbsp;1/sqrt(Hz)");
    lineTbl.insertData(row, 2, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Hz");
    lineTbl.insertData(row, 5, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;%");

    for (it=lineTable.begin() ; it != lineTable.end() ; it++) {

      if(liveTime>0) duty = int(9999.*it->event*strideTime/liveTime)/100.;
      else duty = 0;

      if(it->event>4 && (it->SNR>2*limSNR || duty>10.)){
	row = lineTbl.addRow();
	nn = int(100.*it->frequency);
	if(!(nn-(nn/10)*10)) nn++;
	//	duty = double(nn)/100.;
        lineTbl.insertData(row, 0, double(nn)/100.);
	sprintf(cbuf,"&nbsp;&nbsp;&nbsp;&nbsp;%13.2e",it->amplitude);
        lineTbl.insertData(row, 1, cbuf);
	sprintf(cbuf,"&nbsp;&nbsp;&nbsp;<%9.2f",it->width);
        lineTbl.insertData(row, 2, cbuf);
	sprintf(cbuf,"&nbsp;&nbsp&nbsp;%13.2e",it->SNR);
        lineTbl.insertData(row, 3, cbuf);
        lineTbl.insertData(row, 4, long(it->event));
        lineTbl.insertData(row, 5, duty);
      }
    }

    doc.addObject(lineTbl);


    html::table legTbl("Legend");

    legTbl.addColumn("");
    row = legTbl.addRow();
    legTbl.insertData(row, 0, 
     "Seed frequency - line frequency specified in the configuration file.");
    row = legTbl.addRow();
    legTbl.insertData(row, 0, 
     "Current frequency - instantaneous line frequency measured by LineMonitor");
    row = legTbl.addRow();
    legTbl.insertData(row, 0, 
     "Average frequency - average line frequency measured by LineMonitor.");
    row = legTbl.addRow();
    legTbl.insertData(row, 0, 
     "Intensity - sqrt of line power  in adc counts.");
    row = legTbl.addRow();
    legTbl.insertData(row, 0, 
     "SNR - root square of the line and noise PSD ratio");
    row = legTbl.addRow();
    legTbl.insertData(row, 0, 
     "entries - there is one measurement (entry) per stride");
    row = legTbl.addRow();
    legTbl.insertData(row, 0, 
     "width - freqeuncy band where the line PSD exceeds the threshold (-R option)");
    row = legTbl.addRow();
    legTbl.insertData(row, 0, 
     "duty cycle - fraction of time when the line PSD is above the threshold");

    doc.addObject(legTbl);

//----------------------------------  Add observed line table.

    html::table lineTblx("Observed Lines");

    lineTblx.addColumn("frequency");
    lineTblx.addColumn("&nbsp;&nbsp;&nbsp;amplitude");
    lineTblx.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;width");
    lineTblx.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;SNR");
    lineTblx.addColumn("&nbsp;&nbsp;&nbsp;&nbsp;entries");
    lineTblx.addColumn("&nbsp;&nbsp;duty cycle");

    row = lineTblx.addRow();
    lineTblx.insertData(row, 0, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Hz");
    lineTblx.insertData(row, 1, "&nbsp;&nbsp;&nbsp;1/sqrt(Hz)");
    lineTblx.insertData(row, 2, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Hz");
    lineTblx.insertData(row, 5, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;%");

    for (it=lineTable.begin() ; it != lineTable.end() ; it++) {

      if(liveTime>0) duty = int(9999.*it->event*strideTime/liveTime)/100.;
      else duty = 0;

      if(it->event>4 && !(it->SNR>2*limSNR || duty>10.)){
	row = lineTblx.addRow();
	nn = int(100.*it->frequency);
	if(!(nn-(nn/10)*10)) nn++;
	//	duty = double(nn)/100.;
        lineTblx.insertData(row, 0, double(nn)/100.);
	sprintf(cbuf,"&nbsp;&nbsp;&nbsp;&nbsp;%13.2e",it->amplitude);
        lineTblx.insertData(row, 1, cbuf);
	sprintf(cbuf,"&nbsp;&nbsp;&nbsp;<%9.2f",it->width);
        lineTblx.insertData(row, 2, cbuf);
	sprintf(cbuf,"&nbsp;&nbsp&nbsp;%13.2e",it->SNR);
        lineTblx.insertData(row, 3, cbuf);
        lineTblx.insertData(row, 4, long(it->event));
	duty = int(9999.*it->event*strideTime/liveTime)/100.;
        lineTblx.insertData(row, 5, duty);
      }
    }

    doc.addObject(lineTblx);

    }

//----------------------------------  Write the document

    ofstream f(string(ofile).c_str());
    html::writer w(f);
    doc.write(w);
}





