/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_XSIL_HH
#define XSIL_XSIL_HH

#include "xsil/xobj.hh"
#include <list>
#include <string>

class Time;

/**  \ingroup IO
  *  \defgroup IO_xsil Xsil document construction, parsing and manipulation.
  *  \{
  */

/**  Namespace %xsil contains the classes that represent structures within 
  *  an xsil document. Xsil documents can be parsed and built into a tree
  *  structure of xobj-based objects. The xobj tree can then be read, edited 
  *  or written to an xsil xml document.
  *  \brief Xsil document construction and parsing package.
  */
namespace xsil {
    class array;
    class comment;
    class param;
    class table;
    class xsil_time;

    /**  The %XSIL class is the light-weight container for encoded xsil objects.
      *  @memo Xsil document container.
      *  @author John Zweizig
      *  @version 1.0; Last modified March 6, 2008
      */
    class XSIL : public xobj {
    public:
	using xobj::find;
   
	/**  %XSIL default constructor.
	  *  @memo default constructor.
	  */
	XSIL(void);

	/**  %XSIL named structure constructor.
	  *  @memo Named structure constructor.
	  *  @param Name Pointer to structure name string.
	  *  @param Type Pointer to structure type string.
	  */
	XSIL(const char* Name, const char* Type=0);

	/**  %XSIL structure copy constructor.
	  *  @memo Copy constructor.
	  *  @param x Structure definition to be copied.
	  */
	XSIL(const XSIL& x);

	/**  Destructor.
	  *  @memo Destructor.
	  */
	virtual ~XSIL(void);

	/**  Clone a structure.
	  *  @memo Clone a structure.
	  *  @return pointer to a new object.
	  */
	virtual XSIL* Clone(void) const;

	/**  Write XML structure to the specified writer.
	  *  @memo Write XML.
	  *  @param ostr XML output stream object.
	  */
	void Spew(Xwriter& ostr) const;

	/**  Find an object with the specified name and object type inside 
	  *  the container. The first object of the specified type with the 
	  *  requested name is returned. If the type or name are specified 
	  *  as an empty string, any type or name is considered to match. 
	  *  Find returns a null if the specified name isn't found.
	  *  @memo find an object.
	  *  @param nm Name of requested object.
	  *  @param ty Type of requested object.
	  *  @return Constant pointer to the requested object or NULL.
	  */
	virtual const xobj* find(const std::string& nm, 
				 const std::string& ty="") const;

	/**  Find an object with the specified name and object type inside 
	  *  the container. The first object of the specified type with the 
	  *  requested name is returned. If the type or name are specified 
	  *  as an empty string, any type or name is considered to match. 
	  *  Find returns a null if the specified name isn't found.
	  *  @memo find an object.
	  *  @param nm Name of requested object.
	  *  @param ty Type of requested object.
	  *  @return Pointer to the requested object or NULL.
	  */
	virtual xobj* find(const std::string& nm, 
			   const std::string& ty="");

	/**  Perform a recursive search through this object and any objects
	  *  contained in it for the specified named object.
	  *  @memo find an object.
	  *  @param nm Name of requested object.
	  *  @param ty Type of requested object.
	  *  @return Pointer to the requested object or NULL.
	  */
	const xobj* findr(const std::string& nm, 
			  const std::string& ty="") const;

	/**  Perform a recursive search through this object and any objects
	  *  contained in it for the specified named object.
	  *  @memo find an object.
	  *  @param nm Name of requested object.
	  *  @param ty Type of requested object.
	  *  @return Pointer to the requested object or NULL.
	  */
	xobj* findr(const std::string& nm, 
		    const std::string& ty="");

	/**  Clone a generic object and add the clone to the object list.
	  *  @memo Add an object.
	  *  @param x object to be cloned into the %XSIL structure.
	  *  @return Pointer to the added structure definition.
	  */
	xobj* addObject(const xobj& x);

	/**  Add an existing generic object to the object list.
	  *  @memo Add an object.
	  *  @param x object to be added to the %XSIL structure.
	  *  @return Pointer to the added structure definition.
	  */
	xobj* addObject(xobj* x);

	/**  Add a named %XSIL substructure.
	  *  @memo Add an %XSIL substructure.
	  *  @param Name Pointer to structure name string.
	  *  @param Type Pointer to structure type string.
	  *  @return Pointer to the added structure definition.
	  */
	virtual XSIL* addXSIL(const char* Name, const char* Type=0);

	/**  Add a named %XSIL array to the %XSIL structure.
	  *  @memo Add an array.
	  *  @param Name Pointer to structure name string.
	  *  @param Type Pointer to structure type string.
	  *  @param Unit Pointer to structure units string.
	  *  @return Pointer to the added array definition.
	  */
	array* addArray(const char* Name, const char* Type=0, const char* Unit=0);

	/**  Add a comment to the %XSIL structure.
	  *  @memo Add a comment.
	  *  @param Text Pointer to the comment text string.
	  *  @return Pointer to the added comment definition.
	  */
	comment* addComment(const char* Text=0);

	/**  Add a parameter to the %XSIL structure.
	  *  @memo Add a parameter.
	  *  @param Name  Pointer to parameter name string.
	  *  @param Type  Pointer to parameter type string.
	  *  @param Unit  Pointer to parameter units string.
	  *  @param Value Pointer to parameter value string.
	  *  @return Pointer to the added parameter definition.
	  */
	param* addParam(const char* Name, const char* Type=0, const char* Unit=0, 
			const char* Value=0);

	/**  Add a double float parameter to the %XSIL structure.
	  *  @memo Add a double float parameter.
	  *  @param Name  Pointer to parameter name string.
	  *  @param Unit  Pointer to parameter units string.
	  *  @param Value Double float parameter value.
	  *  @return Pointer to the added parameter definition.
	  */
	param* addParam(const char* Name, const char* Unit, double Value);

	/**  Add an integer parameter to the %XSIL structure.
	  *  @memo Add an integer parameter.
	  *  @param Name  Pointer to parameter name string.
	  *  @param Unit  Pointer to parameter units string.
	  *  @param Value Integer parameter value.
	  *  @return Pointer to the added parameter definition.
	  */
	param* addParam(const char* Name, const char* Unit, int Value);

	/**  Add a table to the %XSIL structure.
	  *  @memo Add a table.
	  *  @param Name Pointer to the table name string.
	  *  @param Type Pointer to the table type string.
	  *  @return Pointer to the added table definition.
	  */
	table* addTable(const char* Name, const char* Type=0);

	/**  Add an xsil_time to the %XSIL structure.
	  *  @memo Add a xsil_time.
	  *  @param Name  Pointer to the xsil_time name string.
	  *  @param Type  Pointer to the xsil_time type string.
	  *  @param Value Pointer to the xsil_time value string.
	  *  @return Pointer to the added xsil_time parameter definition.
	  */
	xsil_time* addTime(const char* Name, const char* Type=0, 
			   const char* Value=0);

	/**  Add an xsil_time to the %XSIL structure.
	  *  @memo Add a xsil_time.
	  *  @param Name Pointer to the xsil_time name string.
	  *  @param t Xsil_Time value;
	  *  @return Pointer to the added xsil_time parameter definition.
	  */
	xsil_time* addTime(const char* Name, const Time& t);

	/**  Test whether the object is a container for object interface.
	  *  \brief Test for container
	  *  \return Always true - %XSIL is a container.
	  */
	bool container(void) const;

	/**  Remove an object from the %XSIL structure.
	  *  @memo Remove an object.
	  *  @param x Pointer to the object to be removed from the %XSIL 
	  *           structure.
	  */
	void erase(const xobj* x);

	/**  Return the object type.
	  *  @memo Object type name.
	  *  @return Pointer to the object name string.
	  */
	virtual const char* getObjType(void) const;

    private:
	/**  List of objects contained in the %XSIL structure.
	  */
	typedef std::list<xobj*> ObjList;
	typedef ObjList::iterator objlist_iterator;
	typedef ObjList::const_iterator const_objlist_iterator;
	ObjList mList;
    };
}

/** \}
  */

#endif //  XSIL_XSIL_HH
