/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_METATABLE_HH
#define XSIL_METATABLE_HH

#include "xsil/table.hh"
#include "xsil/stream.hh"
#include <string>
#include <list>

namespace xsil {

    /**  Specialization of xsil table class that incorporates automatic 
      *  filling and reference capabilities. The user defines the %MetaTable
      *  column structure by calling a defineColumn method for each data 
      *  column. The second argument of defineColumn defines the data type
      *  and the address of the mail-box variable that will contain the
      *  appropriate data each time a new row is added with via the putRow
      *  method.
      *  @brief Xsil table with automatic filling.
      *  @ingroup IO_xsil
      *  @author John Zweizig
      *  @version 1.1; Last modified March 5, 2008
      */
    class MetaTable : public table {
      
	//------------------------------  Class to hold data pointers
	/**  Class to hold pointer to the data location from which the 
	  *  table will be filled.
	  *  @brief Data pointer class.
	  *  @ingroup IO_xsil
	  */
	 class eventcolumn {
	 public:
	     /** Enumeration of the data types accepted by the %MetaTable class.
	       */
	     enum d_type {
		 t_none,
		 t_chars,
		 t_charu,
		 t_string,
		 t_int4s,
		 t_longs,
		 t_real4,
		 t_real8,
		 t_cite
	     };

	     /**  Construct a null data pointer.
	       *  @brief Default constructor.
	       */
	     eventcolumn(void);

	     /**  Specify the data type, address and optional parameter for 
	       *  one table column.
	       *  @brief Construct a table column data pointer.
	       *  @param t Column data type.
	       *  @param a Address of the data variable.
	       *  @param p Parameter used by citations only.
	       */
	     eventcolumn(d_type t, const void* a, const char* p=0);

	     /**  Destroy an eventcolumn.
	       *  @brief Destructor.
	       */
	     ~eventcolumn(void);

	     ///  Column data type
	     d_type data_Type;
	     ///  Pointer to variable that will contain data
	     const void* data_Addr;
	     ///  Column parameter (table name for 
	     std::string data_Param;
	 };

	//----------------------------------  Column list data type
    typedef std::list<eventcolumn> ColumnList;
    typedef ColumnList::const_iterator col_iter;

  public:

    /**  Construct an empty Meta-table.
      *  \brief Empty table constructor.
      */
    MetaTable(void);

    /**  Construct an empty named ligolw metadata table.
      *  \brief Named meta-table constructor.
      *  \param Name %MetaTable name attribute.
      *  \param Type %MetaTable type attribute.
      */
    MetaTable(const char* Name, const char* Type=0);

    /**  Construct a copy of the argument %MetaTable.
      *  \brief Copy constructor.
      *  \param x %MetaTable to be copied.
      */
    MetaTable(const MetaTable& x);

    /**  Create an identical copy of the meta-table and return a pointer to
      *  the copy.
      *  @memo Clone a Meta-table.
      */
    // MetaTable* Clone(void) const;

    /**  Destroy a meta-table.
      *  @memo Meta-table destructor.
      */
    virtual ~MetaTable(void);

    /**  Check the %MetaTable data addresses. If \a csize is non-zero it is 
      *  assumed to be the length of a %MetaTable based object containing 
      *  all the data items. In this case, a test is made that the data 
      *  addresses are inside the structure. An error message is printed to 
      *  \c stdout if an address is outside the structure. If \a csize is  
      *  zero, all data entries and addresses are printed to \c stdout.
      *  \brief Check %MetaTable pointers.
      *  \param title Table name.
      *  \param csize Structure length or zero.
      */
    void check(const char* title, int csize=0) const;

    /**  Clear all table column data.
      *  @memo Clear %MetaTable.
      */
    void clear(void);

    /**  Specify a table column. The column name and type are included.
      *  An address field specifies a static address from which the data 
      *  will be copied.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Addr Pointer to the data field.
      */
    void defineColumn(const char* Name, const int* Addr);

    /**  Specify a table column. The column name and type are included.
      *  An address field specifies a static address from which the data 
      *  will be copied.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Addr Pointer to the data field.
      */
     void defineColumn(const char* Name, const long* Addr);

    /**  Specify a table column. The column name and type are included.
      *  An address field specifies a static address from which the data 
      *  will be copied.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Addr Pointer to the data field.
      */
     void defineColumn(const char* Name, const float* Addr);

    /**  Specify a table column. The column name and type are included.
      *  An address field specifies a static address from which the data 
      *  will be copied.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Addr Pointer to the data field.
      */
     void defineColumn(const char* Name, const double* Addr);

    /**  Specify a table column. The column name and type are included.
      *  An address field specifies a static address from which the data 
      *  will be copied.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Addr Pointer to the data field.
      *  @param ilwc Data type "ilwd:char" (true) or lstring (false).
      */
     void defineColumn(const char* Name, const char* Addr,
		      bool ilwc=false);

    /**  Specify a table column. The column name and type are included.
      *  An address field specifies a static address from which the data 
      *  will be copied.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Addr Pointer to the data field.
      *  @param ilwc Data type "ilwd:char" (true) or lstring (false).
      */
     void defineColumn(const char* Name, const std::string* Addr, 
		      bool ilwc=false);

    /**  Specify a table column. The column name and type are included.
      *  An address field specifies a static address from which the data 
      *  will be copied.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Addr Pointer to the data field.
      */
     void defineColumn(const char* Name, const UCVec* Addr);

    /**  Specify a table reference column.
      *  @memo Add a column to the table.
      *  @param Name Column name attribute field.
      *  @param Addr Pointer to the table to be referenced.
      *  @param Col Name of column to be referenced.
      */
     void defineColumn(const char* Name, const MetaTable* Addr,
		       const char* Col=0);

    /**  Get the number of data columns that have already been defined.
      *  @memo Get the number of columns.
      *  @return Current number of columns.
      */
    int getNColumn(void) const;

    /**  Get the number of data rows that have already been defined.
      *  \brief Get the number of rows.
      *  \return Number of rows defined in table.
      */
    int getNRow(void) const;

    /**  Copy all specified data fields (one for each table column) into the 
      *  data stream.
      *  @memo Copy an event into the data stream.
      */
    void putRow(void);

    /**  Delete the contents of the table data stream.
      *  @memo Reset the data stream.
      */
    void resetStream(void);

    /**  Return a reference string to the last row of the table. The 
      *  citation is of the form \<table\>:\<column\>:\<row\>.
      *  \brief Get a reference to the last row of the table.
      *  \param col Column name to be referenced
      *  \return Reference string.
      */
    std::string citeTable(const std::string& col) const;

    /**  Return a reference string to the specified row of the table. The 
      *  citation is of the form \<table\>:\<column\>:\<row\>. No validity
      *  check is performed on either the column name or the row number.
      *  \brief Get a reference to the last row of the table.
      *  \param col Column name to be referenced
      *  \param row number to be referenced.
      *  \return Reference string.
      */
    std::string citeTable(const std::string& col, int row) const;

  private:
    std::string getTableName(void) const;
    std::string getItemName(const std::string& x) const;

  private:
    std::string mGroup;
    ColumnList  mCList;
    int         mRow;
  };

  //====================================  Inline row and column counts.
  inline int
  MetaTable::getNRow(void) const {
      return mRow;
  }

  inline int
  MetaTable::getNColumn(void) const {
      return mCList.size();
  }

}  // namespace xsil

#endif  //  XSIL_METATABLE_HH
