#include "PConfig.h"
#include <signal.h>
#include <cstdio>
#include <cstdlib>
#include <cerrno>
#include <cstring>
#include <string>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <setjmp.h>
#include <iostream>
#include <sstream>
#include <algorithm>
#include "web/webserver.hh"
#include "web/webclient.hh"
#include "goption.hh"
#include "fdstream.hh"
#include "Time.hh"
#include "Interval.hh"
#include "sockutil.h"
#ifdef HAVE_ZUTIL
#include "zlib.h"
#include "zutil.h"
#endif
#ifndef SHUT_RDWR
#define SHUT_RDWR 2
#endif

#define HTML_MAXIMUM_REQUEST_SIZE 4096

//======================================  Log file time stamp format
#define TIMESTAMP_FMT "%Y %3M %02d %02H:%02N:%02S%Z: "

namespace web {
   using namespace std;

   const char* const help_text = 
   "usage: program [-p 'port']";

   const double TIMEOUT = 60.0;
   const int kGZIPlevel = 9;
   const int gz_magic[2] = {0x1f, 0x8b}; /* gzip magic header */



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// utility functions                                                    //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   static string trim (const char* p)
   {
      while (isspace (*p)) p++;
      string s = p;
      while ((s.size() > 0) && isspace (s[s.size()-1])) {
         s.erase (s.size() - 1);
      }
      return s;
   }

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// request_t::header_sort                                               //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool request_t::header_sort::operator () (const std::string& s1, 
                     const std::string& s2) const 
   {
      return strcasecmp (s1.c_str(), s2.c_str()) < 0; 
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// request_t::browser_t                                                 //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   request_t::browser_t::browser_t (const char* p)
   {
      // Determine browser/OS
      // http://www.mozilla.org/docs/web-developer/sniffer/browser_type.html
      memset (this, 0, sizeof (browser_t));
      string agt = p ? p : "";
      if (agt.empty()) {
         return;
      }
      // convert all characters to lowercase to simplify testing
      for (string::iterator i = agt.begin(); i != agt.end(); ++i) {
         *i = tolower (*i);
      }
      // *** BROWSER VERSION ***
      // Note: On IE5, these return 4, so use fIsIe5up to detect IE5.
      string::size_type pos = agt.find_first_of ("0123456789");
      if (pos != string::npos) {
         sscanf (agt.c_str() + pos, "%i", &fMajor);
         sscanf (agt.c_str() + pos, "%f", &fMinor);
      }
      // Note: Opera and WebTV spoof Navigator.  We do strict client detection.
      // If you want to allow spoofing, take out the tests for opera and webtv.
      fIsNav  = ((agt.find("mozilla")!=string::npos) && 
                (agt.find("spoofer")==string::npos)
                && (agt.find("compatible")==string::npos) && 
                (agt.find("opera")==string::npos)
                && (agt.find("webtv")==string::npos) && 
                (agt.find("hotjava")==string::npos));
      fIsNav2 = (fIsNav && (fMajor == 2));
      fIsNav3 = (fIsNav && (fMajor == 3));
      fIsNav4 = (fIsNav && (fMajor == 4));
      fIsNav4up = (fIsNav && (fMajor >= 4));
      fIsNavOnly = (fIsNav && ((agt.find(";nav")!=string::npos) ||
                              (agt.find("; nav")!=string::npos)) );
      fIsNav6 = (fIsNav && (fMajor == 5));
      fIsNav6up = (fIsNav && (fMajor >= 5));
      fIsGecko = (agt.find("gecko") != string::npos);
   
      // Internet Explorer
      fIsIe     = ((agt.find("msie")!=string::npos) && 
                  (agt.find("opera") == string::npos));
      fIsIe3    = (fIsIe && (fMajor < 4));
      fIsIe4    = (fIsIe && (fMajor == 4) &&
                  (agt.find("msie 4")!=string::npos) );
      fIsIe4up  = (fIsIe && (fMajor >= 4));
      fIsIe5    = (fIsIe && (fMajor == 4) &&
                  (agt.find("msie 5.0")!=string::npos) );
      fIsIe5_5  = (fIsIe && (fMajor == 4) &&
                  (agt.find("msie 5.5") !=string::npos));
      fIsIe5up  = (fIsIe && !fIsIe3 && !fIsIe4);
      fIsIe5_5up =(fIsIe && !fIsIe3 && !fIsIe4 && !fIsIe5);
      fIsIe6    = (fIsIe && (fMajor == 4) && 
                  (agt.find("msie 6.")!=string::npos) );
      fIsIe6up  = (fIsIe && !fIsIe3 && !fIsIe4 && !fIsIe5 && !fIsIe5_5);
   
      // KNOWN BUG: On AOL4, returns false if IE3 is embedded browser
      // or if this is the first browser window opened.  Thus the
      // variables fIsAol, fIsAol3, and fIsAol4 aren't 100% reliable.
      fIsAol   = (agt.find("aol") != string::npos);
      fIsAol3  = (fIsAol && fIsIe3);
      fIsAol4  = (fIsAol && fIsIe4);
      fIsAol5  = (agt.find("aol 5") != string::npos);
      fIsAol6  = (agt.find("aol 6") != string::npos);
   
      fIsOpera = (agt.find("opera") != string::npos);
      fIsOpera2 = (agt.find("opera 2") != string::npos || 
                  agt.find("opera/2") != string::npos);
      fIsOpera3 = (agt.find("opera 3") != string::npos || 
                  agt.find("opera/3") != string::npos);
      fIsOpera4 = (agt.find("opera 4") != string::npos || 
                  agt.find("opera/4") != string::npos);
      fIsOpera5 = (agt.find("opera 5") != string::npos || 
                  agt.find("opera/5") != string::npos);
      fIsOpera5up = (fIsOpera && !fIsOpera2 && !fIsOpera3 && !fIsOpera4);
   
      fIsWebtv = (agt.find("webtv") != string::npos);
      fIsTVNavigator = ((agt.find("navio") != string::npos) || 
                       (agt.find("navio_aoltv") != string::npos)); 
      fIsAOLTV = fIsTVNavigator;
   
      fIsHotjava = (agt.find("hotjava") != string::npos);
      fIsHotjava3 = (fIsHotjava && (fMajor == 3));
      fIsHotjava3up = (fIsHotjava && (fMajor >= 3));
   
      // *** JAVASCRIPT VERSION CHECK ***
      if (fIsNav2 || fIsIe3) fJsVersion = 1.0;
      else if (fIsNav3) fJsVersion = 1.1;
      else if (fIsOpera5up) fJsVersion = 1.3;
      else if (fIsOpera) fJsVersion = 1.1;
      else if ((fIsNav4 && (fMinor <= 4.05)) || fIsIe4) fJsVersion = 1.2;
      else if ((fIsNav4 && (fMinor > 4.05)) || fIsIe5) fJsVersion = 1.3;
      else if (fIsHotjava3up) fJsVersion = 1.4;
      else if (fIsNav6 || fIsGecko) fJsVersion = 1.5;
      // NOTE: In the future, update this code when newer versions of JS
      // are released. For now, we try to provide some upward compatibility
      // so that future versions of Nav and IE will show they are at
      // *least* JS 1.x capable. Always check for JS version compatibility
      // with > or >=.
      else if (fIsNav6up) fJsVersion = 1.5;
      // NOTE: ie5up on mac is 1.4
      else if (fIsIe5up) fJsVersion = 1.3;
      // HACK: no idea for other browsers; always check for JS version with > or >=
      else fJsVersion = 0.0;
   
      // *** PLATFORM ***
      fIsWin   = ( (agt.find("win")!=string::npos) || 
                 (agt.find("16bit")!=string::npos) );
      // NOTE: On Opera 3.0, the userAgent string includes "Windows 95/NT4" on all
      //        Win32, so you can't distinguish between Win95 and WinNT.
      fIsWin95 = ((agt.find("win95")!=string::npos) || 
                 (agt.find("windows 95")!=string::npos));
   
      // is this a 16 bit compiled version?
      fIsWin16 = ((agt.find("win16")!=string::npos) || 
                 (agt.find("16bit")!=string::npos) || 
                 (agt.find("windows 3.1")!=string::npos) || 
                 (agt.find("windows 16-bit")!=string::npos) );  
   
      fIsWin31 = ((agt.find("windows 3.1")!=string::npos) ||
                 (agt.find("win16")!=string::npos) ||
                 (agt.find("windows 16-bit")!=string::npos));
   
      fIsWinme = ((agt.find("win 9x 4.90")!=string::npos));
      fIsWin2k = ((agt.find("windows nt 5.0")!=string::npos));
   
      // NOTE: Reliable detection of Win98 may not be possible. It appears that:
      //       - On Nav 4.x and before you'll get plain "Windows" in userAgent.
      //       - On Mercury client, the 32-bit version will return "Win98", but
      //         the 16-bit version running on Win98 will still return "Win95".
      fIsWin98 = ((agt.find("win98")!=string::npos) ||
                 (agt.find("windows 98")!=string::npos));
      fIsWinnt = ((agt.find("winnt")!=string::npos) ||
                 (agt.find("windows nt")!=string::npos));
      fIsWin32 = (fIsWin95 || fIsWinnt || fIsWin98 || 
                 (agt.find("win32")!=string::npos) ||
                 (agt.find("32bit")!=string::npos));
   
      fIsOs2   = ((agt.find("os/2")!=string::npos) || 
                 (agt.find("ibm-webexplorer")!=string::npos));
   
      fIsMac    = (agt.find("mac")!=string::npos);
      // hack ie5 js version for mac
      if (fIsMac && fIsIe5up) fJsVersion = 1.4;
      fIsMac68k = (fIsMac && ((agt.find("68k")!=string::npos) || 
                             (agt.find("68000")!=string::npos)));
      fIsMacppc = (fIsMac && ((agt.find("ppc")!=string::npos) || 
                             (agt.find("powerpc")!=string::npos)));
   
      fIsSun   = (agt.find("sunos")!=string::npos);
      fIsSun4  = (agt.find("sunos 4")!=string::npos);
      fIsSun5  = (agt.find("sunos 5")!=string::npos);
      fIsSuni86= (fIsSun && (agt.find("i86")!=string::npos));
      fIsIrix  = (agt.find("irix") !=string::npos);    // SGI
      fIsIrix5 = (agt.find("irix 5") !=string::npos);
      fIsIrix6 = ((agt.find("irix 6") !=string::npos) || 
                 (agt.find("irix6") !=string::npos));
      fIsHpux  = (agt.find("hp-ux")!=string::npos);
      fIsHpux9 = (fIsHpux && (agt.find("09.")!=string::npos));
      fIsHpux10= (fIsHpux && (agt.find("10.")!=string::npos));
      fIsAix   = (agt.find("aix") !=string::npos);      // IBM
      fIsAix1  = (agt.find("aix 1") !=string::npos);    
      fIsAix2  = (agt.find("aix 2") !=string::npos);    
      fIsAix3  = (agt.find("aix 3") !=string::npos);    
      fIsAix4  = (agt.find("aix 4") !=string::npos);    
      fIsLinux = (agt.find("inux")!=string::npos);
      fIsSco   = (agt.find("sco")!=string::npos) ||
         (agt.find("unix_sv")!=string::npos);
      fIsUnixware = (agt.find("unix_system_v")!=string::npos); 
      fIsMpras    = (agt.find("ncr")!=string::npos); 
      fIsReliant  = (agt.find("reliantunix")!=string::npos);
      fIsDec   = ((agt.find("dec")!=string::npos) || 
                 (agt.find("osf1")!=string::npos) || 
                 (agt.find("dec_alpha")!=string::npos) ||
                 (agt.find("alphaserver")!=string::npos) || 
                 (agt.find("ultrix")!=string::npos) ||
                 (agt.find("alphastation")!=string::npos)); 
      fIsSinix = (agt.find("sinix")!=string::npos);
      fIsFreebsd = (agt.find("freebsd")!=string::npos);
      fIsBsd = (agt.find("bsd")!=string::npos);
      fIsUnix  = ((agt.find("x11")!=string::npos) || 
                 fIsSun || fIsIrix || fIsHpux || fIsSco ||fIsUnixware || 
                 fIsMpras || fIsReliant || fIsDec || fIsSinix || fIsAix || 
                 fIsLinux || fIsBsd || fIsFreebsd);
   
      fIsVms   = ((agt.find("vax")!=string::npos) || 
                 (agt.find("openvms")!=string::npos));
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// request_t                                                            //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool request_t::read (int fd)
   {
      fType = rInvalid;
      fUrl = "";
      fHttpVersion = 10;
   
      // get some data
      std::string line;
      bool emptyline = false;
      bool firstline = true;
      Time t0 = Now();
      do {
         // check timeout
         Time t1 = Now();
         if (t1 - t0 > Interval (TIMEOUT)) {
            cerr << "Timeout during read" << endl;
            return false;
         }
         char buf[HTML_MAXIMUM_REQUEST_SIZE];
         memset (buf, 0, sizeof (buf));
         // set descriptor to non-blocking
         int flags;
         if ((flags = fcntl (fd, F_GETFL, 0)) == -1) {
            perror("webserver: Unable to get fd flags");
            return false;
         }
         if (fcntl (fd, F_SETFL, flags | O_NONBLOCK) == -1) {
            perror("webserver: Unable to set fd flags");
            return false;
         }
         // read as many characters as possible
         int n = ::read (fd, buf, sizeof (buf));
         // set descriptor back to its original state
         if (fcntl (fd, F_SETFL, flags) == -1) {
            perror("webserver: Unable to set fd flags");
            return false;
         }
         // continue if nothing was read
         if ((n == 0) || ((n == -1) && (errno == EAGAIN))) {
            const timespec tick = {0, 500000000};
            nanosleep (&tick, 0);
            continue;
         }
         if (n < 0) {
            perror("webserver: Request read error");
            return false;
         }
         // cout << endl << "++++++++++++++++++++++++++++++++++" << endl;
         // cout << "READ BUFFER = ";
         // cout.write (buf, n);
         // cout << endl << "++++++++++++++++++++++++++++++++++" << endl;
         // analyze return string
         line += string (buf, 0, n);
         do {
            string l;
            string::size_type pos = line.find ("\r\n");
            if (pos != string::npos) {
               l = trim (line.substr (0, pos).c_str());
               if (!firstline) emptyline = l.empty();
               line.erase (0, pos + 2);
            } 
            else {
               return false;
            }
            if (!l.empty()) {
               /// Request line
               if (firstline) {
                  // parse request
                  string::size_type pos = l.find ("HTTP/1.");
                  if (pos != string::npos) {
                     if (isdigit (l[pos+7])) {
                        fHttpVersion = 10 + (l[pos+7] - '0');
                     }
                     l.erase (pos);
                  }
                  if (l.find ("GET") == 0) {
                     fType = rGet;
                     l.erase (0, 3);
                  }
                  else if (l.find ("POST") == 0) {
                     fType = rPost;
                     l.erase (0, 4);
                  }
                  else if (l.find ("HEAD") == 0) {
                     fType = rHead;
                     l.erase (0, 4);
                  }
                  else if (l.find ("OPTIONS") == 0) {
                     fType = rOptions;
                     l.erase (0, 7);
                  }
                  else if (l.find ("PUT") == 0) {
                     fType = rPut;
                     l.erase (0, 3);
                  }
                  else if (l.find ("DELETE") == 0) {
                     fType = rDelete;
                     l.erase (0, 6);
                  }
                  else if (l.find ("TRACE") == 0) {
                     fType = rTrace;
                     l.erase (0, 5);
                  }
                  else if (l.find ("CONNECT") == 0) {
                     fType = rConnect;
                     l.erase (0, 7);
                  }
                  else {
                     fType = rInvalid;
                     return false;
                  }
                  fUrl = trim (l.c_str());
               }
               // header information
               else {
                  string::size_type pos = l.find (":");
                  if (pos != string::npos) {
                     string name (l.begin(), l.begin() + pos);
                     l.erase (0, pos + 1);
                     string token = trim (l.c_str());
                     if (!name.empty()) {
                        fHeader[name] = token;
                     }
                  }
               }
               firstline = false;
            } 
         } while (!line.empty() && !emptyline);
      } while (!emptyline);
   
      // read post data here
      if (fType == rPost) {
         header_type::iterator contlen = fHeader.find ("Content-length");
         int len = 0;
         if (contlen != fHeader.end()) {
            len = atoi (contlen->second.c_str());
         }
         while ((int)line.size() < len) {
            // check timeout
            Time t1 = Now();
            if (t1 - t0 > Interval (TIMEOUT)) {
               cerr << "Timeout during read" << endl;
               return false;
            }
            char buf[HTML_MAXIMUM_REQUEST_SIZE];
            memset (buf, 0, sizeof (buf));
            // set descriptor to non-blocking
            int flags;
            if ((flags = fcntl (fd, F_GETFL, 0)) == -1) {
               perror("webserver: Unable to get fd flags");
               return false;
            }
            if (fcntl (fd, F_SETFL, flags | O_NONBLOCK) == -1) {
               perror("webserver: Unable to set fd flags");
               return false;
            }
            // read as many characters as possible
            int n = ::read (fd, buf, sizeof (buf));
            // set descriptor back to its original state
            if (fcntl (fd, F_SETFL, flags) == -1) {
               perror("webserver: Unable to set fd flags");
               return false;
            }
            // continue if nothing was read
            if ((n == 0) || ((n == -1) && (errno == EAGAIN))) {
               const timespec tick = {0, 500000000};
               nanosleep (&tick, 0);
               continue;
            }
            if (n < 0) {
               perror("webserver: Post data read error");
               return false;
            }
            // add to buffer
            line += string (buf, 0, n);
         }
         Reserve (len > 0 ? len + 1 : 0);
         if (len > 0) {
            memcpy (fData, line.data(), len);
            fData[len] = 0; // make sure it is NULL terminated
         }
      }
      // Browser 
      header_type::iterator i = fHeader.find ("User-Agent");
      string user_agent;
      if (i != fHeader.end()) {
         user_agent = i->second;
      }
      fBrowser = browser_t (user_agent.c_str());
      return true;
   }

//______________________________________________________________________________
   std::string request_t::GetDemangledUrl() const
   {
      return http_request::demangle (fUrl);
   }

#ifdef HAVE_ZUTIL
//////////////////////////////////////////////////////////////////////////
//                                                                      //
// compress_gzip  (similar to compress but writes a gzip 'file'         //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   int ZEXPORT compress_gzip (Bytef *dest, uLongf *destLen, 
                     const Bytef *source, uLong sourceLen, int level) 
   {
      // write gzip header
      sprintf((char*)dest, "%c%c%c%c%c%c%c%c%c%c", 
             gz_magic[0], gz_magic[1], Z_DEFLATED, 
             0 /*flags*/, 0,0,0,0 /*time*/, 0 /*xflags*/, OS_CODE);
      // normal deflate
      z_stream stream;
      int err;
      stream.next_in = (Bytef*)source;
      stream.avail_in = (uInt)sourceLen;
      stream.next_out = dest + 10L; // after header
      stream.avail_out = (uInt)*destLen - 10L;
      if ((uLong)stream.avail_out != *destLen - 10L) 
         return Z_BUF_ERROR;
   
      stream.zalloc = (alloc_func)0;
      stream.zfree = (free_func)0;
      stream.opaque = (voidpf)0;
   
      err = deflateInit2 (&stream, level, Z_DEFLATED, -MAX_WBITS, 
         DEF_MEM_LEVEL, Z_DEFAULT_STRATEGY);
      if (err != Z_OK) 
         return err;
   
      err = deflate (&stream, Z_FINISH);
      if (err != Z_STREAM_END) {
         deflateEnd(&stream);
         return err == Z_OK ? Z_BUF_ERROR : err;
      }
      *destLen = stream.total_out + 10L;
      // write CRC and Length
      uLong crc = crc32 (0L, source, sourceLen);
      for (int n = 0; n < 4; ++n, ++*destLen) {
         dest[*destLen] = (int)(crc & 0xff);
         crc >>= 8;
      }
      uLong len = stream.total_in;
      for (int n = 0; n < 4; ++n, ++*destLen) {
         dest[*destLen] = (int)(len & 0xff);
         len >>= 8;
      }
      err = deflateEnd (&stream);
      return err;
   }
#endif

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// response_t                                                           //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   bool response_t::TryCompress()
   {
#ifdef HAVE_ZUTIL
      // check compression
      if (!fCanCompress || !fData || (fLen <= 0)) {
         return false;
      }
      unsigned long newlen = 
         (unsigned long) (fLen * 1.001 + 32 + 18);
      unsigned long alloclen = newlen;
      char* newdata = new char [newlen];
      // int ret = ::compress2((unsigned char*)newdata, &newlen, 
                           // (unsigned char*)fData, fLen, kGZIPlevel);
      int ret = compress_gzip ((unsigned char*)newdata, &newlen, 
                           (unsigned char*)fData, fLen, kGZIPlevel);
      if (ret == Z_OK) {
         if (newlen > alloclen) {
            cerr << "#############################################" << endl;
            cerr << "Allocation error in gzip!!!" << endl;
         }
         // cout << "Size = " << fLen << " NewSize = " << newlen << endl;
         // cout << "Compression ratio = " << (float)newlen/fLen << endl;
         delete [] fData;
         fData = newdata;
         fLen = newlen;
         AddHeader ("Content-Encoding", "gzip");
      }
      else {
         delete [] newdata;
      }
      return (ret == Z_OK);
#else
      return false;
#endif
   }

//______________________________________________________________________________
   bool response_t::write (int fd)
   {
      cout << "Response = " << fStatusLine << endl;
      // write status line
      if (::write (fd, fStatusLine.c_str(), fStatusLine.size()) < 0) {
         return false;
      }
      if (::write (fd, "\r\n", 2) < 0) {
         return false;
      }
      // Header
      for (header_type::iterator i = fHeader.begin(); 
          i != fHeader.end(); ++i) {
         string line = i->first + ": " + i->second + "\r\n";
         // cout << "Header = " << line;
         if (::write (fd, line.c_str(), line.size()) < 0) {
            return false;
         }
      }
      // Empty line
      if (::write (fd, "\r\n", 2) < 0) {
         return false;
      }
      // Body
      if (fData) {
         if (::write (fd, fData, fLen) < 0) {
            return false;
         }
      }
      return true;
   }

//______________________________________________________________________________
   void response_t::RemoveHeader (const std::string& name){
      for (header_type::iterator i = fHeader.begin(); i != fHeader.end(); ) {
         if (strcasecmp (i->first.c_str(), name.c_str()) == 0) {
            i = fHeader.erase (i);
         }
         else {
            ++i;
         }
      }
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// connect_control_C                                                    //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
extern "C"
   void* webserver_control_C (void* f) 
   {
      // set signal mask to include Ctrl-C */
      sigset_t set;
      if ((sigemptyset (&set) != 0) ||
         (sigaddset (&set, SIGTERM) != 0) ||
         (sigaddset (&set, SIGINT) != 0)) {
         cerr << "Unable to connect Ctrl-C" << endl;
         return 0;
      }
      while (1) {
         // wait for ctrl-C signal
         int sig;
         sigwait (&set, &sig);
         ((basic_server*)f)->Interrupt (sig);
      }
      return 0;
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// trap handler                                                        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   static pthread_key_t sigjmp_key;
   static bool sigjmp_err = false;
   static int sigjmp_retry = 0;

//______________________________________________________________________________
extern "C" {
   void webserver_create_sigjmp (void) 
   {
      if (pthread_key_create (&sigjmp_key, 0) != 0) {
         sigjmp_err = true;
      }
   }

//______________________________________________________________________________
   void webserver_trap_handler (int sig)
   {
      // count traps; make sure they are not more than allowed
      // if yes, we are probably in a infinite loop, so kill it
      static int trap_count = 0;
      sigjmp_buf* buf = 0;
      if (++trap_count > sigjmp_retry + 10) {
         ::exit(1);
      }
   
      // Normal trap porcessing
      fprintf (stderr, "Trapped signal %i (retry count is %i)\n", 
              sig, sigjmp_retry);
      buf = (sigjmp_buf*)pthread_getspecific (sigjmp_key);
      if (!buf) {
         exit (1);
      }
   
      if (sigjmp_retry > 0) {
         sigjmp_retry--;
         siglongjmp (*buf, 1); // continue
      }
      else {
         siglongjmp (*buf, 2); // quit
      }
   }

}



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// webserver                                                            //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   basic_server::basic_server (int argc, char** argv, int max, 
                     bool ctrlC, int maxretry)
   : fCache (0, false), fLog (0), fMainSocket (-1), fPort (kDefaultPort)
   {
      // ignore SIGPIPE 
      struct sigaction siga;
      siga.sa_handler = SIG_IGN;
      sigemptyset (&siga.sa_mask);
      siga.sa_flags = SA_RESTART;
      sigaction (SIGPIPE, &siga, 0);
   
      // install signal handler for for ^c and TERM
      // mask Ctrl-C signal
      if (ctrlC) {
         sigset_t set;
         if ((sigemptyset (&set) != 0) ||
            (sigaddset (&set, SIGINT) != 0) ||
            (sigaddset (&set, SIGTERM) != 0) ||
            (pthread_sigmask (SIG_BLOCK, &set, 0) != 0)) {
            cerr << "Unable to connect Ctrl-C" << endl;
         }
         // create thread which handles Ctrl-C
         else {
            pthread_attr_t	tattr;
            int			status = -1;
            if (pthread_attr_init (&tattr) == 0) {
               pthread_attr_setdetachstate (&tattr, PTHREAD_CREATE_DETACHED);
               pthread_attr_setscope (&tattr, PTHREAD_SCOPE_PROCESS);
               pthread_t tid;
               status = pthread_create (&tid, &tattr, webserver_control_C, 
                                    this);
               pthread_attr_destroy (&tattr);
            }
            if (status != 0) {
               cerr << "Unable to connect Ctrl-C" << endl;
            }
         }
      }
   
      // Create sigjmp buffer (per thread variable!)
      // Setup long jmp buffer for main thread
      // Install trap handler
      sigjmp_retry = maxretry;
      const char* env = ::getenv ("DMTWEBSERVER_RETRY");
      if (env) {
         sigjmp_retry = atoi (env);
	 cout << "Set retry count to " << sigjmp_retry << endl;
      }
      if (sigjmp_retry < 0) sigjmp_retry = -1;
      if (sigjmp_retry >= 0) {
         static pthread_once_t sigjmp_create = PTHREAD_ONCE_INIT;
         pthread_once (&sigjmp_create, webserver_create_sigjmp);
         if (!sigjmp_err) {
            sigjmp_buf* buf = (sigjmp_buf*)malloc (sizeof (sigjmp_buf)); 
            pthread_setspecific (sigjmp_key, buf);
            // setup trap handler 
            struct sigaction siga;
            siga.sa_handler = webserver_trap_handler;
            sigemptyset (&siga.sa_mask);
            siga.sa_flags = 0;
            sigaction (SIGSEGV, &siga, 0);
            sigaction (SIGILL, &siga, 0);
            sigaction (SIGBUS, &siga, 0);
            sigaction (SIGFPE, &siga, 0);
         }
      }
   
      if (max <= 0) max = 5;
      sem_init (&fThreadCount, 0, max);
      // parse options
      gdsbase::option_string opt (argc, argv, "p:h");
      if (!opt.getOpt ('p', fPort)) {
         cerr << help_text << endl;
         return;
      }
      if (opt.opt('h')) {
         cout << help_text << endl;
         return;
      }
      // create the socket
      fMainSocket = ::socket (PF_INET, SOCK_STREAM, 0);
      if (fMainSocket == -1) {
         perror("webserver: Unable to create socket");
         return;
      }
      // this will allow the server to be restarted without a wait
      int val = 1;
      socklen_t optlen = sizeof (val);
      if (setsockopt (fMainSocket, SOL_SOCKET, SO_REUSEADDR, &val, optlen) == -1) {
         perror("webserver: Unable to reuse socket");
      }
      // bind socket
      struct sockaddr_in name;
      name.sin_family = AF_INET;
      name.sin_port = htons (fPort);
      name.sin_addr.s_addr = htonl (INADDR_ANY);
      if (::bind(fMainSocket, (struct sockaddr*) &name, 
		 sizeof (name)) < 0) {
         perror("webserver: Error binding socket");
         ::close (fMainSocket);
         fMainSocket = -1;
         return;
      }
   }

//______________________________________________________________________________
   basic_server::~basic_server()
   {
       thread::semlock lockit (fThreadMux);
       sigjmp_buf* buf = (sigjmp_buf*)pthread_getspecific (sigjmp_key);
       if (buf) {
	   pthread_setspecific (sigjmp_key, 0);
	   free(buf);
       }
       for (thread_list::iterator i = fThreads.begin(); 
          i != fThreads.end(); ++i) {
         pthread_cancel (*i);
      }
      fThreads.clear();
      Close();
   }

//______________________________________________________________________________
   void basic_server::Listen()
   {
      if (fMainSocket == -1) {
         return; 
      }
      sigjmp_buf* buf = 0;
      if (!sigjmp_err) {
         buf = (sigjmp_buf*)pthread_getspecific (sigjmp_key);
      }
      while (1) {
         int		c_sock;		/* client socket */
         struct sockaddr_in c_addr;	/* client address */
         socklen_t	c_len;		/* client addr length */
         int		ret = 0;	/* sigsetjmp return */
         // setup long jmp
         if (!buf || ((ret = sigsetjmp (*buf, 1)) == 0)) {
         
            // listen at port address
            if (::listen (fMainSocket, 1) == -1) {
               perror("webserver: Error while listening to socket");
               return;
            }
            // accept connection
            c_len = sizeof (c_addr);
            c_sock = ::accept (fMainSocket, 
                              (struct sockaddr*)&c_addr, &c_len);
            if (c_sock == -1) {
               perror("Error while accepting connection");
               return;
            }
            // Spawn thread
            // cout << "Spawn server thread" << endl;
            Spawn (c_sock, (struct sockaddr*)&c_addr, c_len);
         
         // continue listening
         }
      
         // always abort in main on trap
         if (ret != 0) {
            Close();
	    string date=TimeString(Now(), TIMESTAMP_FMT);
            if (fLog && fMux.trylock_timed (1000)) {
	      *fLog << date << "Abort after trap in main" << endl << flush;
               fMux.unlock();
            }
            ::exit (1);
         }
      }
   }

//______________________________________________________________________________
   void basic_server::Close()
   {
      if (fMainSocket != -1) {
         int sock = fMainSocket;
         fMainSocket = -1;
         if (::shutdown (sock, SHUT_RDWR)) {
	     perror("webserver: Failed to shut down main socket");
	 }
         if (::close (sock)) {
	     perror("webserver: Failed to close main socket");
	 }
      }
   }

//______________________________________________________________________________
   void basic_server::Interrupt (int sig)
   {
      cout << "Signal for " << (sig == SIGINT ? "^C" : "SIGTERM") << 
         " received. Terminating..." << endl;
      Close();
      ::exit (1);
   }

//______________________________________________________________________________
   struct thread_arg {
      int 		csock;
      struct sockaddr_in caddr;
      int 		clen;
      basic_server* 		obj;
   };

//______________________________________________________________________________
extern "C" 
   void* thread_web (void* voidarg)
   {
      thread_arg*  arg = (thread_arg*) voidarg;
      arg->obj->Parse (arg->csock, (struct sockaddr*)&arg->caddr, 
                      arg->clen);
      delete arg;
      return 0;
   }

//______________________________________________________________________________
   void basic_server::Spawn (int csock, struct sockaddr* caddr, int clen)
   {
      sem_wait (&fThreadCount);
   
      pthread_attr_t		tattr;
      int			status;
      pthread_t			tid;
      // set thread parameters: detached & process scope
      if (pthread_attr_init (&tattr) != 0) {
         return;
      }
      pthread_attr_setdetachstate (&tattr, PTHREAD_CREATE_DETACHED);
      pthread_attr_setscope (&tattr, PTHREAD_SCOPE_PROCESS);
      // create argument list
      thread_arg* arg = new thread_arg;
      arg->obj = this;
      arg->csock = csock;
      int len = sizeof (struct sockaddr_in);
      if (clen < len) len = clen;
      memcpy (&arg->caddr, caddr, len);
      arg->clen = clen;
      // create thread
      status = pthread_create (&tid, &tattr, thread_web, (void*) arg);
      pthread_attr_destroy (&tattr);
      if (status != 0) {
         cerr << "Unable to create child thread" << endl;
         delete arg;
      }
      else {
         thread::semlock lockit (fThreadMux);
         fThreads.push_back (tid);
      }
   }

//______________________________________________________________________________
   bool basic_server::Parse (int csock, struct sockaddr* caddr, int clen)
   {
      bool status = false;
   
      // setup long jmp
      sigjmp_buf* buf = 0;
      int ret = 10;	/* sigsetjmp return */
      if (!sigjmp_err) {
         buf = (sigjmp_buf*)malloc (sizeof (sigjmp_buf)); 
         pthread_setspecific (sigjmp_key, buf);
      }
      if (!buf || ((ret = sigsetjmp (*buf, 1)) == 0)) {
      
         int resplen = 0;
         // Get request
         request_t request;
         if (request.read (csock)) {
            // Process request
            response_t response;
            response.SetStatusLine ("HTTP/1.0 200 OK");
            // check for compression
            request_t::header_type::const_iterator encod =
               request.Header().find ("Accept-Encoding");
            if ((encod != request.Header().end()) &&
               (encod->second.find ("gzip") != string::npos)) {
               response.SetCanCompress();
            }
            // call request method
            if (Request (request, response)) {
               // response
               if (response.write (csock)) {
                  resplen = response.GetLength();
                  status = true; 
               }
            }
            else {
               // error
               response_t error;
               error.SetStatusLine ("HTTP/1.0 500 Internal Server Error");
               error.write (csock);
            }
         }
         else {
            // error
            response_t error;
            error.SetStatusLine ("HTTP/1.0 501 Not Implemented");
            error.write (csock);
         }
         // write log
         if (fLog) {
            struct in_addr addr;
            addr = ((struct sockaddr_in*)caddr)->sin_addr;

	    //--------------------------  Get a printable IP addr
	    char hostIP[64];
	    inet_ntop(AF_INET, &addr, hostIP, sizeof(hostIP));

#ifdef WEBSRV_HOSTNAME
	    //--------------------------  Optional host name
            char hostname[256];
            if (nsilookup (&addr, hostname) < 0) {
               hostname[0] = 0;
            }
#endif
            string date =  TimeString(Now(), TIMESTAMP_FMT);
            thread::semlock lockit (fMux);

	    *fLog << date << "Request for \"" << request.GetDemangledUrl() 
		  << "\" from " << hostIP;
#ifdef WEBSRV_HOSTNAME
	    if (hostname[0]) *fLog << " (" << hostname << ")";
#endif
	    *fLog << "; return "; 
	    if (status) *fLog << resplen / 1024.0 << " kB";
	    else        *fLog << "error";
	    *fLog << endl << flush;
	 }
      }

      if (buf) {
	  pthread_setspecific (sigjmp_key, 0);
	  free(buf);
	  buf = 0;
      }

      // quit
      if (ret == 2) {
         cout << "Close server connection" << endl;
         // close socket
         Close();
	 if (::shutdown (csock, SHUT_RDWR)) {
	   perror("webserver: Failed to shut down client socket");
	 }
	 if (::close (csock)) {
	   perror("webserver: Failed to close client socket");
	 }
         string date = TimeString (Now(), TIMESTAMP_FMT);
         if (fLog && fMux.trylock_timed (1000)) {
            *fLog << date << "Abort after trap" << endl << flush;
            fMux.unlock();
         }
         ::exit (1);
      }
      if (::shutdown (csock, SHUT_RDWR)) {
	perror("webserver: Failed to shut down client socket");
      }
      if (::close (csock)) {
	perror("webserver: Failed to close client socket");
      }

      // remove TID from list
      pthread_t tid = pthread_self();
      fThreadMux.lock();
      for (thread_list::iterator i = fThreads.begin(); 
          i != fThreads.end(); ) {
         if (*i == tid) {
            i = fThreads.erase (i);
         }
         else {
            ++i; 
         }
      }
      fThreadMux.unlock();
      // decrease thread count
      sem_post (&fThreadCount);
   
      // log trap continuation 
      if (ret == 1) {
         string date = TimeString(Now(), TIMESTAMP_FMT);
         if (fLog && fMux.trylock_timed (1000)) {
            *fLog << date << "Continue after trap" << endl << flush;
            fMux.unlock();
         }
      }
   
      return status;
   }

//______________________________________________________________________________
   bool basic_server::Request (const request_t& request,
                     response_t& response)
   {
      switch (request.GetType()) {
         case request_t::rGet:
            {
               if (CheckCache (request, response)) {
                  return true;
               }
               return RequestGet (request, response);
            }
         case request_t::rHead:
            {
               return RequestHead (request, response);
            }
         case request_t::rPost:
            {
               return RequestPost (request, response);
            }
         case request_t::rConnect:
            {
               return RequestConnect (request, response);
            }
         case request_t::rPut:
            {
               return RequestPut (request, response);
            }
         case request_t::rDelete:
            {
               return RequestDelete (request, response);
            }
         case request_t::rOptions:
            {
               return RequestOptions (request, response);
            }
         case request_t::rTrace:
            {
               return RequestTrace (request, response);
            }
         default:
            {
               response.SetStatusLine ("HTTP/1.0 501 Not Implemented");
               return true;
            }
      }
   }

//______________________________________________________________________________
   bool basic_server::RequestConnect (const request_t& request,
                     response_t& response)
   {
      response.SetStatusLine ("HTTP/1.0 501 Not Implemented");
      return true;
   }

//______________________________________________________________________________
   bool basic_server::RequestGet (const request_t& request,
                     response_t& response) 
   {
      cerr << "GET processing..." << endl;
      response.SetStatusLine ("HTTP/1.0 404 Not found");
      return true; 
   }

//______________________________________________________________________________
   bool basic_server::RequestHead (const request_t& request,
                     response_t& response) 
   {
      bool ret = RequestGet (request, response);
      response.SetData (0, 0);
      return ret;
   }

//______________________________________________________________________________
   bool basic_server::RequestPost (const request_t& request,
                     response_t& response)
   {
      response.SetStatusLine ("HTTP/1.0 501 Not Implemented");
      return true;
   }

//______________________________________________________________________________
   bool basic_server::RequestPut (const request_t& request,
                     response_t& response)
   {
      response.SetStatusLine ("HTTP/1.0 501 Not Implemented");
      return true;
   }

//______________________________________________________________________________
   bool basic_server::RequestDelete (const request_t& request,
                     response_t& response)
   {
      response.SetStatusLine ("HTTP/1.0 501 Not Implemented");
      return true;
   }

//______________________________________________________________________________
   bool basic_server::RequestOptions (const request_t& request,
                     response_t& response)
   {
      response.SetStatusLine ("HTTP/1.0 501 Not Implemented");
      return true;
   }

//______________________________________________________________________________
   bool basic_server::RequestTrace (const request_t& request,
                     response_t& response)
   {
      response.SetStatusLine ("HTTP/1.0 501 Not Implemented");
      return true;
   }

//______________________________________________________________________________
   bool basic_server::CheckCache (const request_t& request, 
                     response_t& response)
   {
      // check if cacheing is enabled
      if (fCache.getMax() == 0) {
         return false;
      }
      // lookup
      webcache::cachepage page;
      if (!fCache.lookup (request.GetUrl(), page)) {
         return false;
      }
      // check expiration time
      if (page.getTime() < Now()) {
         return false;
      }
      cout << "From cache \"" << request.GetDemangledUrl() << "\"" << endl;
      if (page.size() > 0) {
         // set status line
         response.SetStatusLine ("HTTP/1.0 200 OK");
      	 // set header
         response.GetHeader() = page.header();
      	 // set content
         response.Reserve (page.size());
         memcpy (response.GetData(), page.get(), page.size());
      	 // try to compress
         if (page.allowCompress()) {
            response.TryCompress();
         }
         // Set content length
         if (response.GetLength() > 0) {
            response.RemoveHeader ("Content-Length");
            ostringstream ostr;
	    ostr << response.GetLength();
            response.AddHeader ("Content-Length", ostr.str());
         }
      }
      else {
         response.SetStatusLine ("HTTP/1.0 404 Not found");
      }
      return true;
   }


}
