//
//    Stand alone application to create a shared memory partition.
//
//    Syntax:
//       smcreate <name> [-nbuf <nbuf>] [-lbuf <lbuf>]
//
//       <name>    name of the partition to be created
//       <nbuf>    number of buffers 
//       <lbuf>    length of each buffer in bytes.
//
#include <stdlib.h>
#include <iostream>
#include "lsmp_prod.hh"
#include <string>

using namespace std;

int 
main(int argc, const char ** argv) {
    std::string name;
    int nbuf = 4, lbuf = 2048;
    bool lock = false;
    //-------------------------------  Get the command line arguments
    bool argerr(false), ra(false);

    for (int i=1 ; i<argc ; i++) {
        std::string argi = argv[i];
        if (argi == "-nbuf") {
	    nbuf = strtol(argv[++i], 0, 0);
	} else if (argi == "-lbuf") {
	    lbuf = strtol(argv[++i], 0, 0);
	} else if (argi == "-lock") {
	    lock = true;
	} else if (argi == "+readall" || argi == "+ra") {
	    ra = true;
	} else if (i == 1 && (argi[0] != '-') && (argi[0] != '?')) {
	    name = argi;
	} else {
	    std::cerr << "smcreate: Invalid argument " << argi << std::endl;
	    argerr = true;
	}
    }

    //-------------------------------  Check the parition parameters
    if (name.empty()) {
	cerr << "smcreate: No partition name specified." << endl;
	argerr = true;
    }

    if (nbuf <= 0) {
	cerr << "smcreate: Number of buffers must be > 0" << endl;
        argerr = true;
    }

    if (lbuf <= 0) {
	cerr << "smcreate: Buffer length must be > 0" << endl;
        argerr = true;
    }

    //-------------------------------  Print an error message if appropriate.
    if (argerr) {
        cerr << "smcreate: Create a shared memory partition" << endl;
	cerr << "Syntax:" << endl;
	cerr << *argv << " <partition-name> [-nbuf <number-of buffers>]"
	     << " [-lbuf <buffer-length>] [+readall | +ra]" << endl;
	exit(1);
    }

    //-------------------------------  Do the dirty work. Make sure the 
    //                                 partition isn't deleted immediately 
    LSMP_PROD* part = new LSMP_PROD(name.c_str(), nbuf, lbuf);
    if (lock) part->lock(true);
    if (ra) part->bufmode(3);
    part->keep();
    delete part;
    exit(0);
}
