/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "lmsg/Buffer.hh"
#include "lmsg/BufferPool.hh"
#include "lmsg/Message.hh"
#include "lmsg/ErrorList.hh"
#include "lmsg/Socket.hh"
#include "lmsg/SocketPool.hh"
#include "lmsg/TransportMsg.hh"
#include "lmsg/TransInput.hh"
#include "lmsg/TransOutput.hh"
#include <iostream>

using namespace lmsg;
using namespace std;

//======================================  Destructor
TransportMsg::~TransportMsg(void) {
    close();
}

//======================================  Constructor
TransportMsg::TransportMsg(void) 
  : mSockPool(&defaultLMsgSocketPool), mSocket(0),  
  mBuffPool(&defaultLMsgBufferPool), mDebug(0), mReplyAddr(0) 
{
}

//======================================  Get socket address
MsgAddr 
TransportMsg::getAddr(void) const {
    if (isOpen()) return mSocket->getAddr();
    else          return MsgAddr();
}

//======================================  Is socket connected?
bool
TransportMsg::isConnected(void) const {
    if (!isOpen()) return false;
    return mSocket->isConnected();
}

//======================================  Close the socket
void
TransportMsg::close(void) {
    if (mSocket) {
        if (isConnected()) disconnect();
	mSocket->async(false);
        mSocket->Return();
	mSocket = 0;
    }
}

//======================================  Kill the socket?
void
TransportMsg::Purge(void) {
    if (mSocket) {
        delete mSocket;
	mSocket = 0;
    }
}

//======================================  Open the socket
error_type 
TransportMsg::open(long flags, const MsgAddr* addr) {
    if ((flags & o_client)) {
        mSocket = mSockPool->getSocket(SocketPool::s_reusable);
    } else {
        mSocket = mSockPool->getSocket(SocketPool::s_new, addr);
    }
    if (!isOpen()) return NotOpen;
    mSocket->setDebug(mDebug);
    if ((flags & o_async)) mSocket->async(true);
    mSocket->setPool(mSockPool);
    mSocket->setPool(mBuffPool);
    return OK;
}

//======================================  Send a message
error_type 
TransportMsg::send(const MsgHeader& hdr, const char* b) {
    if (!hdr.getDest()) return BadAddress;
    if (!mBuffPool) return NoPool;
    Buffer* buf = mBuffPool->getBuffer();
    if (!buf) return NoBuffer;
    index_type len = hdr.getMsgLength();
    if ((index_type)buf->getDataSize() < len) return SizeError;
    buf->getHeader() = hdr;
    memcpy(buf->getDataAddr(), b, len);
    buf->setDataLength(len);
    return send(buf);
}

//======================================  Send a message
error_type 
TransportMsg::send(const MsgHeader& hdr, const Message& msg) {
    if (!hdr.getDest()) return BadAddress;
    if (!mBuffPool) return NoPool;
    Buffer* buf = mBuffPool->getBuffer();
    if (!buf) return NoBuffer;
    buf->getHeader() = hdr;
    TransOutput out(*buf);
    try {
        msg.getData(out);
    } catch (exception& e) {
        if (getDebug()) {
	    cout << "Exception: " << e.what() << " in TransportMsg::send" 
		 << endl;
	    cout << "Output buffer length: " << buf->getSize() << endl;
	}
	buf->Return();
	return SizeError;
    }
    index_type len = out.getNBytes();
    buf->setDataLength(len);
    buf->getHeader().setMsgLength(len);
    buf->getHeader().setMsgType(msg.getType());
    return send(buf);
}

//======================================  Send message contained in a buffer
error_type 
TransportMsg::send(Buffer* b) {
    if (!isOpen()) return NotOpen;
    if (mReplyAddr == MsgAddr(0)) b->getHeader().setSource(getAddr());
    else                          b->getHeader().setSource(mReplyAddr);
    if (mDebug > 1) {
	cerr << "TransportMsg - send message. Header:" << endl;
	b->getHeader().Dump(cerr);
    }
    b->getHeader().Export();
    index_type rc = mSocket->send(b->getHeader().getDest(), b);
    if (rc) b->Return();
    return rc;
}

//======================================  Receive a message
error_type 
TransportMsg::receive(MsgHeader& hdr, char* Data, size_type maxlen, 
			    wtime_type timeout) {
    Buffer* buf;
    index_type rc = receive(&buf, timeout);
    if (rc) return rc;
    hdr = buf->getHeader();
    size_type len = hdr.getMsgLength();
    if (len > maxlen) rc = SizeError;
    else              memcpy(Data, buf->getDataAddr(), len);
    buf->Return();
    return rc;
}

//======================================  Receive a message
error_type 
TransportMsg::receive(MsgHeader& hdr, Message& msg, wtime_type timeout) {
    Buffer* buf;
    index_type rc = receive(&buf, timeout);
    if (rc) return rc;
    hdr = buf->getHeader();
    if (hdr.getMsgType() != msg.getType()) {
        if (getDebug()) {
	    cout << "Message type: " << hdr.getMsgType() << " not expected (" 
		 << msg.getType() << ")" << endl;
	}
        rc = Invalid;
    } else {
        TransInput in(*buf);
	try {
	    msg.setData(in);
	} catch (exception& e) {
	    if (getDebug()) {
	        cout << "Exception: " << e.what() 
		     << " in TransportMsg::receive" << endl;
	    }
	    rc = SizeError;
	}
    }
    buf->Return();
    return rc;
}

//======================================  Receive a message
error_type 
TransportMsg::receive(Buffer** Data, wtime_type timeout) {
    if (!isOpen()) return NotOpen;
    index_type rc;
    if (timeout >= 0.0) {
        rc = mSocket->wait(timeout);
	if (rc) return rc;
    }
    rc = mSocket->receive(Data);
    if (rc) return rc;
    (*Data)->getHeader().Import();
    // (*Data)->getHeader().getSource() = mSocket->getPartner();
    MsgAddr* SrcAddr = &((*Data)->getHeader().getSource());
    if (SrcAddr->getIPAddr() == 0) {
        SrcAddr->setIPAddr(mSocket->getPartner().getIPAddr());
    }
    if (mDebug > 1) {
	cerr << "TransportMsg - receive message. Header:" << endl;
	(*Data)->getHeader().Dump(cerr);
    }
    return OK;
}

//======================================  Wait for a message to arrive
error_type 
TransportMsg::waitMsg(wtime_type timeout) const {
    if (!isOpen()) return NotOpen;
    return mSocket->wait(timeout);
}

error_type 
TransportMsg::waitAny(wtime_type timeout, int N, const TransportMsg* list[]) {
    if (N <= 0) return NotOpen;
    const Socket** sList = new const Socket*[N];
    int nOpen = 0;
    for (int i=0 ; i<N ; i++) {
        if (list[i]->isOpen()) sList[nOpen++] = list[i]->mSocket;
    }
    error_type rc =  Socket::waitAny(timeout, nOpen, sList);
    delete[] sList;
    return rc;
}

//======================================  Connect to a partner.
error_type 
TransportMsg::connect(const MsgAddr& addr) {
    if (!addr) return BadAddress;
    if (!isOpen()) return NotOpen;
    return mSocket->connect(addr);
}

//======================================  Break connection.
error_type 
TransportMsg::disconnect(void) {
    if (!isOpen()) return NotOpen;
    return mSocket->disconnect();
}

void 
TransportMsg::setSocketPool(SocketPool* pool) {
    mSockPool = pool;
}

void 
TransportMsg::setBufferPool(BufferPool* pool) {
    mBuffPool = pool;
}

//======================================  Set the debug level
void 
TransportMsg::setDebug(index_type level) {
    mDebug = level;
    if (isOpen()) mSocket->setDebug(level);
}

//======================================  Set the reply address
void 
TransportMsg::setReplyAddr(const MsgAddr& rep) {
    mReplyAddr = rep;
}
