/* -*- mode: c++; c-basic-offset: 3; -*- */

#ifndef JSON_STACK_HH
#define JSON_STACK_HH

#include <json/value.h>
#include <stdexcept>
#include <string>
#include <vector>

class jsonStack {
public:
   typedef std::vector<bool>        bool_vect;
   typedef std::vector<double>      dble_vect;
   typedef std::vector<std::string> string_vect;
public:
   jsonStack(void);
   jsonStack(const std::string& file);
   size_t depth(void) const;
   bool empty(void) const;
   size_t fetch_data(const std::string& name, bool& value) const;
   size_t fetch_data(const std::string& name, double& value) const;
   size_t fetch_data(const std::string& name, std::string& value) const;
   size_t fetch_data(const std::string& name, bool_vect& value) const;
   size_t fetch_data(const std::string& name, dble_vect& value) const;
   size_t fetch_data(const std::string& name, string_vect& value) const;

   size_t fetch_value(const Json::Value& name, bool& value) const;
   size_t fetch_value(const Json::Value& name, double& value) const;
   size_t fetch_value(const Json::Value& name, std::string& value) const;
   size_t fetch_value(const Json::Value& name, bool_vect& value) const;
   size_t fetch_value(const Json::Value& name, dble_vect& value) const;
   size_t fetch_value(const Json::Value& name, string_vect& value) const;
   bool iterate(void);
   bool isArray(void) const;
   bool isMember(const std::string& name) const;
   bool isStruct(void) const;
   size_t length(void) const;
   std::vector<std::string> getNames(void) const;
   bool parse(const std::string& file);
   
   /**  Push an empty element onto the stack.
     *  \brief Push an empty element.
     *  \return Always true (successful)
     */
   bool push_element(void);

   /**  Push a named element onto the stack.
     *  \brief Push a named element.
     *  \return true if the named element was found and pushed.
     */
   bool push_element(const std::string& name);
   
   /**  Push a specified json value.
     *  \brief Push a specified element.
     *  \return Always true (successful)
     */
   bool push_element(const Json::Value& val);

   /**  Pop the top json element off of the stack.
     *  \brief pope an element
     *  \return true if the stck is not empty after the pop operation.
     */
   bool pop(void);

   const Json::Value& value(void) const;
private:
   Json::Value& value(void);
private:
   enum eltype {
      k_element,
      k_array,
      k_structure
   };

  struct stack_element {
    stack_element(void);
    stack_element(const Json::Value& se);
    Json::Value _value;
    eltype      _type;
    Json::ArrayIndex _size;
    Json::ArrayIndex _iter;
  };
  std::vector<stack_element> _stack;
  size_t _stack_size;
};

//======================================  Inline functions
inline size_t
jsonStack::depth(void) const {
  return _stack.size();
}

inline bool
jsonStack::empty(void) const {
  return _stack.empty();
}

inline bool 
jsonStack::isArray(void) const {
   return value().isArray();  
}

inline bool 
jsonStack::isMember(const std::string& name) const {
   return value().isMember(name); 
}

inline bool 
jsonStack::isStruct(void) const {
   return value().isObject();
}

inline size_t
jsonStack::length(void) const {
  return value().size();
}

inline const Json::Value&
jsonStack::value(void) const {
  if (empty()) throw std::runtime_error("Operation on empty json stack");
  return _stack.back()._value;
}

inline Json::Value& 
jsonStack::value(void) {
  if (empty()) throw std::runtime_error("Operation on empty json stack");
  return _stack.back()._value;
}

#endif     // !defined(JSON_STACK_HH)
