/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "jsonStack.hh"
#include <json/reader.h>
#include <fstream>

using namespace std;

//======================================  Json stack model constructor
jsonStack::jsonStack(void) {
   _stack.reserve(16);
}

//======================================  Json stack model initializer
jsonStack::jsonStack(const std::string& file) {
   _stack.reserve(16);
   parse(file);
}

//======================================  Fetch bool data element
size_t 
jsonStack::fetch_data(const std::string& name, bool& val) const {
   if (!isMember(name)) return 0;
   return fetch_value(value()[name], val);
}

//======================================  Fetch numeric data element
size_t 
jsonStack::fetch_data(const std::string& name, double& val) const {
   if (!isMember(name)) return 0;
   return fetch_value(value()[name], val);
}

//======================================  Fetch string data element
size_t 
jsonStack::fetch_data(const std::string& name, std::string& val) const {
   if (!isMember(name)) return 0;
   return fetch_value(value()[name], val);
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_data(const std::string& name, std::vector<bool>& val) const {
   if (!isMember(name)) return 0;
   return fetch_value(value()[name], val);
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_data(const std::string& name, dble_vect& val) const {
   if (!isMember(name)) return 0;
   return fetch_value(value()[name], val);
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_data(const std::string& name, string_vect& val) const {
   if (!isMember(name)) return 0;
   return fetch_value(value()[name], val);
}

//======================================  Fetch bool data element
size_t 
jsonStack::fetch_value(const Json::Value& el, bool& val) const {
   if (!el.isBool()) return 0;
   val = el.asBool();
   return 1;
}

//======================================  Fetch numeric data element
size_t 
jsonStack::fetch_value(const Json::Value& el, double& val) const {
   if (!el.isNumeric()) return 0;
   val = el.asDouble();
   return 1;
}

//======================================  Fetch string data element
size_t 
jsonStack::fetch_value(const Json::Value& el, std::string& val) const {
   val = el.asString();
   return 1;
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_value(const Json::Value& vec, bool_vect& val) const {
   Json::ArrayIndex N = 0;
   val.clear();
   if (isArray()) {
      N = vec.size();
      val.reserve(N);
      for (Json::ArrayIndex i=0; i < N; i++) {
	 if (!vec[i].isBool()) N = 0;
	 else                  val.push_back(vec[i].asBool());
      }
   } else {
      bool b;
      val.resize(1);
      N = fetch_value(vec, b);
      if (N) val[0] = b;
   }
   return N;
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_value(const Json::Value& vec, dble_vect& val) const {
   Json::ArrayIndex N = 0;
   val.clear();
   if (isArray()) {
      N = vec.size();
      val.reserve(N);
      for (Json::ArrayIndex i=0; i < N; i++) {
	 if (!vec[i].isNumeric()) N = 0;
	 else                     val.push_back(vec[i].asDouble());
      }
   } else {
      val.resize(1);
      N = fetch_value(vec, val.at(0));
   }
   return N;
}

//======================================  Fetch bool data vector
size_t 
jsonStack::fetch_value(const Json::Value& vec, string_vect& val) const {
   Json::ArrayIndex N = 0;
   val.clear();
   if (isArray()) {
      N = vec.size();
      val.reserve(N);
      for (Json::ArrayIndex i=0; i < N; i++) {
	 val.push_back(vec[i].asString());
      }
   } else {
      val.resize(1);
      N = fetch_value(vec, val.at(0));
   }
   return N;
}

//======================================  Get a list of members
std::vector<std::string>
jsonStack::getNames(void) const {
   return value().getMemberNames();
}

//======================================  Parse out the file
bool
jsonStack::parse(const std::string& file) {
   //------------------------------------  Read the configuration file.
   ifstream in(file.c_str());
   if (!in.is_open()) {
      throw runtime_error(string("unable to open file: ") + file);
   }
   string conf_str;
   getline(in, conf_str, (char)EOF);
   if (conf_str.empty()) {
      throw runtime_error(string("Configuration file: ") + file + " is empty");
   }
   Json::Value  doc;
   Json::Reader rdr;
   rdr.parse(conf_str.data(), conf_str.data() + conf_str.size(), doc);
   push_element(doc);
   return true;
}

//======================================  Push a sub-document onto the stack
bool 
jsonStack::push_element(void) {
   stack_element el;
   _stack.push_back(el);
   return true;
}

//======================================  Push a sub-document onto the stack
bool 
jsonStack::push_element(const std::string& name) {
   if (!isMember(name)) return false;
   return push_element(value()[name]);
}

//======================================  Push a sub-document onto the stack
bool 
jsonStack::push_element(const Json::Value& doc) {
   _stack.push_back(stack_element(doc));
   return true;
}
   
//======================================  Push a sub-document onto the stack
bool 
jsonStack::pop(void) {
   if (empty()) throw logic_error("jsonStack: Attempt to pop from empty stack");
   _stack.pop_back();
   return !empty();
}

//======================================  Iterate over array elements
bool
jsonStack::iterate(void) {
   if (empty()) {
      throw logic_error("jsonStack: Attempted operation on empty stack");
   }
   stack_element& el = _stack.back();
   if (el._iter >= el._size) return false;
   push_element(el._value[el._iter++]);
   return true;
}

//======================================  stack_element default constructor
jsonStack::stack_element::stack_element(void) {
   _size  = 0;
   _iter  = 0;
}

//======================================  stack_element data constructor
jsonStack::stack_element::stack_element(const Json::Value& doc) {
   _value = doc;
   _size  = doc.size();
   _iter  = 0;
}
