/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: fftype							*/
/*                                                         		*/
/* Module Description: Basic data types for framefast and fantom	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 26Sep00  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: fftype.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_FFTYPE_H
#define _LIGO_FFTYPE_H

#include "PConfig.h"
#include <complex>
#ifdef HAVE_INT_TYPES
#include <inttypes.h>
#endif

/** @name Atomic frame data types
    Data types used by the frame standard for integers and reals.
   
    @memo Atomic frame data types
    @author Written October 2000 by Daniel Sigg
    @version 1.0
    @ingroup IO_framefast
 ************************************************************************/

//@{

namespace framefast {

   const int kDefaultFrameVersion = 6;

#ifdef HAVE_INT_TYPES
   /// CHAR
   typedef int8_t int_1s_t;
   /// CHAR_U
   typedef uint8_t int_1u_t;
   /// INT_2S
   typedef int16_t int_2s_t;
   /// INT_2U
   typedef uint16_t int_2u_t;
   /// INT_4S
   typedef int32_t int_4s_t;
   /// INT_4U
   typedef uint32_t int_4u_t;
   /// INT_8S
   typedef int64_t int_8s_t;
   /// INT_8U
   typedef uint64_t int_8u_t;
#else // take a guess
   // CHAR
   typedef char int_1s_t;
   // CHAR_U
   typedef unsigned char int_1u_t;
   // INT_2S
   typedef short int_2s_t;
   // INT_2U
   typedef unsigned short int_2u_t;
   // INT_4S
   typedef int int_4s_t;
   // INT_4U
   typedef unsigned int int_4u_t;
   // INT_8S
   typedef long long int_8s_t;
   // INT_8U
   typedef unsigned long long int_8u_t;
#endif

   /// REAL_4
   typedef float real_4_t;
   /// REAL_8
   typedef double real_8_t;
   /// COMPLEX_8
   typedef std::complex<float> complex_8_t;
   /// COMPLEX_16
   typedef std::complex<double> complex_16_t;

/** Frame type. Frame types supported by fantom.
    
    @memo Frame type.
 ************************************************************************/
   enum frametype {
   /// Unknown/invalid
   NONE = 0,
   /// full frame
   FF = 1,
   /// second-trend frame
   STF = 2,
   /// minute-trend frame 
   MTF = 3,
   /// Reduced data set
   RDS = 4
   };

//@}


/** @name Byte order support
    Inlines routines to swap the byte order of atomic types.
   
    @memo Byte order support
    @author Written October 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/

//@{

#ifndef __CINT__
   /// Swap
   inline void swap (int_8u_t* ww);
   /// Swap
   inline void swap (int_8s_t* ww);
#endif
   /// Swap
   inline void swap (int_4u_t* w);
   /// Swap
   inline void swap (int_4s_t* ww);
   /// Swap
   inline void swap (int_2u_t* w);
   /// Swap
   inline void swap (int_2s_t* ww);
   /// Swap
   inline void swap (int_1s_t* ww);
   /// Swap
   inline void swap (int_1u_t* ww);
   /// Swap
   inline void swap (real_8_t* ww);
   /// Swap
   inline void swap (real_4_t* w);
   /// Swap
   inline void swap (complex_8_t* ww);
   /// Swap
   inline void swap (complex_16_t* ww);
   /// Check byte order
   inline bool littleendian ();

}
//@}



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// inlines						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
namespace framefast {
#define byteswap2(a,b) ((char *)&(a))[0] = ((char *)&(b))[1]; ((char *)&(a))[1] = ((char *)&(b))[0];
#define byteswap4(a,b) ((char *)&(a))[0] = ((char *)&(b))[3]; ((char *)&(a))[1] = ((char *)&(b))[2];((char *)&(a))[2] = ((char *)&(b))[1];((char *)&(a))[3] = ((char *)&(b))[0];
#define byteswap8(a,b) ((char *)&(a))[0] = ((char *)&(b))[7]; ((char *)&(a))[1] = ((char *)&(b))[6];((char *)&(a))[2] = ((char *)&(b))[5];((char *)&(a))[3] = ((char *)&(b))[4];((char *)&(a))[4] = ((char *)&(b))[3]; ((char *)&(a))[5] = ((char *)&(b))[2]; ((char *)&(a))[6] = ((char *)&(b))[1]; ((char *)&(a))[7] = ((char *)&(b))[0];

#ifndef __CINT__
   inline void swap (int_8u_t* ww)
   {
      int_8u_t tmp;
      byteswap8(tmp, *ww);
      *ww = tmp;
   }

//______________________________________________________________________________
   inline void swap (int_8s_t* ww) 
   {
      swap ((int_8u_t*)ww);
   }
#endif

//______________________________________________________________________________
   inline void swap (int_4u_t* w)
   {
      int_4u_t tmp;
      byteswap4(tmp, *w);
      *w = tmp;
   }

//______________________________________________________________________________
   inline void swap (int_4s_t* ww) 
   {
      swap ((int_4u_t*)ww);
   }

//______________________________________________________________________________
   inline void swap (int_2u_t* w)
   {
      int_2u_t tmp;
      byteswap2(tmp, *w);
      *w = tmp;
   }

//______________________________________________________________________________
   inline void swap (int_2s_t* ww) 
   {
      swap ((int_2u_t*)ww);
   }

//______________________________________________________________________________
   inline void swap (int_1s_t* ww)
   {
   }

//______________________________________________________________________________
   inline void swap (int_1u_t* ww)
   {
   }

//______________________________________________________________________________
   inline void swap (real_8_t* ww)
   {
      real_8_t tmp;
      byteswap8(tmp, *ww);
      *ww = tmp;
   }

//______________________________________________________________________________
   inline void swap (real_4_t* w)
   {
      real_4_t tmp;
      byteswap4(tmp, *w);
      *w = tmp;
   }

//______________________________________________________________________________
   inline void swap (complex_8_t* ww)
   {
      struct complex {
         real_4_t re;
         real_4_t im;
      };
      swap (&((complex*)ww)->re);
      swap (&((complex*)ww)->im);
   }

//______________________________________________________________________________
   inline void swap (complex_16_t* ww)
   { 
      struct complex {
         real_8_t re;
         real_8_t im;
      } ;
      swap (&((complex*)ww)->re);
      swap (&((complex*)ww)->im);
   }

   inline bool littleendian () {
      int_4u_t test = 0; *(int_1u_t*) &test = 1;
      return (test == 1);
   }

#undef byteswap2
#undef byteswap4
#undef byteswap8
}

#endif // _LIGO_FFTYPE_H
