/* -*- mode: c++; c-basic-offset: 4; -*- */
VCSID("@(#)$Id: NDS1Socket.cc 7582 2016-03-01 22:25:48Z john.zweizig@LIGO.ORG $");
//
#ifndef __EXTENSIONS__
#define __EXTENSIONS__
#endif
#ifndef _BSD_SOURCE
#define _BSD_SOURCE
#endif
#include "PConfig.h"
#include "NDS1Socket.hh"
#include "sockutil.h"
#include "Time.hh"
#include "Interval.hh"
#include <cstdio>
#include <cerrno>
#include <unistd.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <sys/file.h>
#include <iostream>
#include <sstream>

#define _TIMEOUT 10000000	/* 10 sec */
#ifdef P__WIN32
#define MSG_DONTWAIT 0
#endif

union int2float {
    int   i;
    float f;
};

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Forwards							        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

using namespace sends;
using namespace std;
using namespace thread;

//======================================  Constructors
NDS1Socket::NDS1Socket() 
    : mSocket(-1), mGetAll(false)
{
}

NDS1Socket::NDS1Socket(const string& ipaddr, int ipport, long RcvBufferLen) 
    : mSocket(-1), mGetAll(false)
{   
    //----------------------------------  Open the socket
    open(ipaddr, ipport, RcvBufferLen);
}

//======================================  Destructor
NDS1Socket::~NDS1Socket() {
    if (isOpen()) close();
}

//======================================  Open a socket
int 
NDS1Socket::open(const string& ipaddr, int ipport, long RcvBufferLen) 
{
    semlock lockit (mux);
    struct sockaddr_in socknam;
    char version[4];
    long size;
   
    //----------------------------------  Make sure the socket isn't open
    if (isOpen()) return -1;
   
    //----------------------------------  Get a socket
    mSocket = socket(PF_INET,SOCK_STREAM,0);
    if (mSocket < 0) 
	return -1;
      
    //----------------------------------  Bind the socket to a port.
    socknam.sin_family      = AF_INET;
    socknam.sin_port        = 0;
    socknam.sin_addr.s_addr = 0;
    int len                 = sizeof(socknam);
    if (::bind(mSocket, (struct sockaddr *)&socknam, len) < 0) 
	return -1;

    //----------------------------------  Connect to the server.
    socknam.sin_family      = AF_INET;
    socknam.sin_port        = htons (ipport);
    if (nslookup (ipaddr.c_str(), &socknam.sin_addr) < 0) {
	return -1;
    }

    wait_time timeout = _TIMEOUT / 1000000;
    if (connectWithTimeout (mSocket, (struct sockaddr *)&socknam, 
			    sizeof (socknam), timeout) < 0)
	return -1;

    mOpened = true;
   
    //----------------------------------  Set the receive buffer length
    setRcvBufLen(mSocket, RcvBufferLen);

    //----------------------------------  Get the server version number
    mVersion = 0;
    mRevision = 0;
    int rc = SendRequest("version;", version, 4, &size, timeout );
    if (rc || (size != 4)) {
	::close(mSocket);
	mOpened = false;
	return rc ? rc : -1;
    }

    mVersion = CVHex(version, 4);
    rc = SendRequest("revision;", version, 4, &size, timeout);
    if (rc || (size != 4)) {
	::close(mSocket); 
	mOpened = false;
	return rc ? rc : -1;
    }

    mRevision = CVHex(version, 4);
    if (mDebug) cerr << "Connected to server version " << Version() << endl;

    //----------------------------------  Done, Return.
    return rc;
}

//======================================  Close a socket
void NDS1Socket::close(void) 
{
    semlock lockit (mux);
    if (mOpened) {
	StopWriter();
	SendRequest("quit;");
	::close(mSocket);
	mOpened = false;
    }
    mRequest_List.clear();
    mWriterType = NoWriter;
}

//======================================  flush data from socket
void NDS1Socket::flush () {
    semlock lockit (mux);
    const int buflen = 16 * 1024;
    char text[buflen];
   
    int i = 0;
    int rc;
    do {
	rc = recv (mSocket, text, buflen, MSG_DONTWAIT);
    } while ((rc >= buflen) && (++i < 100));
}

//======================================  Add a channel to the channel list
int 
NDS1Socket::AddChannel(const string& chan, chantype ty, double rate) {
    semlock lockit (mux);

    //----------------------------------  Set the all flag if all specified.
    if (string(chan) == "all") {
	mGetAll = true;
	mRequest_List.clear();
    }
    if (mGetAll) return 1;
    return DAQC_api::AddChannel(chan, ty, rate);
}

//======================================  Get a list of available channels
int 
NDS1Socket::addAvailable(chantype chant, long gps, vector<DAQDChannel>& list,
		      wait_time timeout) {
    return addAvailable(list, timeout);
}

//======================================  Get a list of available channels
int 
NDS1Socket::Available (std::vector<DAQDChannel>& list, wait_time timeout) {
    list.clear();
    return addAvailable(list, timeout);
}

//======================================  Get a list of available channels
int 
NDS1Socket::addAvailable (std::vector<DAQDChannel>& list, wait_time timeout) {
    semlock lockit (mux);
    char buf[1024];
    // extendedList - allow more channels.  Original protocol only allowed 4 bytes for
    //  the number of channels, but there's lots more than that with mVersion 11 and higher,
    //  so allow 8 bytes.
    bool extendedList = ((mVersion > 11) || 
			 ((mVersion == 11) && (mRevision >= 3)));
    bool longNames = (mVersion >= 12) ;		// longer names introduced in version 12.
    int	 nameLength  = 40 ;			// Either 40 or MAX_CHNNAME_SIZE
    
   
    //----------------------------------  Request a list of channels.
    int rc = 0;
    if (extendedList) {
	rc = SendRequest("status channels 2;", buf, 8);
    }
    else {
	rc = SendRequest("status channels;", buf, 4);
    }
    if (rc) 
	return -1;
   
    //----------------------------------  Get the number of channels.
    int nw= extendedList ? CVHex(buf, 8) : CVHex(buf, 4);
    int recsz = 52;
    if ((mVersion == 9) || (mVersion == 10)) recsz = 60;
    if (mVersion >= 11) recsz = 124;
    if (extendedList) recsz = 128;
    if (longNames)
    {
       recsz = 128 + (MAX_CHNNAME_SIZE -40) ;
       nameLength = MAX_CHNNAME_SIZE ;
    }

    //----------------------------------  Skip DAQ data rate.
    if (!extendedList) rc = RecvRec(buf, 4, true);
   
    //----------------------------------  Fill in the list (test)
    list.reserve(list.size() + nw);
    DAQDChannel chn;
    for (int i = 0 ; i < nw ; i++) {
	rc = RecvRec(buf, recsz, true);
	if (rc <recsz) return -1;
	int j=nameLength-1;
	while (j>=0 && isspace(*(buf+j))) j--;
	chn.mName = string(buf,j+1);
	int group = 0;
	if (extendedList) {
	    int2float i2f;
	    chn.mRate  = CVHex (buf + nameLength, 8);
	    //chn.mNum   = CVHex (buf + 48, 8);
	    group = CVHex (buf + nameLength + 16, 4);
	    //chn.mBPS = CVHex (buf + 56, 4);
	    chn.mDatatype = static_cast<datatype>(CVHex (buf + nameLength + 20, 4));
	    i2f.i = CVHex(buf + nameLength + 24, 8);
	    chn.mGain = i2f.f;
	    i2f.i = CVHex(buf + nameLength + 32, 8);
	    chn.mSlope = i2f.f;
	    i2f.i = CVHex(buf + nameLength + 40, 8);
	    chn.mOffset = i2f.f;
	    int j=39; 
	    while (j>=0 && isspace( *(buf + nameLength + 48 + j) ))  j--;
	    chn.mUnit = string(buf + nameLength + 48, j + 1);
	}
	else {
	    chn.mRate  = CVHex(buf+40, 4);
	    //chn.mNum = CVHex(buf+44, 4);
	    group = CVHex(buf+48, 4);
	    if (recsz > 52) {
		//chn.mBPS = CVHex(buf+52, 4);
		chn.mDatatype = static_cast<datatype>(CVHex(buf+56, 4));
	    } 
	    else {
		//chn.mBPS = 0;
		chn.mDatatype = _undefined;
	    }
	    if (recsz > 60) {
		int2float i2f;
		i2f.i = CVHex(buf+60, 8);
		chn.mGain = i2f.f;
		i2f.i = CVHex(buf+68, 8);
		chn.mSlope = i2f.f;
		i2f.i = CVHex(buf+76, 8);
		chn.mOffset = i2f.f;
		int j = 39;
		while (j>=0 && isspace( *(buf+84+j) )) j--;
		chn.mUnit = string(buf+84, j+1);
	    }
	    else {
		chn.mGain = 0;
		chn.mSlope = 0;
		chn.mOffset = 0;
		chn.mUnit.clear();
	    }
	}

	//------------------------------  Set the channel type code.
	switch (group) {
	case 0:
	    chn.mChanType = cOnline;
	    break;
	case 1000:
	case 1001:
	    chn.mChanType = cMTrend;
	    break;
	default:
	    chn.mChanType = cUnknown;
	}
	list.push_back (chn);
    }
    return nw;
}

//======================================  Receive proprietary data into buffer
int 
NDS1Socket::GetData(wait_time timeout) {

    //----------------------------------  Lock the socket
    semlock lockit (mux);
    if (mWriterType != DataWriter) return -1;

    int rc = -2;
    while (rc == -2) {
        rc = RecvData(timeout);
        if (rc == -2) {
            long block_len = mRecvBuf.ref_header().Blen 
                           - (sizeof(DAQDRecHdr) -sizeof(int));
            rc = RecvReconfig(block_len, timeout);
        }
    }
    return rc;
}

//======================================  Receive frame data into a buffer
long
NDS1Socket::GetFrame(char *buffer, long length) {
    semlock lockit (mux);
    if (mWriterType != FrameWriter) 
	return -1;
    long len = RecvRec(buffer, length, false, -1);
    return len;
}

//======================================  Receive frame name into a buffer
long 
NDS1Socket::GetName(char *buffer, long length) {
    semlock lockit (mux);
    if (mWriterType != NameWriter) 
	return -1;
    return RecvRec(buffer, length, false,  -1);
}

//======================================  Request a stream of frame data
int NDS1Socket::RequestFrames() {
    semlock lockit (mux);
   
    //----------------------------------  Build the request
    ostringstream request;
    request << "start frame-writer ";
    if (mGetAll) {
	request << "all;";
    } 
    else {
	request << "{";
	for (channel_iter i = mRequest_List.begin(); 
	     i != mRequest_List.end() ; i++) {
	    request << "\"" << i->mName << "\"";
	}
	request << "};";
    }
   
    //----------------------------------  Send the request
    int rc = SendRequest(request.str(), mWriter, sizeof(mWriter));
    if (rc) 
	return rc;
    mWriterType = FrameWriter;
   
    //----------------------------------  Read in the offline flag
    int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false);
    if (ldata != sizeof(mOffline)) 
	return ldata;
   
    //----------------------------------  Done, return
    return rc;
}

//======================================  Request file names
int NDS1Socket::RequestNames (wait_time timeout) {
    semlock lockit (mux);
    int rc = SendRequest("start name-writer all;", mWriter, sizeof(mWriter),
			 0, timeout);
    if (rc) 
	return rc;
    mWriterType = NameWriter;
   
    //----------------------------------  Read in the offlne flag
    rc = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
    if (rc != sizeof(mOffline)) 
	return -1;
    return 0;
}

//======================================  Request a stream of channel data
int NDS1Socket::RequestOnlineData (double stride, wait_time timeout) 
{
    semlock lockit (mux);
    //----------------------------------  Build the request

    ostringstream request;
    if (stride < 1.0) {
	request << "start fast-writer ";
    }
    else {
	request << "start net-writer ";
    }
    if (mGetAll) {
	request << "all;";
    } 
    else {
	request << "{";
	for (channel_iter i = mRequest_List.begin() ; i != mRequest_List.end() ; i++) {
	    request << "\"" << i->mName << "\"";
	}
	request << "};";
    }
    if (mDebug)
	cerr << "NDS request = " << request.str() << endl;
   
    //----------------------------------  Send the request
    int rc = SendRequest(request.str(), mWriter, sizeof(mWriter),
			 0, timeout);
    if (mDebug) cerr << mWriter << " = " << CVHex (mWriter, 8) << endl;
    if (rc) {
	return rc;
    }
    mWriterType = DataWriter;
   
    //----------------------------------  Read in the header
    int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
    if (ldata != sizeof(mOffline)) 
	return ldata;
   
    //----------------------------------  Done, return
    return rc;
}

//======================================  Request a stream of channel data
int NDS1Socket::RequestData (unsigned long start, unsigned long duration,
			     wait_time timeout)
{
    //----------------------------------  Any data?
    count_type N = mRequest_List.size();
    if (!N && !mGetAll) {
	cerr << "No channels requested" << endl;
	return 1;
    }

    //----------------------------------  Get unique control of this socket
    semlock lockit (mux);

    //----------------------------------  Scan channel types
    chantype rqType = cUnknown;
    for (count_type i=0; i<N; ++i) {
	chantype cType = mRequest_List[i].mChanType;
	if (cType == rqType || cType == cUnknown) continue;
	if (rqType == cUnknown) {
	    rqType = cType;
	} else {
	    cerr << "Incompatible channel types" << endl;
	    return 2;
	}
    }

    //----------------------------------  Build the request
    ostringstream request;
    switch (rqType) {
    case cOnline:
	request << "start net-writer " << start << " " << duration;
	break;
    case cMTrend:
	request << "start trend 60 net-writer " << start << " " << duration;
	break;
    case cSTrend:
	request << "start trend net-writer " << start << " " << duration;
	break;
    default:
	break;
    }

    //----------------------------------  add the channel list.
    if (mGetAll) {
	request << " all;";
    } else {
	request << " {";
	for (count_type i=0; i<N; ++i) {
	    request << "\"" << mRequest_List[i].mName << "\"";
	}
	request << "};";
    }

    if (mDebug) {
	cerr << "NDS data request = " << request.str() << endl;
    }

    //----------------------------------  Send the request
    int rc = SendRequest(request.str(), mWriter, sizeof(mWriter),
			 0, timeout);
    if (mDebug) cerr << mWriter << " = " << CVHex (mWriter, 8) << endl;
    if (rc) {
	return rc;
    }
    mWriterType = DataWriter;
   
    //----------------------------------  Read in the online status code
    int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
    if (mDebug) cerr << mOffline << endl;
    if (ldata != sizeof(mOffline)) 
	return ldata;
   
    //----------------------------------  Done, return
    return rc;
}

//======================================  Request trend data
int NDS1Socket::RequestTrend (unsigned long start, unsigned long duration,
			      bool mintrend, wait_time timeout)
{
    semlock lockit (mux);

    //----------------------------------  Scan channel types
    count_type N = mRequest_List.size();
    for (count_type i=0; i<N; ++i) {
	switch (mRequest_List[i].mChanType) {
	case cMTrend:
	    if (!mintrend) return 2;
	    break;
	case cSTrend:
	    if (mintrend) return 2;
	    break;
	case cUnknown:
	case cOnline:
	case cRaw:
	    if (mintrend) mRequest_List[i].mChanType = cMTrend;
	    else          mRequest_List[i].mChanType = cSTrend;
	    break;
	default:
	    return 2;
	}
    }

    //----------------------------------  Build the request
    return RequestData(start, duration, timeout);
}

//======================================  Remove a channel from the list
void 
NDS1Socket::RmChannel(const string& chan) 
{
    semlock lockit (mux);
    if (chan == "all") {
	mGetAll = false;
	mRequest_List.clear();
    }
    else {
	DAQC_api::RmChannel(chan);
    }
}

//======================================  Receive proprietary data into buffer
int 
NDS1Socket::RecvData(wait_time timeout) 
{
    //----------------------------------  Read in the data header.
    int rc = RecvRec(reinterpret_cast<char*>(&mRecvBuf.ref_header()), 
		     sizeof(DAQDRecHdr), true, timeout);
    if (rc != sizeof(DAQDRecHdr)) return -1;

    SwapHeader();
    DAQDRecHdr& h = mRecvBuf.ref_header();
    if (mDebug) {
	cerr << "Record Header: BLen=" << h.Blen << " Secs=" << h.Secs 
	     << " GPS=" << h.GPS << " NSec=" << h.NSec << " SeqNum=" 
	     << h.SeqNum << endl;
    }
    long nSecs = h.Secs;
    if (nSecs == -1) return -2;

    //----------------------------------  Calculate the record length.
    int ndata = h.Blen - (sizeof(DAQDRecHdr) - sizeof(int));
    if (ndata == 0) 
	return 0;
    if (ndata  < 0) 
	return -1;
   
    //----------------------------------  Read in the data body
    mRecvBuf.reserve(ndata);
    rc = RecvRec(mRecvBuf.ref_data(), ndata, true, timeout);
    if (rc <= 0) return rc;

    //----------------------------------  Read in the data body
    count_type boff = 0;
    for (channel_iter i=mRequest_List.begin(); i != mRequest_List.end(); i++) {
	i->mBOffset = boff;
	int nbyt = DAQDChannel::datatype_size(i->mDatatype);
	i->mStatus  = i->nwords(nSecs) * nbyt;
	boff += i->mStatus;
    }
    SwapData();
    return rc;
}

//======================================  Receive data into a buffer
struct nds1_chan_status {
    float signal_offset;
    float signal_slope;
    int status;
};

//======================================  Receive data into a buffer
int
NDS1Socket::RecvReconfig(count_type block_len, wait_time maxwait){

    //-----------------------------------  Get number of channels.
    count_type nchannels = block_len / sizeof(nds1_chan_status);
    if (nchannels * sizeof(nds1_chan_status) != block_len) {
	cerr << "Channel reconfigure block length has bad length "
	     << block_len << endl;
	return -1;
    }
    if (nchannels != mRequest_List.size()) return -1;

    //----------------------------------  Loop over channels, read status.
    for (count_type i=0; i < nchannels; i++) {
	DAQDChannel* lp = &mRequest_List[i];
	int dtype;
	if (RecvFloat(lp->mOffset) != sizeof(int)) return -1;
	if (RecvFloat(lp->mSlope)  != sizeof(int)) return -1;
	if (RecvInt(dtype)         != sizeof(int)) return -1;
    }
    return -2;
}

//======================================  Stop data transfer
int NDS1Socket::StopWriter() {
    semlock lockit (mux);
   
    //----------------------------------  Make sure a writer is running.
    if (mWriterType == NoWriter) 
	return -1;
   
    //----------------------------------  Build the request
    ostringstream request;
    request << "kill net-writer " << CVHex (mWriter, 8) << ";";
   
    //----------------------------------  Send the request
    int rc = SendRequest(request.str(), mWriter, 0);
    mWriterType = NoWriter;
   
    return rc;
}

//======================================  Get a list of available channels
int 
NDS1Socket::Times (chantype ty, unsigned long& start, unsigned long& duration, 
		   wait_time timeout) 
{
    semlock lockit (mux);

    //----------------------------------  Build the request
    string request;
    switch (ty) {
    case cSTrend:
	request = "status trend filesys;";
	break;
    case cMTrend:
	request = "status minute-trend filesys;";
	break;
    default:
	request = "status main filesys;";
    }
   
    //----------------------------------  Send the request
    int rc = SendRequest(request, mWriter, sizeof(mWriter), 0, timeout);
    if (mDebug) cerr << mWriter << " = " << CVHex (mWriter, 8) << endl;
    if (rc) {
	return rc;
    }
    mWriterType = DataWriter;
   
    //----------------------------------  Read in the header
    int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
    if (mDebug) cerr << mOffline << endl;
    if (ldata != sizeof(mOffline)) 
	return ldata;
   
    //----------------------------------  Get an empty data block
    rc = RecvRec(reinterpret_cast<char*>(&mRecvBuf.ref_header()), 
		 sizeof(DAQDRecHdr), true, timeout);
    if (rc) {
	if (mDebug) cerr << "times failed" << rc << endl;
	return rc;
    }
    SwapHeader();
    start    = mRecvBuf.ref_header().GPS;
    duration = mRecvBuf.ref_header().Secs;
    //----------------------------------  Done, return
    return rc;
}

//======================================  Send a request, wait for response.
int
NDS1Socket::SendRequest(const string& text, char *reply, long length, 
			long *Size, wait_time timeout) {
    char status[5];
   
    //----------------------------------  Send the request
    if (mDebug) cerr << "Request: " << text << endl;
    int rc = SendRec (text.c_str(), text.size(), timeout);
    if (rc <= 0) {
	if (mDebug) cerr << "send ret1 = " << rc << endl; 
	return rc;
    }
   
    //----------------------------------  Return if no reply expected.
    if (reply == 0) 
	return 0;
   
    //----------------------------------  Read the reply status.
    rc = RecvRec (status, 4, true, timeout);
    if (rc != 4) {
	if (mDebug) cerr << "send ret2 = " << rc << endl; 
	return -1;
    }
    status[4] = 0;
    if (mDebug) cerr << "Status: " << status << endl;
    rc = CVHex(status, 4);
    if (rc) 
	return rc;
   
    //----------------------------------  Read the reply text.
    if (length != 0) {
	rc = RecvRec(reply, length, true, timeout);
	if (rc < 0) {
	    if (mDebug) cerr << "send ret3 = " << rc << endl; 
	    return rc;
	}
	if (rc < length) reply[rc] = 0;
	if (mDebug) cerr << "reply: " << reply << endl;
	if (Size)       *Size = rc;
    }
    return 0;
}

//======================================  Receive data into a buffer
int 
NDS1Socket::RecvRec(char *buffer, long length, bool readall, wait_time maxwait){
    Time stop;
    int 	fileflags;
    char* point = buffer;
    int   nRead = 0;
    int poll = maxwait >= 0;
    if (poll) {
	stop = Now() + Interval (maxwait);
    }
    bool timedout = false;
    do {
	// use select to test for timeout
	if (poll || mAbort) {
	    wait_time   tWait = maxwait;
	    if (mAbort) tWait = 0.1;
	    int nset = socketWait(mSocket, tWait, wm_read);
	    if (nset < 0) {
		perror("NDS1Socket: Error in select()");
		return -12;
	    }
	    if (nset == 0) {
		// timeout?
		if (errno == 0) {
		    if (!mAbort || *mAbort) {
			return -13;
		    }
		}
		// signal
		else {
		    cerr << "NDS1Socket::RecvRec Signal received in select "
			 << string().insert((unsigned int)0, 44, '+') << endl;
		    continue;
		}
	    }
	    // compute how much time is left in timeout
	    if (poll) {
		maxwait = wait_time(stop - Now());
		if (maxwait < 0) {
		    maxwait = 0;
		    timedout = true;
		}
	    }
	    // continue if no data and abort not set
	    if ((nset == 0) && mAbort && !timedout) {
		continue;
	    }
	    /* set socket to non blocking */
	    if ((fileflags = fcntl (mSocket, F_GETFL, 0)) == -1) {
		return -1;
	    }
	    if (fcntl (mSocket, F_SETFL, fileflags | O_NONBLOCK) == -1) {
		return -1;
	    }
	}
	int nB = recv(mSocket, point, length - nRead, 0);
	//cerr << "recv returns with " << nB << endl;
	if (poll || mAbort) {
	    if (mDebug && (nB == 0)) { 
		cerr << "RecvRec with zero length" << endl;
	    }
	    fcntl (mSocket, F_SETFL, fileflags & !O_NONBLOCK);
	}
	if (nB == -1) {
	    if (mDebug)
		cerr << "RecvRec failed with errno " << errno << endl;
	    return -10;
	}
	point += nB;
	nRead += nB;
	if (timedout || (mAbort && *mAbort)) {
	    return -13;
	}
    } while (readall && (nRead < length));
    if (mDebug) cerr << "RecvRec read " << nRead << "/" << length << endl;
    return nRead;
}


//======================================  Send data from a buffer
int 
NDS1Socket::SendRec(const char *buffer, long length, wait_time maxwait) {
    int 	fileflags;
    const char* point = buffer;
    int   nWrite = 0;
    int poll = (maxwait == 0);
   
    bool timedout = false;
    do {
	// use select to test for timeout
	if (poll || mAbort) {
	    int nset;
	    wait_time wstep = mAbort ? 0.1 : maxwait;

	    Time now = Now();
	    // wait

	    nset = socketWait(mSocket, wstep, wm_write);
	    if (nset < 0) {
		perror("NDS1Socket: Error in select()");
		return -12;
	    }
	    else if (nset == 0) {
		// timeout error
		if (!mAbort || *mAbort) {
		    return -13;
		}
	    }
	    // compute how much time is left in timeout
	    if (poll) {
		maxwait = wait_time( now + Interval(maxwait) - Now() );
		if (maxwait < 0) maxwait = 0;
	    }
	    // continue if no data sent and abort not set
	    if ((nset == 0) && mAbort && !timedout) {
		continue;
	    }
	    /* set socket to non blocking */
	    if ((fileflags = fcntl (mSocket, F_GETFL, 0)) == -1) {
		return -1;
	    }
	    if (fcntl (mSocket, F_SETFL, fileflags | O_NONBLOCK) == -1) {
		return -1;
	    }
	}
	int nB = send (mSocket, point, length - nWrite, 0);
	if (poll || mAbort) {
	    fcntl (mSocket, F_SETFL, fileflags & !O_NONBLOCK);
	}
	if (nB == -1) {
	    if (mDebug)
		cerr << "SendRec failed with errno " << errno << endl;
	    return -10;
	}
	point += nB;
	nWrite += nB;
	if (timedout || (mAbort && *mAbort)) {
	    return -13;
	}
    } while (nWrite < length);
    if (mDebug) cerr << "SendRec write " << nWrite << "/" << length << endl;
    return nWrite;
}

//======================================  Wait for a message to arrive
//                                        WaitforData() <= 0 means no data.
int 
NDS1Socket::WaitforData (bool poll) {
    int nset = 0;
    if (poll) nset = socketWait(mSocket,  0.0, wm_read);
    else      nset = socketWait(mSocket, -1.0, wm_read);
    return nset;
}
