/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "frame_name.hh"
#include <iostream>
#include <sstream>
#include <cmath>
#include <cstdio>
#include <unistd.h>
#include <sys/stat.h>

using namespace std;

//=============================================================================
//
//           Static Utility functions
//
//=============================================================================

const char*
ENV_DIGITS(void) {
    return "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_0123456789";
}

//======================================  Resolve all environment variables
std::string
frame_name::resolve_env(const std::string& str) {
    string s = str;
    size_t inx = s.find_first_of('$');
    while (inx != string::npos) {
	size_t endenv = s.find_first_not_of(ENV_DIGITS(), inx+1);
	if (endenv == string::npos) endenv = s.size();
	string var = s.substr(inx+1, endenv-inx-1);
	const char* val = getenv(var.c_str());
	if (val) {
	    s.replace(inx, endenv-inx, val);
	} else {
	    s.erase(inx, endenv-inx);
	}
	inx = s.find('$', inx);
    }
    return s;
}

std::string 
frame_name::make_absolute(const std::string& path) {
    char cwd[1024];
    string r = path;
    if (r[0] != '/' && getcwd(cwd, sizeof(cwd))) {
        r.insert(0, string(cwd) + "/");
    }
    return r;
}

//======================================  Print first N digits of an integer
static int
put_int(string& s, unsigned long d, int count) {
    unsigned long  dig = d%10;
    unsigned long rest = d/10;
    int     N = 0;
    if (rest) N = put_int(s, rest, count);
    if (!count || N < count) s += char('0' + dig);
    return N+1;
}

//======================================  Path function.
static string 
getPath(const std::string& outdir, const Time& start, const Time& end) {
    Time::ulong_t tStart = start.getS();
    Time::ulong_t tEnd   = end.getS();
    if (end.getN()) tEnd++;
    string s;
    string::size_type N = outdir.size();
    for (string::size_type i=0; i<N; ++i) {
	if (outdir[i] != '%') {
	    s += outdir[i];
	} else {
	    int count = 0;
	    char fmt  = outdir[++i];
	    while (fmt >= '0' && fmt < '9') {
		count *= 10;
		count += fmt - '0';
		fmt =  outdir[++i];
	    }
	    switch(fmt) {
	    case 'g':
		put_int(s, tStart, count);
		break;
	    case 'r':
		put_int(s, tStart/int(pow(10.0,count)+0.5), 0);
		break;
	    case 'd':
		put_int(s, tEnd-tStart, count);
		break;
	    default:
		s += fmt;
		break;
	    }
	}
    }
    return s;
}

//======================================  Recursive makedir
static bool
recursive_make_dir(const std::string& dir, int depth=1, int mode=0776) {
    if (dir.empty()) return true;
    if (!access(dir.c_str(), X_OK | W_OK)) return true;
    if (depth < 1) {
	cerr << "make_dir: request to create more than <depth> directories." 
	     << endl;
	return false;
    } 
    string::size_type inx=dir.find_last_of("/");
    if (inx != 0 && inx != string::npos && 
	!recursive_make_dir(dir.substr(0,inx), depth-1, mode)) return false;
    if (!mkdir(dir.c_str(), mode)) return true;
    perror("frame_name::make_dir: mkdir failed");
    return false;
}

//=============================================================================
//
//           frame_name class implementation
//
//=============================================================================

//======================================  Default constructor.
frame_name::frame_name(void)
    : _prefx("frame"), _ext("gwf"), _dt(1), _deep(2), _online(false)
{}

//======================================  Default constructor.
frame_name::frame_name(const std::string& dir_pat, const std::string& pfx_pat, 
		       int dt, const std::string& ext)
  :  _prefx(pfx_pat), _ext(ext), _dt(dt), _deep(2), _online(false)
{
    set_directory(dir_pat);
}

//======================================  Get the directory name
std::string
frame_name::dir_name(const Time& t) const {
    return getPath(_dir, t, t);
}

//======================================  Construct a frame file path
std::string
frame_name::file_path(const Time& t, int dt) const {
    if (online()) return _dir;
    string dirt = dir_name(t);
    ostringstream path;
    if (!dirt.empty()) path << dirt << "/";
    if (!t) {
	path << _prefx;
    } else {
	path << _prefx << "-" << t.getS() << "-";
	if (!dt) path << _dt;
	else     path << dt;
    }
    if (!_ext.empty()) path << "." << _ext;
    return path.str();
}

//======================================  make the directory.
bool
frame_name::make_dir(const std::string& dir, int deep, int mode) {
    if (!deep) deep = _deep;
    return recursive_make_dir(dir, deep, mode);
}

//======================================  rename the temporary file
bool
frame_name::move_to_path(const string& temp, const Time& t0, int dt) {
    string path = file_path(t0, dt);
    return rename(temp.c_str(), path.c_str()) == 0;
}

//======================================  Set directory pattern 
void 
frame_name::set_directory(const std::string& dir_pat) {
    _dir = resolve_env(dir_pat);
    if (_dir.empty()) return;

    //----------------------------------  Remove trailing '/'s
    string::size_type inx = _dir.size();
    while (--inx && _dir[inx] == '/') _dir.erase(inx,1);

    _online = (_dir.substr(0,8) == "/online/");
}

//======================================  Set directory pattern
void
frame_name::set_depth(int deep) {
    _deep = deep;
}

//======================================  Set default frame length
void 
frame_name::set_dt(int dt) {
    _dt = dt;
}

//======================================  Set frame file extension
void 
frame_name::set_extension(const std::string& ext) {
    _ext = ext;
}

//======================================  Set frame file prefix
void 
frame_name::set_prefix(const std::string& pfx) {
    _prefx = pfx;
}

//======================================  Split up a template
void
frame_name::split(const std::string& pat) {
    string tpat = pat;
    string::size_type inx = tpat.find_last_of('/');
    if (inx == 7 && tpat.substr(0,8) == "/online/") {
	set_directory(tpat);
	return;
    }
    
    if (inx != string::npos) {
	set_directory(tpat.substr(0, inx));
	tpat.erase(0, inx+1);
    }
    inx = tpat.find_last_of('.');
    if (inx != string::npos) {
	set_extension(tpat.substr(inx+1));
	tpat.erase(inx);
    }
    inx = tpat.find_first_of('-');
    if (inx != string::npos) inx = tpat.find_first_of('-', inx+1);
    set_prefix(tpat.substr(0, inx));
}

//======================================  Construct a temporary file path
std::string
frame_name::temp_path(const Time& t, int dt) const {
    ostringstream path;
    path << dir_name(t) << "/." << _prefx << "-" << t.getS() << "-";
    if (!dt) path << _dt;
    else     path << dt;
    path << ".tmp";
    return path.str();
}
