/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    DMT frame data access implementation
//
#include "Dacc.hh"
#include "Channel.hh"
#include <stdio.h>
#include <iostream>

using namespace std;

using FrameCPP::IFrameStream;

//  dTcontig is the maximum acceptable discontinuity in nsec.
static const int dTcontig = 1;

//======================================  Error code translation.
const char*
Dacc::getMsgText(int err) {
    switch (err) {
    case 0:
	return "Successful completion.";
    case -1:
	return "Frame start not contiguous to previous data.";
    case -2:
	return "Sample rate isn't compatible with previous data.";
    case -3:
	return "Requested data not found in current frame.";
    case -4:
	return "Error reading frame.";
    case -5:
	return "Frame data are not self-consistent.";
    case -6:
	return "TSeries is not allocated.";
    case -7:
	return "Unsupported data type.";
    case -8:
	return "Signal received while reading.";
    case -9:
	return "Invalid data in structure.";
    default:
	return "Unknown error code";
    }
    return "Unknown error code";
}

//======================================  Default (Null) constructor.
Dacc::Dacc(void) 
  : mFillTime(0), mOffset(0.0), mStride(0.0), mFillRun(false), mNoWait(false),
    mIgnoreMissChan(false)
{
}

//======================================  Construct and connect to source.
Dacc::Dacc(const std::string& Source, const Time& STime) 
  : mFillTime(0), mOffset(0.0), mStride(0.0), mFillRun(false), mNoWait(false),
    mIgnoreMissChan(false)
{
    addFile(Source);
    DaccIn::open();
    while (seek(STime) == -8);
}

//======================================  Construct and connect to source.
Dacc::Dacc(IFrameStream* reader) 
  : DaccIn(reader), mFillTime(0), mOffset(0.0), mStride(0.0), 
    mFillRun(false), mNoWait(false), mIgnoreMissChan(false)
{
}

//======================================  Destructor.
Dacc::~Dacc() {
    DaccIn::close();
    mChanList.clear();
}

//======================================  Add a channel to the request list.
void 
Dacc::addChannel(const std::string& Name, int Decimate, TSeries **TSptr) {
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
        if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    Channel newchan(Name, TSptr, Decimate);
    if (getDebug()) newchan.setDebug(true);
    mChanList.push_back(newchan);
}

//======================================  Add a channel to the request list.
void 
Dacc::addFSeries(const std::string& Name, FSeries **FSptr) {
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) {
        if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    Channel newchan(Name, FSptr);
    if (getDebug()) newchan.setDebug(true);
    mChanList.push_back(newchan);
}

//======================================  Add a channel to the request list.
void 
Dacc::addRaw(const std::string& Name, int Decimate, TSeries **TSptr) {
    chan_iter i=findChannel(Name, Channel::kRaw);
    if (i != mChanList.end()) {
        if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    Channel newchan(Name, TSptr, Decimate, Channel::kRaw);
    if (getDebug()) newchan.setDebug(true);
    mChanList.push_back(newchan);
}

//======================================  Add a channel to the request list.
void 
Dacc::addProcessed(const std::string& Name, int Decimate, TSeries **TSptr) {
    chan_iter i=findChannel(Name, Channel::kProcessed);
    if (i != mChanList.end()) {
        if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    Channel newchan(Name, TSptr, Decimate, Channel::kProcessed);
    if (getDebug()) newchan.setDebug(true);
    mChanList.push_back(newchan);
}

//======================================  Add a channel to the request list.
void 
Dacc::addSimulated(const std::string& Name, int Decimate, TSeries **TSptr) {
    chan_iter i=findChannel(Name, Channel::kSimulated);
    if (i != mChanList.end()) {
        if (getDebug()) cout << "Existing channel " << Name 
			     << " entry replaced." << endl;
	mChanList.erase(i);
    }
    Channel newchan(Name, TSptr, Decimate, Channel::kSimulated);
    if (getDebug()) newchan.setDebug(true);
    mChanList.push_back(newchan);
}

//======================================  See if channel was requested.
bool 
Dacc::isChannelRead(const std::string& Name) const {
    return findChannel(Name) != mChanList.end();
}

//======================================  List requested channels
ostream&
Dacc::list(ostream& out) const {
    char buf[2048];
    sprintf (buf, "Channel                  Decimation  Latest-Time   Pointer\n");
    out << buf;
    for (const_chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
        sprintf (buf, "%-25s %9i %12li  %08zx \n", i->getName(), i->getDecim(),
		 i->getLast().getS(), (ptrdiff_t)i->refSeries());
        out << buf;
    }
    return out;
}

//======================================  Set the Nowait flag
void 
Dacc::setNoWait(bool now) {
    mNoWait = now;
}

//======================================  Set the default stride.
void 
Dacc::setStride(Interval Dt) {
    mStride=Interval(Dt);
}

//======================================  Find channel in request list.
Dacc::const_chan_iter 
Dacc::findChannel(const string& Name, Channel::ChanType t) const {
    for (const_chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
        if (i->Equal(Name, t)) return i;
    }
    return mChanList.end();
}

Dacc::chan_iter 
Dacc::findChannel(const string& Name, Channel::ChanType t) {
    for (chan_iter i=mChanList.begin(); i != mChanList.end(); i++) {
        if (i->Equal(Name, t)) return i;
    }
    return mChanList.end();
}

//======================================  Remove channel from request list.
void 
Dacc::rmChannel(const std::string& Name) {
    chan_iter i=findChannel(Name);
    if (i != mChanList.end()) mChanList.erase(i);
}

//======================================  Read a valid frame if necessary
int 
Dacc::synch(void) {
    //---------------------------------- Read a new frame if none in memory.
    if (!haveFrame()) {
        if (!waitData(mNoWait)) {
	    return -8;
	} else if (nextFrame()) {
	    cerr  << "synch: Unable to fetch next frame"  << endl;
	    return -4;
	} else {
	    mOffset = 0.0;
	}
    }
    //----------------------------------  Have we read past end of frame?
    else if (mOffset >= getDt()) {
	Time tStream = getCurrentTime();
        endFrame();
        if (!waitData(mNoWait)) {
	    return -8;
	} else if (nextFrame()) {
	    cerr  << "synch: Unable to fetch next frame"  << endl;
	    return -4;
	} else if (tStream > getTime() && tStream < getTime() + getDt()) {
	    mOffset = tStream - getTime();
	} else {
	    mOffset = 0.0;
	}
    }
    return 0;
}

//======================================  Fill requested series containers.
//    Error codes:
//      0  Successful completion
//     -1  Frame start not contiguous to previous data
//     -2  Sample rate isn't compatible with previous data.
//     -3  Requested data not found in current frame 
//     -4  Error reading frame
//     -5  Frame data are not self-consistent.
//     -6  TSeries is not allocated.
//     -7  Unsupported data type
//     -8  Interrupter by signal or no data and nowait.
//     -9  Invalid data (NaN, Inf or Unnormalized).
int 
Dacc::fillData(Interval Stride, bool start) {

    //----------------------------------  Set up to fill in data.
    int rc = 0;
    if (start || !mFillRun) {
	mFillTime = Time(0);
        //----------------------------------  Get/Validate stride
	if (double(Stride) != 0)       mFillStride = Stride;
	else if (double(mStride) != 0) mFillStride = mStride;
	else                           mFillStride = Interval(1.0);
	zeroChans(mFillStride);

	//----------------------------------  Remember the start time
	rc = synch();
	if (rc) return rc;
	mFillTime = getCurrentTime();

	mFillRun = true;
    }

    //----------------------------------  Loop over frames
    Interval dT(0.0);
    while (mFillOffset < mFillStride && !rc) {

        //------------------------------  Get a new frame if necessary
        rc = synch();
	if (rc) break;

	//------------------------------  Get raw data, Time residual
        dT = getDt() - mOffset;
	if (mFillOffset + dT > mFillStride) dT = mFillStride - mFillOffset;
	if (dT <= Interval(0.0)) continue;

	//------------------------------  Fill data into the channels.
	rc = fillChans(mOffset, dT);
	if (rc == -1 || rc == -8) continue;


	//------------------------------  Bump the time pointer.
	mOffset += dT;
	if (DaccIn::isOnline() && mOffset >= getDt()) endFrame();
	if (getDebug()>1) cout << "iStr/tStride/mOffset/dT =" << mFillOffset 
			       << "/" << mFillStride << "/" << mOffset 
			       << "/" << dT << endl;
	mFillOffset += dT;
    }
    mFillRun = (rc == -8);
    if (rc && !mFillRun) mFillTime = Time(0);
    return rc;
}

//======================================  Fill data from a single frame
void
Dacc::zeroChans(Interval dT) {
    for (chan_iter i=mChanList.begin() ; i != mChanList.end() ; i++) {
        i->allocate(dT);
    }
    mFillOffset = Interval(0.0);
}

//======================================  Fill data from a single frame
int
Dacc::fillChans(Interval Offset, Interval dT) {
    int rc = 0;

    //----------------------------------  Fill all channels with this frame
    for (chan_iter i=mChanList.begin(); i != mChanList.end(); ++i) {
	const char* name = i->getName();
	Channel::ChanType typ = i->getType();
	int ordered = 0;

	//------------------------------  Look for Adc data
	if (typ == Channel::kUnknown || typ == Channel::kRaw) {
	    fradcdata_pointer adc;
	    ordered = DaccIn::findAdcOrdered(name, adc);
	    if (ordered) rc = i->FillSeries(adc, getTime(), Offset, dT);
	}

	//------------------------------  Look for Proc data
	if (!ordered && (typ == Channel::kUnknown || 
			 typ == Channel::kProcessed ||
			 typ == Channel::kFSeries)) {
	    frprocdata_pointer proc;
	    ordered = DaccIn::findProcOrdered(name, proc);
	    if (ordered) rc = i->FillSeries(proc, getTime(), Offset, dT);
	}

	//------------------------------  Look for sim data
	else if (!ordered && typ == Channel::kSimulated) {
	    frsimdata_pointer sim;
	    ordered = DaccIn::findSimOrdered(name, sim);
	    if (ordered) rc = i->FillSeries(sim, getTime(), Offset, dT);
	}
	if (rc) return rc;

	//------------------------------  Reorder if order < 0.  
	chan_iter j = i;
	if ( ordered < 0 && i != mChanList.begin()) {
	    chan_iter save = i--;
	    j = mChanList.insert(i, *save);
	    mChanList.erase(save);
	    if (getDebug() > 5) cout << "Reorder channels " 
				     << i->getName() << " <-> " 
				     << j->getName() << endl;
	}

	if (ordered) {
	    if (!mFillOffset) j->reserve(mFillStride);
	}

	//------------------------------  Fill the channel series.
	else if (!mIgnoreMissChan) {
	    cerr << "fillData: Channel " << name << " not found" << endl;
	    return -3;
	}
    }
    return rc;
}

//======================================  Flush the specified data
int 
Dacc::flush(Interval Stride) {

    //----------------------------------  Get/Validate stride
    if (double(Stride) <= 0) {
        endFrame();
	return 0;
    }

    //----------------------------------  Loop over frames
    Interval dT(0.0);
    Time TCur(0,0);
    for (Interval iStr(0.0) ; iStr<Stride ; iStr += dT) {

        //--------------------------------  Get a new frame if necessary
        if (!haveFrame()) {
	    if (nextFrame()) return -4;
	    mOffset = 0.0;
	}

	//----------------------------------  Make sure this is contiguous.
	if (!TCur) {
	    TCur = getCurrentTime();
	} else if (!Almost(TCur, getCurrentTime(), dTcontig)) {
	    if (getDebug()) {
	        cout << "Break in Dacc::flush: Previous, current times: "
		     << TCur << ", " << getCurrentTime() << endl;
	    }
	    break;
	}

	//---------------------------------  Get raw data, Time residual
        dT = getDt() - mOffset;
	if (iStr + dT > Stride) dT = Stride-iStr;
	if (dT <= Interval(0.0)) dT = 0.0;

	//---------------------------------  Bump the time pointer.
	mOffset += dT;
	TCur = getCurrentTime(); // Avoid error accumulation
	if (mOffset >= getDt()) endFrame();
    }
    return 0;
}

//======================================  Open a file for input.
int
Dacc::seek(Time STime) {
    int rc = synch();
    while (!rc && STime > getCurrentTime()) {
        Interval Delta_t = STime - getCurrentTime();
	if (double(Delta_t) < 1e-6) break;
	rc = flush(Delta_t);
    }
    return rc;
}

//======================================  Get a reference to the named Channel
const TSeries* 
Dacc::refData(const std::string& name) const {
    const_chan_iter i=findChannel(name); 
    if (i != mChanList.end()) return i->refSeries();
    return (const TSeries*) 0;
}

TSeries* 
Dacc::refData(const string& name) {
    chan_iter i=findChannel(name);
    if (i != mChanList.end()) return i->refSeries();
    return (TSeries*) 0;
}

const FSeries* 
Dacc::refFData(const std::string& name) const {
    const_chan_iter i=findChannel(name); 
    if (i != mChanList.end()) return i->refFSeries();
    return (const FSeries*) 0;
}

FSeries* 
Dacc::refFData(const std::string& name) {
    chan_iter i=findChannel(name);
    if (i != mChanList.end()) return i->refFSeries();
    return (FSeries*) 0;
}

void 
Dacc::setChannelFlag(const string& chan, Channel::ChanFlags flag, bool state) {
    chan_iter i=findChannel(chan);
    if (i != mChanList.end()) {
	i->setFlag(flag, state);
    }
}
