/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef GENERATOR_VCO_HH
#define GENERATOR_VCO_HH

#include "generator/DataSource.hh"

namespace generator {

  /**  %VCO is a voltage controlled oscillator, \e i.e. a sinusoid signal 
    *  generator for which the frequency is controlled by an input signal. 
    *  The parameters are:
    *  * \a Vin: input signal (frequency control)
    *  * \a A: sinusoid amplitude.
    *  * \a F0: output frequency when \a Vin == 0.
    *  * \a gain: change in frequency per unit change in Vin. 
    *  * \a Phi: starting phase.
    *
    *  @memo %Sine waveform generator.
    *  @version 1.1; last modified January 30, 2018
    *  @author  John Zweizig
    */
  class VCO : public DataSource {
  public:
    /**  Default constructor.
      *  \brief Construct an empty %VCO object.
      */
    VCO(void);

    /**  Construct a %VCO object and set the parameters.
      *  \brief %Parameter constructor
      *  @param vin  Frequency control input.
      *  @param Amp  Sinusoid amplitude.
      *  @param F0   Frequency when vin == 0.
      *  @param gain Change in frequency per unit change in vin.
      *  @param Phi  Sinusoid phase at \c t=to.
      */
    VCO(const TSeries& vin, double Amp, double F0, double gain, double Phi);

    /**  %VCO Destructor.
      *  \brief Destructor
      */
    ~VCO(void);

    /**  Make an exact copy of a %VCO instance.
      *  @return Pointer to copied object.
      */
    VCO* clone(void) const;

    /**  Return a constant string with the data source type name ("VCO").
      *  @return constant pointer to the character string "VCO".
      */
    const char* getDataType(void) const;
  private:

    /**  Generate the data.
     */
    void getSeries(const Time& t0, Interval dT, int N, gen_sample_type* data);
  private:
    bool mRun;
    double mPhi;
    Time mT0;
  };

  //====================================  Inline methods
  inline const char*
  VCO::getDataType(void) const {
    return "VCO";
  }

} // namespace generator

#endif  // GENERATOR_SINGAUSS_HH
