/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "VCO.hh"
#include "DVecType.hh"
#include <iostream>
#include <cmath>

using namespace std;

namespace generator {

   //===================================  Default constructor
   VCO::VCO(void) 
      : DataSource(kContinuous), mRun(false), mPhi(0)
   {
      TSeries temp;
      setParameter("Vin",  temp);
      setParameter("A",    1.0);
      setParameter("F0",   0.0);
      setParameter("gain", 1.0);
      setParameter("Phi0", 0.0);
   }

   //===================================  Data constructor
   VCO::VCO(const TSeries& vin, double Amp, double F0, double gain, double Phi)
      : DataSource(kContinuous), mRun(false), mPhi(0)
   {
      TSeries temp;
      setParameter("Vin",  temp);
      setParameter("A",    Amp);
      setParameter("F0",   F0);
      setParameter("gain", gain);
      setParameter("Phi0", Phi);
   }

   //===================================  Destructor
   VCO::~VCO(void) {
   }

   //===================================  Clone method
   VCO*
   VCO::clone(void) const {
      return new VCO(*this);
   }

   //===================================  Generate sine time-series
   void
   VCO::getSeries(const Time& t0, Interval dT, int N, gen_sample_type* d) {
      TSeries Vin = getSeriesParam("Vin").extract(t0, dT*double(N));
      if (Vin.getTStep() != dT) {
	 throw runtime_error("VCO: Mismatched time steps");
      }

      //--------------------------------  Get the parameters
      double twopi = 2*M_PI;
      double Ampl  = getNumeric("A");    // Amplitude in user units
      double f0    = getNumeric("F0");   // Offset frequency in Hz
      double gain  = getNumeric("gain"); // Gain in Hz/V
      if (!mRun) {
	 mPhi = getNumeric("Phi0") * twopi / 360.0;
	 mRun = true;
      }

      //-------------------------------  calculate delta-phi
      Vin.Convert(DVector::t_double);
      Vin *= gain * twopi * dT;
      Vin += twopi * f0 * dT;
      const DVectD& dPhi(dynamic_cast<const DVectD&>(*Vin.refDVect())); 
      for (int i=0; i<N; i++) {
	 d[i] += Ampl * sin(mPhi);
	 mPhi += dPhi[i];
      }
      if (mPhi > twopi) mPhi -= long(mPhi / twopi) * twopi;
   }

} // namespace generator
