#ifndef GENERATOR_GENPARAM_HH
#define GENERATOR_GENPARAM_HH

#include <iosfwd>
#include <string>
#include "TSeries.hh"

namespace generator {

  /**  The %GenParam class holds a generation parameter. The parameters
    *  may have string or numeric values or may take fixed or random values.
    *  Random parameters are reset by calls to \c sampleParameter() according
    *  to one of the following distributions:
    *
    *  <table>
    *  <tr><td>Name </td>
    *     <td> Distribution </td>
    *  </tr>
    *  <tr><td>\c step(\<min\>,\<max\>,\<inc\>) </td>
    *     <td> Step from \c \<min\> to \c \<max\> by \c \<inc\> </td>
    *  </tr>
    *  <tr><td>\c xstep(<min>,<max>,<fact>) </td>
    *     <td> Step from \c \<min\> to \c \<max\> by factors of \c \<fact\></td>
    *  </tr>
    *  <tr><td>\c flat(<min>,<max>) </td>
    *     <td>Flat distribution in range \c \<min\> - \c \<max\> </td>
    *  </tr>
    *  <tr><td>\c gauss(<mean>,<sigma>) </td>
    *     <td>Gaussian with mean \c \<mean\> and sigma \c \<sigma\> </td>
    *  </tr>
    *  <tr><td>\c power(N,<min>,<max>) </td>
    *     <td>\f${dN \over dx}\sim x^{N}\f$ in range \c \<min\> -\c \<max\></td>
    *  </tr>
    *  <tr><td>\c exp(b,<min>,<max>) </td>
    *     <td>\f${dN \over dx} \sim e^{-b}\f$ in range \c \<min\> - \c \<max\></td>
    *  </tr>
    *  </table>
    *
    *  @memo Generation parameter class.
    *  @author John Zweizig
    *  @version 1.1; Last modified April 22, 2003
    */
  class GenParam {
  public:
      /** parameter type enumerator.
       */
      enum ParType {
	kString,  ///< String parameter
	kConst,   ///< Numeric constant parameter
	kDistrib, ///< Distribution function parameter
	kSeries   ///< Series parameter
      };

      /**  Distribution type enumerator.
       */
      enum DistType { 
	kStep,   ///< Step distribution
	kGeom,   ///< Geometric step (xstep) distribution.
	kFlat,   ///< Flat flat random sequence
	kGauss,  ///< Gaussian random sequence
	kPower,  ///< Power law random sequence
	kExp     ///< Exponential decay random sequence.
      };

  public:
      /**  Default constructor.
        *  @memo Default constructor.
	*/
      GenParam(void);

      /**  Destroy a parameter object
        *  @memo Destructor.
        */
      ~GenParam(void);

      /**  Dump the parameter type and current value. In the case of 
        *  random distributions, the distribution type and parameters
	*  are also given.
        *  @memo Dump the parameter type and value.
	*  @param out Standard output stream to receive dump text.
	*  @return Reference to the output stream.
        */
      std::ostream& dump(std::ostream& out) const;

      /**  Put the parameter definition string to the output stream.
        *  @memo Dump the parameter definition string.
	*  @param out Standard output stream. 
	*  @return Reference to the output stream.
        */
      std::ostream& dumpDefinition(std::ostream& out) const;

      /**  Get the distribution identifier from the name.
        *  @memo Get the named distribution ID.
	*  @param name Distribution name to look up.
	*  @return ID of named distribution.
	*/
      static DistType getDistID(const std::string& name);

      /**  Get the name of the specified distribution ID.
        *  @memo Get name of distribution \a id.
	*  @param id Distribution identifier.
	*  @return Constant pointer to ditribution name string.
        */
      static const char* getDistName(DistType id);

      /**  Get a numeric value of the parameter. The value of string 
        *  parameters is converted to a numeric value.
	*  @memo Get numeric parameter value.
	*  @return Numeric value of the parameter.
        */
      double getNumeric(void) const;

      /**  Get the parameter type of this parameter.
        *  @memo Get parameter type.
	*  @return Parameter type enumeration.
	*/
      ParType getParType(void) const;

      /**  Get a string value of the class parameter.
        *  @memo Get parameter string value.
	*  @return Constant pointer to parameter value string.
	*/
      const char* getString(void) const;

      /**  Get a string value of the class parameter.
        *  @memo Get parameter string value.
	*  @return Constant pointer to parameter value string.
	*/
      const TSeries& getSeries(void) const;

      /**  Test whether parameter has a constant value.
        *  @memo Test whether parameter is constant.
        *  @return true if parameter is constant (not a distribution).
        */
      bool isConst(void) const;

      /**  Test whether parameter steps over specified values.
        *  @memo Test if parameter is step distribution.
	*  @return true if parameter is a step distribution.
        */
      bool isStep(void) const;

      /**  Test whether stepping parameter is at the start (indicates 
        *  overflow occurred in last step).
	*  @memo Test if at start of step.
	*  @return true if a start of step.
        */
      bool atStart(void) const;

      /**  Generate a new value for a randomly generated parameter. If 
        *  the parameter is string or numeric constant type, the current
	*  value of the parameter is returned.
	*  @memo Generate parameter according to random distribution.
	*  @return Random  parameter value.
	*/
      double sampleParameter(void);

      /**  Specify a distribution for the random parameter. The distribution
        *  is specified in the form \c "type(p1[[,p2],p3])" .
        *  @memo Set the distribution type.
	*  @param value Distribution definition string.
        */
      void setDistribution(const std::string& value);

      /**  Assign the parameter a numeric value and set the parameter type 
        *  to \c kConst.
        *  @memo Assign parameter a numeric value.
	*  @param value Value to assign to parameter.
	*/
      void setParameter(double value);

      /**  Assign the parameter a string value and set the parameter type
        *  to \c kString .
        *  @memo Assign parameter a string value.
	*  @param value String value to assign to parameter.
	*/
      void setParameter(const std::string& value);

      /**  Assign the parameter a series name and set the parameter type
        *  to \c kSeries.
        *  @memo Assign parameter a string value.
	*  @param TSeries String value to assign to parameter.
	*/
      void setParameter(const TSeries& value);

      /**  Specify a parameter series source name.
        *  \brief Set the series source name.
	*  @param source Source name.
        */
      void setSeries(const std::string& source);

  private:
      ParType     mType;
      std::string mStrValue;
      double      mNumValue;
      DistType    mDistr;
      double      mDistPar[3];
      TSeries     mSeries;
  };


  //====================================  Inline methods
  inline bool 
  GenParam::isConst(void) const {
      return mType != kDistrib;
  }

  inline bool 
  GenParam::isStep(void) const {
      return !isConst() && mDistr == kStep;
  }

  inline bool 
  GenParam::atStart(void) const {
      return isStep() && (mNumValue == mDistPar[0]);
  }

  inline GenParam::ParType
  GenParam::getParType(void) const {
    return mType;
  }
  
  inline const char* 
  GenParam::getString(void) const {
      return mStrValue.c_str();
  }

  inline const TSeries& 
  GenParam::getSeries(void) const {
      return mSeries;
  }

  inline double 
  GenParam::getNumeric(void) const {
      return mNumValue;
  }

} // namespace generator

#endif  // GENERATOR_GENPARAM_HH
