/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef GENCHAN_HH
#define GENCHAN_HH

#include "DataSource.hh"
#include "autopipe.hh"
#include <string>
#include <vector>
#include <memory>

class Pipe;

namespace generator {

   class GenDet;

   /**  The DMT generator %GenChan class combines data from one or more data
    *  sources to produce a simulated data channel. Data from multiple 
    *  sources are optionally filtered and then added to the response series 
    *  each time a new source stride is processed with generate() .
    *  @memo %Channel definition and generation class.
    *  @author J. Zweizig
    *  @version $Id: GenChan.hh 8011 2018-02-26 03:05:35Z john.zweizig@LIGO.ORG $
    */
   class GenChan {
   private:
      /**  The chanComp class stores a component source specification for 
       *  the channel. The channel data is generated from a sum of all
       *  the (appropriately filtered) channel components.
       */
      class chanComp {
      public:
	 /**  Construct a channel component from a Source (specified by a source
	   *  ID) and an optional filter.
	   *  @memo Define a channel component.
	   *  @param ID source identifier.
	   *  @param resp Pointer to a filter Pipe.
	   */
	 chanComp(int ID, const Pipe* resp=0);

	 /**  Copy a channel component object.
	   *  @memo Copy constructor.
	   *  @param x %Channel component to be copied.
	   */
	 chanComp(const chanComp& x);

	 /**  Destroy a channel component object.
	   *  @memo Destructor.
	   */
	 ~chanComp(void);

	 /**  Assign the contents the specified  %chanComp into this.
	   *  @memo Assignment operator.
	   *  @param x Reference to chanComp to be copied.
	   *  @return Reference to the modified object.
	   */
	 chanComp& operator=(const chanComp& x);

	 /**  Get the identifier of the component source.
	   *  @memo Get source ID.
	   *  @return Source identifier.
	   */
	 int getSrcID(void) const;

	 /**  Get the latest time for which the source data are valid.
	   *  @memo Get last valid time.
	   *  @return Last valid time for this component.
	   */
	 Time getLastValid(void) const;

	 /**  Get the filtered source data (response) time series for this 
	   *  component.
	   *  @memo Get component time series.
	   *  @param vec Data source pointer vector.
	   *  @return Component time series.
	   */
	 TSeries getResponse(const std::vector<DataSource*>& vec);

	 /**  Set the latest time for which the filtered source data were
	   *  copied into the channel data.
	   *  \brief Record time of the last data used.
	   *  \param t Last valid time for this component.
	   */
	 void last_valid(const Time& t);

	 /**  Set the component response filter with a clone of the specified
	   *  filter Pipe.
	   *  @memo Set the filter.
	   *  @param resp Reference to the filter to be cloned.
	   */
	 void setResponseFilter(const Pipe& resp);

      private:
	 int  mSrcID;
	 Time mLastSrcData;
	 Time mLastValid;
	 auto_pipe mRespFilter;
      };

      /**  Source list data container type
        */
      typedef std::vector<chanComp> SrcList;

      /**  Source list iterator type
        */
      typedef SrcList::iterator src_iter;

      /**  Source list constant iterator type
        */
      typedef SrcList::const_iterator const_src_iter;

   public:

      /**  Default channel constructor.
        *  @memo Default constructor.
	*/
      GenChan(void);

      /**  %Channel data costructor.
        *  @memo Data constructor.
	*  @param chan  %Channel name string pointer.
	*  @param detid %Channel detector ID.
	*  @param daq   %Channel digitization filter.
	*/
      GenChan(const char* chan, int detid=0, const Pipe* daq=0);

      /**  Add an optionally filtered source to the channel component list.
        *  @memo Add a component
	*  @param src Component source ID.
	*  @param response Reference to the component response filter.
	*  @return Component number of the added source.
	*/
      int addComponent(int src, const Pipe& response);

      /**  Add an optionally filtered source to the channel component list.
        *  @memo Add a component
	*  @param src Component source ID.
	*  @param response Optional pointer to the component response filter.
	*  @return Component number of the added source.
	*/
      int addComponent(int src, const Pipe* response=0);

      /**  Generate the current channel data by filtering source data as
        *  specified in the configuration and summing all filtered sources.
	*  \brief Senerate channel data.
	*  \param det Channel detector information.
	*  \param vec vector of sources.
	*  \return <0 if data have been skipped.
	*/
      int generate(const GenDet& det, const std::vector<DataSource*>& vec);

      /**  Get the channel name
        *  @memo Get channel name.
	*  @return constent pointer to the channel name string.
	*/
      const char* getChannel(void) const;

      /**  Get the detector ID.
        *  @memo Detector ID.
	*  @return identifier of the detector that the channel measures.
	*/
      int getDetID(void) const;

      /**  Get the earliest time for which generated response data are defined.
        *  @memo Get earliest time.
	*  @return Start time of generated data for this channel.
	*/
      Time getEarly(void) const;

      /**  Get the latest time for which generated response data are defined.
        *  @memo Get latest time.
	*  @return End time of generated data for this channel.
	*/
      Time getLatest(void) const;

      /**  Get a reference to the units string.
        *  \brief get the unit string.
	*  \return reference to the unit string.
	*/
      const std::string& getUnits(void) const;

      /**  Release the response data up to a specified time.
        *  @memo Release response data.
	*  @param t Time of first sample to be retained.
	*/
      void release(const Time& t);

      /**  Return the expected response to a specified signal.
        *  @memo Get channel response time series.
	*  @return Constant reference to the channel response time series.
	*/
      const TSeries& response(void) const;

      /**  Set the channel name;
        *  @memo Set the channel name.
	*  @param chan Channel name.
	*/
      void setChannel(const char* chan);

      /**  Set the detector ID for this channel.
        *  @memo  Set the detector ID.
	*  @param det Detector ID.
	*/
      void setDetID(int det);

      /**  The specified filter is cloned and used for dewhitening the noise
        *  @memo Set the dewhitening filter.
	*  @param chan Dewhitening filter.
	*/
      void setDaqChain(const Pipe* chan);

      /**  Set the debugging printout level.
        *  @memo Set the debug level.
	*  @param lvl New debug printout level.
	*/
      void setDebug(int lvl);

      /**  Set the channel sample time \c (1/\<sample_rate\>)
        *  @memo Set the sample time.
	*  @param dt Sample time 
	*/
      void setSample(Interval dt);

      /**  Set the channel sample time \c (1/\<sample_rate\>)
        *  @memo Set the sample time.
	*  @param dt Sample time 
	*/
      void setUnits(const std::string& units);

   private:
      std::string mName;
      int         mDetID;
      SrcList     mSourceList;
      TSeries     mRawResp;
      Time        mRawValid;
      auto_pipe   mDaqChain;
      TSeries     mResponse;
      int         mDebug;
      Interval    mSample;
      std::string mUnits;
   };

   //====================================  return the channel name
   inline int 
   GenChan::chanComp::getSrcID(void) const {
      return mSrcID;
   }

   inline Time 
   GenChan::chanComp::getLastValid(void) const {
      return mLastValid;
   }

   inline const char* 
   GenChan::getChannel(void) const {
      return mName.c_str();
   }

   inline int 
   GenChan::getDetID(void) const {
      return mDetID;
   }

   inline Time 
   GenChan::getEarly(void) const {
      return mResponse.getStartTime();
   }

   inline Time 
   GenChan::getLatest(void) const {
      return mResponse.getEndTime();
   }

   inline const std::string&
   GenChan::getUnits(void) const {
      return mUnits;
   }

   inline const TSeries&
   GenChan::response(void) const {
      return mResponse;
   }
 
} // namespace generator
#endif
