/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "GenChan.hh"
#include "Pipe.hh"
#include <string>
#include <iostream>

using namespace std;

namespace generator {

  //====================================  Default channel constructor
  GenChan::GenChan(void)
    : mDetID(0), mDebug(0), mSample(1./16384.0)
  {
  }

  //====================================  Channel data constructor
  GenChan::GenChan(const char* chan, int detid, const Pipe* daq)
    : mName(chan), mDetID(detid), mDebug(0), 
      mSample(1./16384.0)
  {
      setDaqChain(daq);
  }

  //====================================  Add a data source
  int
  GenChan::addComponent(int id, const Pipe& p) {
      return addComponent(id, &p);
  }

  //====================================  Add a data source
  int
  GenChan::addComponent(int id, const Pipe* p) {
      int rc = mSourceList.size();
      mSourceList.push_back(chanComp(id, p));
      return rc;
  }

  //====================================  Generate data for specified interval
  int
  GenChan::generate(const GenDet& d, const std::vector<DataSource*>& sv) {
      int rc = 0;
      if (mDebug) cout << "Calculate response for channel: " << mName << endl;
      
      //--------------------------------  Loop over component data sources
      Time tLatest(0);
      for (src_iter i=mSourceList.begin(); i<mSourceList.end(); ++i) {
	  if (mDebug) {
	      int src = i->getSrcID();
	      cout << "Adding data from source " << sv[src]->getName() << endl;
	  }
	  TSeries sData(i->getResponse(sv));
	  if (sData.empty()) continue;
	  if (mDebug) cout << "Source data from " << sData.getStartTime() 
			   << " length: " << sData.getInterval() 
			   << " units: " << sData.getUnits() << endl;
	  Time tNext  = sData.getEndTime();
	  Time sStart = sData.getStartTime();
	  Time t_i    = i->getLastValid();
	  if (t_i != Time(0) && t_i < sStart) {
	      cerr << "GenChan: skipping channel: " << getChannel()
		   << " to gps " << sStart << endl;
	      Interval dT = mRawResp.getStartTime() - sStart;
	      mRawResp.eraseStart(dT);
	      mDaqChain.reset();
	      rc = -1;
	  }
	  i->last_valid(tNext);

	  if (mRawResp.isEmpty()) {
	      mRawResp = sData;
	      if (!mUnits.empty()) mRawResp.setUnits(mUnits);
	  } else {
	      if (mRawResp.getEndTime() < tNext) mRawResp.extend(tNext);
	      mRawResp += sData;
	  }
	  if (mDebug > 2) {
	      int src = i->getSrcID();
	      cout << "response after adding source " << sv[src]->getName() 
		   << endl;
	      mRawResp.extract(mRawResp.getStartTime(), 
			       16.0*mRawResp.getTStep()).Dump(cout);
	  }
	  if (!tLatest || tLatest > tNext) tLatest = tNext;
      }

      //--------------------------------  Apply DAQ emulation
      Time      t0 = mRawResp.getStartTime();
      Interval rDt = tLatest - t0;
      if (!mDaqChain.null()) {
	mResponse.Append(mDaqChain(mRawResp.extract(t0, rDt)));
	if (mDebug > 2) {
	  cout << "Response after DAQ transfer function " << endl;
	  mResponse.extract(t0, 16.0*mResponse.getTStep()).Dump(cout);
	}
      }
      else {
	mResponse.Append(mRawResp.extract(t0, rDt));
      }
      mRawResp.eraseStart(rDt);
      return rc;
  }

  //====================================  Release (used) data up to t0
  void 
  GenChan::release(const Time& t0) {
      Interval dT = t0 - mResponse.getStartTime();
      if (double(dT) > 0) mResponse.eraseStart(dT);
  }

  //====================================  Set the channel name.
  void 
  GenChan::setChannel(const char* chan) {
      mName = chan;
  }

  //====================================  Set the detector ID.
  void 
  GenChan::setDetID(int det) {
      mDetID = det;
  }

  //====================================  Set the sample rate.
  void 
  GenChan::setSample(Interval dt) {
      mSample = dt;
  }

  //====================================  Set the DAQ chain filter
  void 
  GenChan::setDaqChain(const Pipe* pipe) {
      if (!pipe) mDaqChain.reset();
      else       mDaqChain.set(*pipe); 
  }

  //====================================  Set the debugging printout level
  void
  GenChan::setDebug(int lvl) {
      mDebug = lvl;
  }

  //====================================  Define a response filter
  void 
  GenChan::setUnits(const std::string& unit) {
      mUnits = unit;
  }

  //====================================  Component descriptor constructor
  GenChan::chanComp::chanComp(int ID, const Pipe* resp)
    : mSrcID(ID), mLastSrcData(0), mLastValid(0)
  {
      if (resp) mRespFilter.set(*resp);
  }

  //====================================  chanComp copy constructor
  GenChan::chanComp::chanComp(const chanComp& x) {
      *this = x;
  }

  //====================================  chanComp destructor
  GenChan::chanComp::~chanComp(void) {
  }

  //====================================  chanComp assignment operator
  GenChan::chanComp&
  GenChan::chanComp::operator=(const chanComp& x) {
      mRespFilter  = x.mRespFilter;
      mSrcID       = x.mSrcID;
      mLastSrcData = x.mLastSrcData;
      mLastValid   = x.mLastValid;
      return *this;
  }

  //====================================  Apply response filter to a TSeries
  TSeries
  GenChan::chanComp::getResponse(const vector<DataSource*>& sv) {
      TSeries ts_out;
      const DataSource& src(*sv[mSrcID]);

      Time t0 = mLastSrcData;
      if (!t0) {
	  t0 = src.getStartTime();
	  if (!t0) return ts_out;
      } else if (t0 < src.getStartTime()) {
	  mRespFilter.reset();
      }
      Interval dT = src.getLatest() - t0;
      // cout << "Input time, length = " << t0 << ", " << dT << endl;
      if (!mRespFilter.null()) {
	  ts_out = mRespFilter(src.getTimeSeries(t0, dT));
	  ts_out.setUnits(src.units());
      } else {
	  ts_out = src.getTimeSeries(t0, dT);
      }
      // cout << "Output time, length = " << ts_out.getStartTime() 
      //      << ", " << ts_out.getInterval() << endl;
      mLastSrcData = t0 + dT;
      return ts_out;
  }

  //====================================  Record the last time generated.
  void 
  GenChan::chanComp::last_valid(const Time& t) {
      mLastValid = t;
  }

  //====================================  Define a response filter
  void 
  GenChan::chanComp::setResponseFilter(const Pipe& resp) {
      mRespFilter.set(resp);
  }

} //namespace generator
