/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GENERATOR_DATA_SOURCE_HH
#define GENERATOR_DATA_SOURCE_HH

#include "TSeries.hh"
#include "GenParam.hh"
#include <string>
#include <map>
#include <iosfwd>
#include <vector>

namespace generator {
    class DataSource;

    /// DataSource dictionary container type.
    typedef std::map<std::string, DataSource*> SourceDict;

    /// DataSource dictionary container constant iterator.
    typedef SourceDict::const_iterator         const_dict_iter;

    /// DataSource dictionary container iterator.
    typedef SourceDict::iterator               dict_iter;

    /// DataSource dictionary value type.
    typedef SourceDict::value_type             dict_node;

    /**  %DataSource is a base class for DMT generator package data sources.
      *  The %DataSource API is used for both continuous and discrete
      *  data sources. Parameters and event times are generated randomly
      *  as specified when the source is defined. Generated transient events 
      *  are automatically recorded for later inclusion in output data.
      *
      *  The typical usage model for a data source is as follows:
      *  - The Data source is created using the DataSource::parse() static
      *    factory method.
      *  - For continuous sources, parameters are generated with a single 
      *    call to \c randomize().
      *  - For transient sources, \c randomize() must be called for each 
      *    event.
      *  - Generate data for each stride by calling \c generate()
      *  - Fetch data from internal buffers with getTimeSeries()
      *  - Buffered data with \c release()
      *
      *  @memo Data source base class.
      *  @version 1.1; last modified November 3, 2004
      *  @author John Zweizig
      */
    class DataSource {
    public:

	/**  Data event repetition mode enumerator.
	 */
	enum RepMode {
	    kContinuous, ///<  Continuous data source
	    kSingle,     ///<  Generate a single event
	    kRate,       ///<  Generate events with Poisson statistics.
	    kFixed,      ///<  Generate events at fixed intervals.
	    kRetrigger   ///<  Retrigger at the end of each trigger.
	};

	/// Parameter map.
	typedef std::map<std::string, GenParam> ParamMap;

	/// Parameter map iterator
	typedef ParamMap::iterator param_iter;

	/// Constant parameter map iterator
	typedef ParamMap::const_iterator const_param_iter;

	/// Generated sample type
	typedef double gen_sample_type;

	/**  The SrcEvent class holds the generation parameters of each 
	 *  generated event. It is patterned after the FrSimEvent structure 
	 *  in the IGWD frame specification.
	 *  @memo Data source event parameter storage class.
	 */
	class SrcEvent {
	public:
	    ///  %Parameter list entry data type
	    typedef std::pair<std::string, float> ParamEntry;

	    ///  %Parameter list data type
	    typedef std::vector<ParamEntry> ParamList;

	    ///  %Parameter list iterator type
	    typedef ParamList::iterator param_iter;

	    ///  %Parameter list constant iterator type
	    typedef ParamList::const_iterator const_param_iter;

	public:
	    /**  Default SrcEvent constructor.
	      *  @memo Default constructor.
	      */
	    SrcEvent(void);

	    /**  SrcEvent data constructor.
	      *  @memo Source data event constructor.
	      *  @param name    Event source name
	      *  @param comment Comment string
	      *  @param t0      Event start time
	      *  @param before  Start time before the event time
	      *  @param after   Digitization time after the event time.
	      *  @param map     Parameter map.
	      */
	    SrcEvent(const std::string& name, const std::string& comment, 
		     const Time& t0, Interval before, Interval after, 
		     const ParamMap& map);

	    /**  Get the comment field
	      *  @memo Get comment field
	      *  @return Comment string pointer
	      */
	    const char* getComment(void) const;

	    /**  Get the input data.
	      *  @memo Get list of inputs.
	      *  @return Input string pointer.
	      */
	    const char* getInputs(void)  const;

	    /**  Get the data source name.
	      *  @memo Get name
	      *  @return Constant name string pointer
	      */
	    const char* getName(void)    const;

	    /**  Get the GPS time of the maximum value.
	      *  @memo Get maximum sample GPS time.
	      *  @return Time  of maximum sample
	      */
	    const Time& getGPSMax(void)  const;

	    /**  Get the duration of the generated waveform before the 
	      *  maximum time.
	      *  @memo Get event time before max time.
	      *  @return Length of time interval before max time.
	      */
	    Interval    getTimeBefore(void) const;

	    /**  Get the duration of the generated waveform after the 
	      *  maximum time.
	      *  @memo Get event time after max time.
	      *  @return Length of time interval after max time.
	      */
	    Interval    getTimeAfter(void) const;

	    /**  Get the waveform amplitude
	      *  @memo Event amplitude
	      *  @return Signal waveform amplitude.
	      */
	    double      getAmplitude(void) const;

	    /**  Get a list of the generated parameters.
	      *  @memo Fetch parameter list reference.
	      *  @return Constant parameter list reference.
	      */
	    const ParamList& getParamList(void) const;
      
	private:
	    std::string mName;        ///< Source name string
	    std::string mComment;     ///< Source comment string
	    std::string mInputs;
	    Time        mGPSMax;
	    Interval    mTimeBefore;
	    Interval    mTimeAfter;
	    double      mAmplitude;
	    ParamList   mParamList;
	};

	/// %Event vector data type
	typedef std::vector<SrcEvent> EventVect;

	/// %Event vector iterator type.
	typedef EventVect::iterator event_iter;

	/// Constant event vector iterator type
	typedef EventVect::const_iterator const_event_iter;
 
    public:
	/**  Static %DataSource dictionary initialization method adds all 
	  *  predefined data sources to the argument dictionary.
	  *  @memo Data source map initialization method.
	  *  @param dict Data source dictionary.
	  */
	static void init_dict(SourceDict& dict);

	/**  Static %DataSource dictionary deltetion function removes all
	  *  predefined data sources from the argument dictionary.
	  *  @memo Data source map destructor.
	  *  @param dict Data source dictionary to be erased.
	  */
	static void kill_dict(SourceDict& dict);

	/**  Static factory method produces data sources from a descriptor 
	  *  string of the form:
	  *  \verbatim
	     <data-source>(<param-1>=<value-1>,...,<param-n>=<value-n>)
             \endverbatim
	  *  @memo Data source factory method.
	  *  @param desc Data source descriptor string.
	  *  @param dict Data source dictionary.
	  *  @return Pointer to a new data source matching the description.
	  */
	static DataSource* parse(const std::string& desc, 
				 const SourceDict& dict);

    public:
	/**  Construct a data source object api and initialize the 
	  *  continuous/transient type code. 
	  *  @memo Default constructor
	  *  @param mode Mode code.
	  */
	explicit DataSource(RepMode mode=kContinuous);

	/**  Destroy an object
	  *  @memo Destructor
	  */
	virtual ~DataSource(void);

	/**  Make an exact copy of a data source
	  *  @memo Clone a data source.
	  *  @return Pointer to the newly created clone object.
	  */
	virtual DataSource* clone(void) const = 0;

	/**  Dump the data source status
	  *  @memo Dump the source status
	  *  @param out Output stream to which the status will be printed.
	  *  @return Reference to the output stream.
	  */
	virtual std::ostream& dump(std::ostream& out) const;


	/**  Dump the source definition string out to the specified output 
	  *  stream.
	  *  @memo Print the source definition.
	  *  @param out Output stream.
	  *  @return Reference to the output stream.
	  */
	virtual std::ostream& dumpDefinition(std::ostream& out) const;

	/**  Test if data source is continuous
	  *  @memo test for continuous source.
	  *  @return True if continuous source.
	  */
	bool isContinuous(void) const;

	/**  Generate events in a specified interval.
	  *  @memo Generate data in specified time interval.
	  *  @param t0 Generation interval start time.
	  *  @param dt Generation Interval duration.
	  */
	void generate(const Time& t0, Interval dt);

	/**  Get the data source type name.
	  *  @memo Source type name
	  *  @return Source type name string pointer.
	  */
	virtual const char* getDataType(void) const = 0;

	/**  Get the debug printout verbosity level.
	  *  @memo Get debug level.
	  *  @return Debug printout level.
	  */
	int  getDebug(void) const;

	/**  Get the length of the digitized signal for the current event.
	  *  @memo Event length
	  *  @return Event length in seconds.
	  */
	virtual Interval getDuration(void) const;

	/**  Get the earliest time not generated.
	  *  @memo Get the end of generated data.
	  *  @return Earliest time not generated.
	  */
	Time getLatest(void) const;

	/**  Get the name of the data source.
	  *  @memo Data source name.
	  *  @return Constant data source name pointer.
	  */
	const char* getName(void) const;

	/**  Get a numeric value of the named parameter.
	  *  @memo Get numeric parameter value.
	  *  @param name Parameter name string.
	  *  @return Parameter numeric value.
	  */
	virtual double getNumeric(const std::string& name) const;

	/**  Get a string value of the named parameter.
	  *  @memo Get string parameter value.
	  *  @param name %Parameter name string.
	  *  @return %Parameter value string.
	  */
	virtual const char* getParameter(const std::string& name) const;

	/**  Get a series value of the named parameter.
	  *  @memo Get series parameter value.
	  *  @param name %Parameter name string.
	  *  @return %Parameter value series.
	  */
	virtual const TSeries& getSeriesParam(const std::string& name) const;
	
	/**  Get the average rate used in generating events.
	  *  \brief Get average event rate.
	  *  \return Requested average event rate.
	  */
	double getRate(void) const;

	/**  Get the sample time (inverse of sample rate).
	  *  @memo Get the sample time
	  *  @return Sample time
	  */
	Interval getSample(void) const;

	/**  Get the offset from the generated event time to the start of the 
	  *  event waveform. A positive offset corresponds to a signal that 
	  *  starts after the generated time.
	  *  @memo Get time offset of wave form start.
	  *  @return Start of digitized waveform relative to the event time.
	  */
	virtual Interval getStartOffset(void) const;

	/**  Get the start of the already generated data.
	  *  @memo Start of generated data
	  *  @return Digitized data start time. 
	  */
	Time   getStartTime(void) const;

	/**  Get the trigger time for the current event.
	  *  @memo Current event time.
	  *  @return Nominal time of current event.
	  */
	Time   getTrigTime(void) const;

	/**  Get the generated time series in the interval from \c t0 to 
	  *  \c t0+dT.
	  *  @memo Get data as a time series.
	  *  @param t0 Start time of requested data
	  *  @param dT Duration of requested data
	  *  @return Generated data in requested interval.
	  */
	TSeries getTimeSeries(const Time& t0, Interval dT) const;

	/**  Get the generated data time series.
	  *  @memo Get data as a time series.
	  *  @return All available generated data.
	  */
	const TSeries& getTimeSeries(void) const;

	/**  Get the data source name.
	  *  @memo Get the source name.
	  *  @return Source name string.
	  */
	virtual std::string getSourceName(void) const;

	/**  Print statistics for the data source to the specified output 
	  *  stream.
	  *  @memo Print data source statistics
	  *  @param out Output stream.
	  *  @return Reference to output stream.
	  */
	virtual std::ostream& print_stats(std::ostream& out) const;

	/**  Chose the time and parameters for the next data segment. If 
	  *  the next trigger time has already been selected (as indicated
	  *  by non-zero value of mNextTrig), randomize returns immediately.
	  *  The event time generation depend on the \c mRptMode variable. 
	  *  - \e kRate: An exponentially distributed random number with
	  *              mean \c 1/mRate is added to the current event time.
	  *  - \e kFixed: A fixed time \c 1/mRate is added to the current
	  *               event time. 
	  *  - \e kSingle: The specified start time is used as the event time. 
	  * 
	  *  The parameters are then generated by calling 
	  *  GenParam::sampleParameter() for each parameter.
	  *  @memo Generate random parameter values.
	  */
	virtual void randomize(void);

	/**  Release generated data before the specified time.
	  *  @memo Release data
	  *  @param t0 Time before which data will be released.
	  */
	void release(const Time& t0);

	/**  Save the current event(s) into the event registry. One or more 
	  *  events may be saved depending on the source.
	  *  \brief Save the current event.
	  *  \param t0 Start of time epoch to save
	  *  \param dT Duration of time epic to save.
	  */
	virtual void saveEvent(const Time& t0, Interval dT);

	/**  Save the specified event into the event registry.
	  *  \brief Save the specified event.
	  *  \param ev Event description.
	  */
	virtual void saveEvent(const SrcEvent& ev);

	/**  Set the source data type,
	  *  @memo Set source data type
	  *  @param dtype Data type name string.
	 */
	void setDataType(const std::string& dtype);

	/**  Set the debug level.
	  *  @memo Set debug level
	  *  @param lvl Debug v
	 */
	void setDebug(int lvl);

	/**  Set the data source name.
	  *  @memo Set data source name.
	  *  @param name Source name.
	 */
	void setName(const std::string& name);

	/**  Set the specified parameter name to a string value.
	  *  @memo Set a named parameter.
	  *  @param name %Parameter name string
	  *  @param val String value for the names parameter.
	 */
	virtual void setParameter(const std::string& name, 
				  const std::string& val);

	/**  Set the specified parameter to a numeric value.
	  *  @memo Set a named parameter.
	  *  @param name %Parameter name string
	  *  @param value Numeric value for the names parameter.
 	  */
	virtual void setParameter(const std::string& name, double value);

	/**  Set the specified parameter to a series value.
	  *  @memo Set a named parameter.
	  *  @param name %Parameter name string
	  *  @param ser  %Time series value for the names parameter.
 	  */
	virtual void setParameter(const std::string& name, const TSeries& ser);

	/**  Set the digitization sample time.
	  *  @memo Set the sample time
	  *  @param dT Sample time (\c 1/rate ) in seconds.
	  */
	void setSample(Interval dT);

	/**  Set the average event rate.
	  *  @memo Set event rate.
	  *  @param rate %Event rate.
	  */
	void setTriggerRate(double rate);

	/**  Set the event repeat interval.
	  *  @memo Set event time difference.
	  *  @param rpt %Event time difference.
	  */
	void setTriggerRepeat(Interval rpt);

	/**  Set the time for the next/only event.
	  *  @memo Set the trigger time
	  *  @param t0 %Time of next/only event.
	  */
	void setTriggerTime(const Time& t0);

	/**  Set the source data units string.
	  *  \brief Set the units string.
	  *  \param unit Unit string.
	  */
	void setUnits(const std::string& unit);

	//----------------------------------  Control which data are saved.
	/**  Get the write flag value. If true, the generate data are to be 
	  *  written to an output frame.
	  *  @memo Get the write flag.
	  *  @return Write flag state.
	  */
	bool getWriteFlag(void) const;    

	/**  Get the save flag value. If true, the event parameters are to be 
	  *  saved for each event.
	  *  @memo Get the save flag.
	  *  @return Save flag state.
	  */
	bool getSaveFlag(void) const;    

	/**  Set the write flags value.
	  *  @memo Set the write flag
	  *  @param yorn New write flag state
	  */
	void setWriteFlag(bool yorn=true);

	/**  Set the save flag value
	  *  @memo Set the save flag
	  *  @param yorn New save flag state
	  */
	void setSaveFlag(bool yorn=true);

	/**  Set all the series parameters from a list of sources.
	  *  \brief Set series parameters from a list.
	  *  \param lsrcs source list
	  */
	void setSeriesParam(const std::vector<DataSource*>& lsrcs,
			    const Time& t0, Interval dT);
	
	/**  Get the unit data string.
	  *  \brief Units string.
	  *  \return Reference to the source data unit string.
	  */
	const std::string& units(void) const;

	//----------------------------------  Get event data
	/**  Get the number of events stored for this input source. If the 
	  *  save flag is not set, this number will always be zero.
	  *  \brief Get number of saved events,
	  *  \return Number of events saved by this data source.
	  */
	unsigned long getNEvent(void) const;

	/**  Return a constant reference to the first event in the list of
	  *  generated events.
	  *  @memo Get a constant reference to the first event info.
	  *  @return Constant reference to first SrcEvent object.
	  */
	const SrcEvent& refEvent(void) const;

	/**  Return a constant reference to the specified (ith) event in the
	  *  list of generated events..
	  *  @memo Get reference to specified event.
	  *  @param i %Event number.
	  *  @return Reference to specified event.
	  */
	const SrcEvent& refEvent(unsigned long i) const;

	/**  Erase the specified number of events from the vector.
	  *  @memo erase events
	  *  @param N Number of events to erase.
	  */
	void eraseEvent(unsigned long N);

    private:
	/**  Get the default sample rate. The time and interval arguments
	  *  provide a time epoch for which the default sample rate is 
	  *  needed.
	  *  \brief Calculate a default sample rate for this source.
	  *  \param t0 Start time of epoch.
	  *  \param dt Duration of epoch
	  *  \return Sample rate.
	  */
	virtual double defaultRate(const Time& t0, Interval dT);

	/**  Get the default data type. The time and interval arguments
	  *  provide a time epoch for which the default data type is needed.
	  *  \brief Calculate a default sample rate for this source.
	  *  \param t0 Start time of epoch.
	  *  \param dt Duration of epoch
	  *  \return Data type
	  */
	virtual long defaultType(const Time& t0, Interval dT);

	/**  Find the specified parameter name.
	  *  @memo Find the named parameter entry
	  *  @param name Name of parameter to be located
	  *  @return Iterator pointing to specified parameter.
	  */
	param_iter findParam(const std::string& name);

	/**  Find the specified parameter name.
	  *  @memo Find the named parameter entry
	  *  @param name Name of parameter to be located
	  *  @return Constant iterator pointing to specified parameter.
	  */
	const_param_iter findParam(const std::string& name) const;

	/**  Copy the generated source data from the specified time interval
	  *  into the specified data vector.
	  *  @memo Get data from the specified time series.
	  *  @param t0   Requested start time.
	  *  @param dT   Data sample interval (1/rate).
	  *  @param N    Number of samples to fetch.
	  *  @param data Preallocated storage for fetched data.
	  */
	virtual void getSeries(const Time& t0, Interval dT, int N, 
			       gen_sample_type* data);

	/**  Copy the generated source data from the specified time interval
	  *  into the referenced data vector.
	  *  @memo Get data from the specified time series.
	  *  @param t0   Requested start time.
	  *  @param dT   Data sample interval (1/rate).
	  *  @param N    Number of samples to fetch.
	  *  @param data Time series - data to be added.
	  */
	virtual void getSourceTSeries(const Time& t0, Interval dT, int N, 
				      TSeries& data);

    private:
	//---------------------------------  Parameters
	RepMode  mRptMode;       ///< Trigger repetition mode.
	double   mRate;          ///< Trigger rate
	Interval mSample;        ///< Sample time (1/<sample-rate>)
	int      mDebug;
	std::string mName;       ///< Source name string
	std::string mUnits;      ///< Units string
	ParamMap mParam;
	enum Flags {
	    flWrite=1,
	    flSvEvents=2
	};
	long      mFlags;
	long      mDataType;

	//---------------------------------  State
	Time     mNextTrig;
	Time     mCurrent;
	Time     mTrigStart;
	Time     mTrigEnd;
	Time     mLatest;
	TSeries  mSeries;
	long      mEventNo;
	EventVect mRecord;

	//----------------------------------  Statistics
	Interval  mSumTime;
    };  //class DataSource

    //====================================  Inline functions.
    inline bool
    DataSource::isContinuous(void) const {
	return mRptMode == kContinuous;
    }

    inline int 
    DataSource::getDebug(void) const {
	return mDebug;
    }

    inline Interval
    DataSource::getDuration(void) const {
	return 1.0;
    }

    inline Time 
    DataSource::getLatest(void) const {
	return mLatest;
    }

    inline const char*
    DataSource::getName(void) const {
	return mName.c_str();
    }

    inline Time
    DataSource::getTrigTime(void) const {
	return mNextTrig;
    }

    inline Interval
    DataSource::getSample(void) const {
	return mSample;
    }

    inline double
    DataSource::getRate(void) const {
	return mRate;
    }

    inline Interval
    DataSource::getStartOffset(void) const {
	return getDuration()*(-0.5);
    }

    inline Time
    DataSource::getStartTime(void) const {
	return mSeries.getStartTime();
    }

    inline bool 
    DataSource::getSaveFlag(void) const {
	return (mFlags & flSvEvents);
    }

    inline bool 
    DataSource::getWriteFlag(void) const {
	return (mFlags & flWrite);
    }

    inline DataSource::param_iter 
    DataSource::findParam(const std::string& name) {
	return mParam.find(name);
    }

    inline DataSource::const_param_iter 
    DataSource::findParam(const std::string& name) const {
	return mParam.find(name);
    }


    //----------------------------------  Get event data
    inline unsigned long 
    DataSource::getNEvent() const {
	return mRecord.size();
    }

    inline const DataSource::SrcEvent& 
    DataSource::refEvent(unsigned long i) const {
	return mRecord[i];
    }

    inline void 
    DataSource::eraseEvent(unsigned long N) {
	if (N) mRecord.erase(mRecord.begin(), mRecord.begin()+N);
    }

    inline const std::string& 
    DataSource::units(void) const {
	return mUnits;
    }

    inline const char* 
    DataSource::SrcEvent::getComment(void) const {
	return mComment.c_str();
    }

    inline const char* 
    DataSource::SrcEvent::getInputs(void)  const {
	return mInputs.c_str();
    }

    inline const char* 
    DataSource::SrcEvent::getName(void)    const {
	return mName.c_str();
    }

    inline const Time& 
    DataSource::SrcEvent::getGPSMax(void)  const {
	return mGPSMax;
    }

    inline Interval
    DataSource::SrcEvent::getTimeBefore(void) const {
	return mTimeBefore;
    }

    inline Interval
    DataSource::SrcEvent::getTimeAfter(void) const {
	return mTimeAfter;
    }

    inline  double
    DataSource::SrcEvent::getAmplitude(void) const {
	return mAmplitude;
    }

    inline const DataSource::SrcEvent::ParamList& 
    DataSource::SrcEvent::getParamList(void) const {
	return mParamList;
    }

} // namespace generator
#endif // GENERATOR_DATA_SOURCE_HH
