#ifndef LEXR_HH
#define LEXR_HH

#include <iosfwd>
#include <string>
#include "Translate.hh"

/** The Lexer class break an input stream into individual tokens.
  * \brief Analyse text.
  */
class Lexr {
public:

  /** Lexographic type codes
   */
  enum LxTags {
    tkEnd,           ///<  End of input
    tkSymbol,        ///<  Symbol name
    tkLiteral,       ///<  Numeric value
    tkString,        ///<  Quote delimited string.
    tkOper,          ///<  Operator string
    tkOParen,        ///<  Open parenthesis  ("(" or "{" in token string)
    tkCParen,        ///<  Close parenthesis (")" or "}" in token string)
    tkOSubs,         ///<  Open parenthesis  ("[" or "<" in token string)
    tkCSubs,         ///<  Close parenthesis ("]" or "<" in token string)
    tkComma          ///<  Delimiter         ("," or ":" in token string)
  };

private:
  /** Character type enumerator
    */
  enum ChartEnum {
      kInvChar,      ///<  Invalid character
      kAlpha,        ///<  Alphabet a-z, A-Z and '_'
      kNumer,        ///<  Number 0-9
      kQuote,        ///<  ' or "
      kParen,        ///<  Parentheses ()[]
      kOper,         ///<  Operator "+-*/=<>&|!%"
      kComma,        ///<  Delimiter ",:"
      kDot,          ///<  Period (used in numbers)
      kSpace         ///<  White space (ignored - " \t\n")
  };

  /**  Enumerated state numbers.
    */
  enum StateID {
      start,         ///<  nothing read yet
      in_symbol,     ///<  reading a symbol
      in_string,     ///<  reading a quoted string
      in_oper,       ///<  reading an operator string.
      numer_int,     ///<  reading the integer part of a number
      numer_fix,     ///<  reading a decimal fraction.
      numer_exp,     ///<  found nnnne
      numer_sexp     ///<  found nnnne[+- ][n]
  };

public:
  /**  Construct a lexicographical analyzer. Initialize the internal 
    *  state tables.
    *  \brief Default constructor.
    */
  Lexr(void);

  /**  Destroy the Lexer instance
    *  \brief Destructor.
    */
  virtual ~Lexr(void) {}

  /**  Get a lexographical token from the specified input stream.
    *  \brief Get a token
    *  \param in Input steam.
    *  \param token Reference to string to receive full token text
    *  \return Enumerated token type.
    */
  virtual LxTags getToken(std::istream& in, std::string& token) const;

  /**  Set the debug printout verbosity.
    *  \brief Set the debug level.
    *  \param lvl Print verbosity level.
    */
  void setDebug(int lvl);

  /**  Add characters to the list of delimiters.
    *  \brief Add delimiter characters.
    *  \param str Delimiter character string.
    */
  void addDelimChars(const char* str);

  /**  Add characters to the list of operators.
    *  \brief Add operator characters.
    *  \param str Operator character string.
    */
  void addOpChars(const char* str);

  /**  Add characters to the list of parentheses.
    *  \brief Add parentheses characters.
    *  \param str Parenthesis character string.
    */
  void addParenChars(const char* str);

  /**  Set the list of delimiter charaters.
    *  \brief Set delimiter characters.
    *  \param str Delimiter character string.
    */
  void setDelimChars(const char* str);

  /**  Set the list of operator charaters.
    *  \brief Set operator characters.
    *  \param str Operator character string.
    */
  void setOpChars(const char* str);

  /**  Set the list of parenthesis characters.
    *  \brief Set parentheses characters.
    *  \param str Parenthesis character string.
    */
  void setParenChars(const char* str);

private:
  Translate<ChartEnum> mTrTable;
  int mDebug;
};

#endif // LEXR_HH
