/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    DMT Data processing base class implementation.
//
//-------------------------------------  Header files.
#include "MultiStream.hh"

#ifndef __CINT__
#include <iostream>
#include "Interval.hh"
#include "Time.hh"
#include <stdexcept>
#include <string>
#include <cstdlib>

#else
#define DACC_ONLDEV "/online/"

#endif // __CINT__

using namespace std;

//======================================  MultiStream constructor.
MultiStream::MultiStream(int argc, const char *argv[]) 
  : mActive(false), mDebug(0), mEOF(false), mAttnMask(0), mMaxFrames(0),
    mAlignMult(0), mAlignOff(0)
{
    //----------------------------------  Parse the arguments
    for (int i=1 ; i<argc && argv ; i++) {
	string argi = argv[i];
	if (argi == "-inlist") {
	    if (++i >= argc) return;
	    getDacc().addSingle(argv[i]);
	} else if (argi == "-inlists") {
	    if (++i >= argc) return;
	    getDacc().addMulti(argv[i]);
	} else if (argi == "-debug") {
	    if (++i >= argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	} else if (argi == "-maxframes") {
	    mMaxFrames = strtol(argv[++i], 0, 0);
	}
    }
    getDacc().setDebug(mDebug);
  
#ifndef __CINT__
    //----------------------------------  Handle signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);

    // mAttn.setMode(SigFlag::kBlock);
    mAttn.add(SIGUSR1);
    mAttn.add(SIGIO);

#else
    gROOT->SetInterrupt(kTRUE);
#endif   // !def(__CINT__)

    //----------------------------------  Verify >1 stream is defined.
    if (!getNStream()) {
	//------------------------------  See if DMTINPUT is defined
	if (getenv("DMTINPUT")) {
	    string inpath = "=";
	    inpath += getenv("DMTINPUT");
	    getDacc().addSingle(inpath);
	}

	//------------------------------  Type LIGOSMPART
	else if (getenv("LIGOSMPART")) {
	    string part = "=/online/";
	    part += getenv("LIGOSMPART");
	    getDacc().addSingle(part);
	}

	//------------------------------  Give up - no input specified.
	else {
	    cout << "No input data streams have been defined for monitor: " 
		 << argv[0] << endl;
	    finish();
	    return;
	}
    }

    //----------------------------------  Open a frame stream (FrameCPP)
    // if (getDacc().open()) return;
    mActive = true;
}

//======================================  Test if MultiStream command line argument
bool
MultiStream::isMultiStreamArg(const char* argc) const {
    string arg(argc);
    if (arg == "-inlist")    return true;
    else if (arg == "-inlists") return true;
    else if (arg == "-debug")     return true;
    else if (arg == "-maxframes") return true;
    return false;
}

//--------------------------------------  MultiStream object destructor.
MultiStream::~MultiStream() 
{
    //----------------------------------  DMT has terminater
    cout << "Data environment has terminated with Term/Attn/finish/EOF =" 
	 << bool(mTerm) << "/" << bool(mAttn) << "/" << !mActive << "/" 
	 << mEOF << endl;

#ifndef __CINT__
    //----------------------------------  Release signals
    mTerm.zero();
    mAttn.zero();

#endif //  __CINT__

    //----------------------------------  Close the Input file/partition.
    getDacc().close();
}

//--------------------------------------  Main processing loop function.
void
MultiStream::MainLoop(void) 
{
    getDacc().open();

    //----------------------------------  Loop until terminated.
    bool newFill = true;
    while(!testTerm() && mActive) {
    
	//------------------------------  Check for an attention interrupt.
	if (testAttn()) {
	    Attention();
	    if (!mActive) continue;
	}
    
	//------------------------------  Get the data identifier
	if (getDacc().synch()) {
	    mEOF = true;
	    cerr << "Synch failed" << endl;
	    break;
	}
    
	//--------------------------  Force the stride alignment
	if (newFill && mAlignMult != 0) {
	    Time t  = getDacc().getCurrentTime();
	    long ts = t.getS();
	    Time Tv = Time(ts - (ts%mAlignMult)) + Interval(mAlignOff);
	    if (Tv < t && !Almost(Tv, t)) Tv += Interval(mAlignMult);
	    if (!Almost(Tv, t)) {
		if (Debug()) cerr << "Aligning to (" << mAlignMult << ", " 
				  << mAlignOff << ") - skip to " << Tv 
				  << endl;
		if (getDacc().seek(Tv)) continue;
		if (getDacc().synch()) continue;
		if (!Almost(getDacc().getCurrentTime(), Tv)) continue;
	    }
	}

    //--------------------------  Read a frame
    int rc;
    try {
	rc = getDacc().fillData(Interval(0.0), newFill);
    } catch(exception& e) {
	cerr << "Exception in fillData: " << e.what() << endl;
	finish();
	break;
    }
    if (rc == 0) {
	try {
	    ProcessData();
	    newFill = true;
	} catch(exception& e) {
	    cerr << "Exception in ProcessData: " << e.what() << endl;
	    finish();
	} catch(...) {
	    cerr << "Unidentified exception in ProcessData." << endl;
	    finish();
	}
    } else if (rc == -8) {
	newFill = false;
    } else if (rc <= -4) {
	cerr << "MainLoop: Error while reading frame -" 
	     << Dacc::getMsgText(rc) << endl;
	mEOF = true;
	break;
    } else if (Debug()) {
	cerr << "MainLoop: Frame missing from input stream." << endl;
	newFill = true;
    }
    }
    
    if (mMaxFrames && mIn.getTotalFrames() >= mMaxFrames) {
	cout << "Maximum frame count (" << mMaxFrames << ") exceeded." 
	     << endl;
	mActive = false;
    }
}

//======================================  Set stride alignment
void
MultiStream::setStrideAlignment(unsigned long mult, double off) {
    mAlignMult = mult;
    if (!mAlignMult) {
        mAlignOff = 0.0;
    } else {
        mAlignOff = off;
	double dMult = mAlignMult;
	mAlignOff -= double(long(mAlignOff/dMult)*dMult);
	if (mAlignOff < 0.0) mAlignOff += dMult;
    }
}


//--------------------------------------  Background interrupt handling.
#ifndef __CINT__  

void
MultiStream::Attention(void) {
}

bool 
MultiStream::testAttn(void) {
    mAttnMask = mAttn.getSigFlags();
    if (mAttnMask) mAttn.clearFlags(mAttnMask);
    return (mAttnMask != 0);
}

bool 
MultiStream::testTerm(void) const {
    return mTerm;
}

//--------------------------------------  Root interrupt handling.
#else   
void
MultiStream::Attention(void) {
    mActive = false;
}

bool 
MultiStream::testAttn(void) {
    gSystem->ProcessEvents();
    mAttnMask = gROOT->IsInterrupted() ? 1 : 0;
    return (mAttnMask != 0);
}

bool 
MultiStream::testTerm(void) {
    return false;
}
#endif
