#include <time.h>
#include "LscEmul.hh"
#include "DynConfig.hh"
#include "DVector.hh"
#include "Interval.hh"
#include "ParseLine.hh"
#include <math.h>
#include <iostream>
#include <iomanip>
#include <sstream>

using namespace std;

//  Notes on LLO LSC Front-End from sources found in 
//       /cvs/cds/project/lsc/fe/lscfe6.c
//  coefficient files in 
//       /cvs/cds/lho/chans/lscCoeff<ifo>_<date>.txt
//
//  1) Data read into channels{}{} 
//  2) asPort.IR, asPort.QR, po1.IR, po1.QR, ref.IR, ref.QR, po2.IR, 
//     po2.QR set from channels, add Epics-defined offsets.
//  3) Apply dewhitening filter or not depending on dwFilter[6-11]
//  4) Apply phase correction, store result in (asPort.IM, asPort.QM, ...)
//  5) Apply input transformation, store in asPort.QO (DARM), po1.IO(Mich?), 
//     po1.QO() and ref.IO(CArm).
//  6) Add constant offset, excitation signal. Store in xxx.[I,Q]X
//  7) Turn on boost filters in acquire mode (not emulated)
//  8) Filter asPort.QX -> postFilter.Lm, po1.Qx -> postFilter.lm, 
//     po1.IX -> postFilter.lp, ref.IX -> postFilter.Lp, multiply by
//     a constant scale factor.
//  9) Add an offset and excitation signal to postFilter.xx's.. save them 
//     in preXForm[0-5].
//

const TSeries ts0;

#define SETTITLE(name) name.setName("name")

//======================================  Construct the emulator
LscEmul::LscEmul(DynConfig& in, const string& ifo) 
  : mDyn(in), mDebug(false), mAs1Pd("AS1"), mAs2Pd("AS2"), mAs3Pd("AS3"),
    mAs4Pd("AS4"), mRflPd("RFL"), mPobPd("PO")
{
    if (!ifo.empty()) setIfo(ifo);
}

//======================================  Destroy the emulator
LscEmul::~LscEmul(void) {
}

//======================================  Get an output TSeries
const TSeries&
LscEmul::getSeries(SeriesOut out) const {
    switch(out) {
    case kPrepAsI:
        return mAsIPrep;
    case kPrepAsQ:
        return mAsQPrep;
    case kPrepPoI:
        return mPoIPrep;
    case kPrepPoQ:
        return mPoQPrep;
    case kPrepRflI:
        return mRflIPrep;
    case kPrepRflQ:
        return mRflQPrep;
    case kDArmCalc:
        return mDArmCalc;
    case kMichCalc:
        return mMichCalc;
    case kPrcCalc:
        return mPrcCalc;
    case kCArmCalc:
        return mCArmCalc;
    }
    return ts0;
}

//======================================  Solve a set of 2 linear equations.
int 
solve22(const double A[4], const double V[2], double X[2]) {
    double det = A[0]*A[3] - A[1]*A[2];
    if (det == 0) return 1;
    double Inv[4];
    Inv[0] =  A[3]/det;
    Inv[1] = -A[2]/det;
    Inv[2] = -A[1]/det;
    Inv[3] =  A[0]/det;
    X[0] = Inv[0] * V[0] + Inv[1] * V[1];
    X[1] = Inv[2] * V[0] + Inv[3] * V[1];
    //cout << "Solving: A= " << A[0] << " " << A[1]  << " " << A[2] 
    //	   << " " << A[3] << endl;
    //cout << "         V= " << V[0] << " " << V[1]  << endl;
    //cout << "         X= " << X[0] << " " << X[1]  << endl;
    return 0;
}

//======================================  preset the history of an IIR filter
void
setHstry(IIRFilter& f, const double inData[2], const double outData[2]) {
    // double fStat[5], A[4], V[2], X[2];
    // f.getStat(0, 5, fStat);
    // double a1 = fStat[0];
    // double a2 = fStat[1];
    // double b1 = fStat[2];
    // double b2 = fStat[3];
    // double b0 = fStat[4];
    // V[0] = outData[0] - b0 * inData[0];
    // V[1] = outData[1] - b0 * inData[1] - (b1 - b0*a1) * inData[0];
    // A[0] = b1 - b0*a1;
    // A[1] = b2 - b0*a2;
    // A[2] = b2 - b0*a2 + b0*a1*a2 - a1*b1;
    // A[3] = b0*a1*a2 - b1*a2;
    // solve22(A, V, X);
    // f.setHistory(0, 2, X);
}

//======================================  Reset all setable time series
void
LscEmul::reset(void) {
    mAs1QRaw.Clear();
    mAs1IRaw.Clear();
    mAs2QRaw.Clear();
    mAs2IRaw.Clear();
    mAs3QRaw.Clear();
    mAs3IRaw.Clear();
    mAs4QRaw.Clear();
    mAs4IRaw.Clear();
    mPoQRaw.Clear();
    mPoIRaw.Clear();
    mRflQRaw.Clear();
    mRflIRaw.Clear();
    mAsQData.Clear();
    mAsIData.Clear();
    mPoQData.Clear();
    mPoIData.Clear();
    mRflQData.Clear();
    mRflIData.Clear();
    mDArmCtrl.Clear();
    mMichCtrl.Clear();
    mPrcCtrl.Clear();
    mCArmCtrl.Clear();
}

//======================================  Get an output TSeries
void
LscEmul::setSeries(SeriesIn in, const TSeries& ts) {
    switch (in) {
    case kRawAs1I:
      mAs1IRaw = ts;
      mAs1IRaw.Convert(DVector::t_double);
      break;
    case kRawAs1Q:
      mAs1QRaw = ts;
      mAs1QRaw.Convert(DVector::t_double);
      break;
    case kRawAs2I:
      mAs2IRaw = ts;
      mAs2IRaw.Convert(DVector::t_double);
      break;
    case kRawAs2Q:
      mAs2QRaw = ts;
      mAs2QRaw.Convert(DVector::t_double);
      break;
    case kRawAs3I:
      mAs3IRaw = ts;
      mAs3IRaw.Convert(DVector::t_double);
      break;
    case kRawAs3Q:
      mAs3QRaw = ts;
      mAs3QRaw.Convert(DVector::t_double);
      break;
    case kRawAs4I:
      mAs4IRaw = ts;
      mAs4IRaw.Convert(DVector::t_double);
      break;
    case kRawAs4Q:
      mAs4QRaw = ts;
      mAs4QRaw.Convert(DVector::t_double);
      break;
    case kRawPoI:
      mPoIRaw = ts;
      mPoIRaw.Convert(DVector::t_double);
      break;
    case kRawPoQ:
      mPoQRaw = ts;
      mPoQRaw.Convert(DVector::t_double);
      break;
    case kRawRflI:
      mRflIRaw = ts;
      mRflIRaw.Convert(DVector::t_double);
      break;
    case kRawRflQ:
      mRflQRaw = ts;
      mRflQRaw.Convert(DVector::t_double);
      break;
    case kDataAsI:
      mAsIData = ts;
      mAsIData.Convert(DVector::t_double);
      break;
    case kDataAsQ:
      mAsQData = ts;
      mAsQData.Convert(DVector::t_double);
      break;
    case kDataPoI:
      mPoIData = ts;
      mPoIData.Convert(DVector::t_double);
      break;
    case kDataPoQ:
      mPoQData = ts;
      mPoQData.Convert(DVector::t_double);
      break;
    case kDataRflI:
      mRflIData = ts;
      mRflIData.Convert(DVector::t_double);
      break;
    case kDataRflQ:
      mRflQData = ts;
      mRflQData.Convert(DVector::t_double);
      break;
    case kDArmCtl:
      mDArmCtrl = ts;
      mDArmCtrl.Convert(DVector::t_double);
      break;
    case kMichCtl:
      mMichCtrl = ts;
      mMichCtrl.Convert(DVector::t_double);
      break;
    case kPrcCtl:
      mPrcCtrl = ts;
      mPrcCtrl.Convert(DVector::t_double);
      break;
    case kCArmCtl:
      mCArmCtrl = ts;
      mCArmCtrl.Convert(DVector::t_double);
    }
    // if (mDebug) {
    //    cout << "Set TSeries values: " << ts.getName() << endl;
    //    ts.extract(ts.getStartTime(), 16.0*ts.getTStep()).Dump(cout);
    // }
}

//======================================  Set the Lsc constants
void
LscEmul::setIfo(const string& ifo) {
    mIFO = ifo;
    mLscPrefix = mIFO + ":LSC-";
    defLscConfig();
}

//======================================  Utility functions
inline TSeries
operator*(const TSeries& ts, double x) {
    TSeries r(ts);
    r *= x;
    return r;
}

inline TSeries
operator*(double x, const TSeries& ts) {
    TSeries r(ts);
    r *= x;
    return r;
}

inline void 
CheckLimit(double* p, int N, double lo, double hi, LscEmul::stat_type& stat) 
{
    if (lo == hi) return;
    for (int i=0 ; i<N ; i++) {
        double d = p[i];
        if (d > hi) {
	    p[i]  = hi;
	    stat |= LscEmul::kLimiter;
	} else if (d < lo) {
	    p[i]  = hi;
	    stat |= LscEmul::kLimiter;
	}
    }
}

//======================================  Emulate the Lsc front end code.
void
LscEmul::emulate(void) {

    //----------------------------------  Reset status flags
    mDArmStat = 0;
    mMichStat = 0;
    mPrcStat = 0;
    mCArmStat = 0;

    //----------------------------------  Get As whitened, phased, etc..
    if (mAs1Pd.active()) {
        mAsIPrep  = mAs1Pd.getPortI(mAs1IRaw, mAs1QRaw);
        mAsQPrep  = mAs1Pd.getPortQ(mAs1IRaw, mAs1QRaw);
    } else {
        mAsQPrep.Clear();
        mAsIPrep.Clear();
    }

    if (mAs2Pd.active()) {
        mAsIPrep += mAs2Pd.getPortI(mAs2IRaw, mAs2QRaw);
        mAsQPrep += mAs2Pd.getPortQ(mAs2IRaw, mAs2QRaw);
    }

    if (mAs3Pd.active()) {
        mAsIPrep += mAs3Pd.getPortI(mAs3IRaw, mAs3QRaw);
        mAsQPrep += mAs3Pd.getPortQ(mAs3IRaw, mAs3QRaw);
    }

    if (mAs4Pd.active()) {
        mAsIPrep += mAs4Pd.getPortI(mAs4IRaw, mAs4QRaw);
        mAsQPrep += mAs4Pd.getPortQ(mAs4IRaw, mAs4QRaw);
    }

    //----------------------------------  Get Po whitened, phased, etc..
    if (mPobPd.active()) {
        mPoIPrep  = mPobPd.getPortI(mPoIRaw, mPoQRaw);
        mPoQPrep  = mPobPd.getPortQ(mPoIRaw, mPoQRaw);
    } else {
        mPoQPrep.Clear();
        mPoIPrep.Clear();
    }

    //----------------------------------  Get Rf whitened, phased, etc..
    if (mRflPd.active()) {
        mRflIPrep  = mRflPd.getPortI(mRflIRaw, mRflQRaw);
        mRflQPrep  = mRflPd.getPortQ(mRflIRaw, mRflQRaw);
    } else {
        mRflQPrep.Clear();
        mRflIPrep.Clear();
    }

    //-----------------------------------  Propogate calculated signals
    if (mAsQData.isEmpty()  && !mAsQPrep.isEmpty())  mAsQData  = mAsQPrep;
    if (mAsIData.isEmpty()  && !mAsIPrep.isEmpty())  mAsIData  = mAsIPrep;
    if (mPoQData.isEmpty()  && !mPoQPrep.isEmpty())  mPoQData  = mPoQPrep;
    if (mPoIData.isEmpty()  && !mPoIPrep.isEmpty())  mPoIData  = mPoIPrep;
    if (mRflQData.isEmpty() && !mRflQPrep.isEmpty()) mRflQData = mRflQPrep;
    if (mRflIData.isEmpty() && !mRflIPrep.isEmpty()) mRflIData = mRflIPrep;

    //-----------------------------------  Perform input transformation
    mDArmCalc = mITMTRX_00 * mAsIData  + mITMTRX_01 * mAsQData 
              + mITMTRX_02 * mPoIData  + mITMTRX_03 * mPoQData  
              + mITMTRX_04 * mRflIData + mITMTRX_05 * mRflQData;

    mPrcCalc  = mITMTRX_10 * mAsIData  + mITMTRX_11 * mAsQData 
              + mITMTRX_12 * mPoIData  + mITMTRX_13 * mPoQData  
              + mITMTRX_14 * mRflIData + mITMTRX_15 * mRflQData;

    mMichCalc = mITMTRX_20 * mAsIData  + mITMTRX_21 * mAsQData 
              + mITMTRX_22 * mPoIData  + mITMTRX_23 * mPoQData  
              + mITMTRX_24 * mRflIData + mITMTRX_25 * mRflQData;

    mCArmCalc = mITMTRX_30 * mAsIData  + mITMTRX_31 * mAsQData 
              + mITMTRX_32 * mPoIData  + mITMTRX_33 * mPoQData  
              + mITMTRX_34 * mRflIData + mITMTRX_35 * mRflQData;

    //-----------------------------------  Filter, filter, filter
    mDArmCalc  = mDArmFm.filter(mDArmCalc);
    mDArmCalc *= mDARM_GAIN;
    mDArmCalc += mDARM_OFFSET;

    mMichCalc  = mMichFm.filter(mMichCalc);
    mMichCalc *= mMICH_GAIN;
    mMichCalc += mMICH_OFFSET;

    mPrcCalc   = mPrcFm.filter(mPrcCalc);
    mPrcCalc  *= mPRC_GAIN;
    mPrcCalc  += mPRC_OFFSET;

    mCArmCalc  = mCArmFm.filter(mCArmCalc);
    mCArmCalc *= mCARM_GAIN;
    mCArmCalc += mCARM_OFFSET;

    //----------------------------------  Force limits
    mDArmCalc.Convert(DVector::t_double);
    double *pDArm = static_cast<double*>(mDArmCalc.refData());
    int N=mDArmCalc.getNSample();
    CheckLimit(pDArm, N, -mDARM_LIMIT, mDARM_LIMIT, mDArmStat);

    mMichCalc.Convert(DVector::t_double);
    double* pMich = static_cast<double*>(mMichCalc.refData());
    CheckLimit(pMich, N, -mMICH_LIMIT, mMICH_LIMIT, mMichStat);

    mPrcCalc.Convert(DVector::t_double);
    double *pPrc = static_cast<double*>(mPrcCalc.refData());
    CheckLimit(pPrc, N,  -mPRC_LIMIT,  mPRC_LIMIT,  mPrcStat);

    mCArmCalc.Convert(DVector::t_double);
    double *pCArm = static_cast<double*>(mCArmCalc.refData());
    CheckLimit(pCArm, N, -mCARM_LIMIT, mCARM_LIMIT, mCArmStat);
}

//======================================  Compare results with expectations
void
compareSeries(const TSeries& test, const TSeries& ref, int flags) {

    //----------------------------------  Look for differences in time scale
    if (test.getStartTime() != ref.getStartTime()) {
        cout << "TSeries start times differ" << endl;
	return;
    }
    int N = ref.getNSample();
    if (int(test.getNSample()) != N) {
        cout << "TSeries numbers of samples differ" << endl;
	return;
    }
    if (test.getTStep() != ref.getTStep()) {
        cout << "TSeries time steps differ" << endl;
	return;
    }
    if (!N) return;

    //----------------------------------  fit test vs ref to a line
    double A[4], V[2], X[2];
    A[0] = double(N);
    A[1] = ref.refDVect()->VSum(0, N);
    A[2] = A[1];
    A[3] = ref * ref;
    V[0] = test.refDVect()->VSum(0, N);
    V[1] = ref * test;
    if (solve22(A,V,X)) cout << "Unable to fit test to ref" << endl;
    else                cout << "linear fit is: " << X[1] 
			     << "x + " << X[0] << endl;

    //----------------------------------  Check fractional error
    TSeries x = test - ref;
    double maxError =  x.getMaximum();
    double minError =  x.getMinimum();
    double fError   =  (maxError >= -minError) ? maxError : -minError;
    cout << "Maximum Error= " << fError << endl;
    const double absCut(0.1);
    if ((flags & 1) != 0 && fError > absCut) {
        int Nerrors = N - x.getNBetween(-absCut, absCut);
	for (int i=0 ; i<N ; i+=8) {
	    int j = i+8;
	    if (j > N) j = N;
	    bool dump = false;
	    for (int k=i ; k<j ; k++) dump |= (fabs(x.getDouble(k)) > absCut);
	    if (dump) {
	        cout << setw(6) << i;
		for (int k=i ; k<j ; k++) cout << setw(9) << test.getDouble(k);
		cout << endl << "      ";
		for (int k=i ; k<j ; k++) cout << setw(9) << ref.getDouble(k);
		cout << endl;
		if (Nerrors > 1000) {
		    cout << "Too many errors (" << Nerrors << ")" << endl;
		    break;
		}
	    }
	}
    }
}

bool
LscEmul::compare(SeriesOut sel) const {
    switch(sel) {
    case kDArmCalc:
        cout << "Testing DArmCalc: stat=" << mDArmStat << " ";
        compareSeries(mDArmCalc, mDArmCtrl, 3);
	break;
    case kMichCalc:
        cout << "Testing MichCalc:  stat=" << mMichStat << " ";
        compareSeries(mMichCalc, mMichCtrl, 3);
	break;
    case kPrcCalc:
        cout << "Testing PrcCalc:  stat=" << mPrcStat << " ";
        compareSeries(mPrcCalc, mPrcCtrl, 3);
	break;
    case kCArmCalc:
        cout << "Testing CArmCalc:  stat=" << mCArmStat << " ";
        compareSeries(mCArmCalc, mCArmCtrl, 3);
	break;
    default:
        break;
    }
    return true;
}

//======================================  Configuration utility
void 
LscEmul::addLscVbl(const string& name, double& vbl, double def) {
    string chan = mLscPrefix + name;
    mDyn.addVbl(chan, vbl, def);
}

//======================================  Define Epics channels
void
LscEmul::defLscConfig(void) {
    if (!mFiltDB.empty()) {
        ostringstream file;
	file << mIFO << "LSC.txt";
	mFiltDB.init(file.str().c_str());
	mDArmFm = mFiltDB["DARM"];
	mCArmFm = mFiltDB["CARM"];
	mMichFm = mFiltDB["MICH"];
	mPrcFm  = mFiltDB["PRC"];
    }

    mAs1Pd.setup(mDyn, mLscPrefix);
    mAs2Pd.setup(mDyn, mLscPrefix);
    mAs3Pd.setup(mDyn, mLscPrefix);
    mAs4Pd.setup(mDyn, mLscPrefix);
    mPobPd.setup(mDyn, mLscPrefix);
    mRflPd.setup(mDyn, mLscPrefix);

    addLscVbl("ITMTRX_00",    mITMTRX_00,   0.0);
    addLscVbl("ITMTRX_01",    mITMTRX_01,   0.0);
    addLscVbl("ITMTRX_02",    mITMTRX_02,   0.0);
    addLscVbl("ITMTRX_03",    mITMTRX_03,   0.0);
    addLscVbl("ITMTRX_04",    mITMTRX_04,   0.0);
    addLscVbl("ITMTRX_05",    mITMTRX_05,   0.0);
    addLscVbl("ITMTRX_10",    mITMTRX_10,   0.0);
    addLscVbl("ITMTRX_11",    mITMTRX_11,   0.0);
    addLscVbl("ITMTRX_12",    mITMTRX_12,   0.0);
    addLscVbl("ITMTRX_13",    mITMTRX_13,   0.0);
    addLscVbl("ITMTRX_14",    mITMTRX_14,   0.0);
    addLscVbl("ITMTRX_15",    mITMTRX_15,   0.0);
    addLscVbl("ITMTRX_20",    mITMTRX_20,   0.0);
    addLscVbl("ITMTRX_21",    mITMTRX_21,   0.0);
    addLscVbl("ITMTRX_22",    mITMTRX_22,   0.0);
    addLscVbl("ITMTRX_23",    mITMTRX_23,   0.0);
    addLscVbl("ITMTRX_24",    mITMTRX_24,   0.0);
    addLscVbl("ITMTRX_25",    mITMTRX_25,   0.0);
    addLscVbl("ITMTRX_30",    mITMTRX_30,   0.0);
    addLscVbl("ITMTRX_31",    mITMTRX_31,   0.0);
    addLscVbl("ITMTRX_32",    mITMTRX_32,   0.0);
    addLscVbl("ITMTRX_33",    mITMTRX_33,   0.0);
    addLscVbl("ITMTRX_34",    mITMTRX_34,   0.0);
    addLscVbl("ITMTRX_35",    mITMTRX_35,   0.0);
    addLscVbl("DARM_OFFSET",  mDARM_OFFSET, 0.0);
    addLscVbl("MICH_OFFSET",  mMICH_OFFSET, 0.0);
    addLscVbl("PRC_OFFSET",   mPRC_OFFSET,  0.0);
    addLscVbl("CARM_OFFSET",  mCARM_OFFSET, 0.0);
    addLscVbl("DARM_GAIN",    mDARM_GAIN,   1.0);
    addLscVbl("MICH_GAIN",    mMICH_GAIN,   1.0);
    addLscVbl("PRC_GAIN",     mPRC_GAIN,    1.0);
    addLscVbl("CARM_GAIN",    mCARM_GAIN,   1.0);
    addLscVbl("DARM_LIMIT",   mDARM_LIMIT,    0);
    addLscVbl("MICH_LIMIT",   mMICH_LIMIT,    0);
    addLscVbl("PRC_LIMIT",    mPRC_LIMIT,     0);
    addLscVbl("CARM_LIMIT",   mCARM_LIMIT,    0);
    addLscVbl("DARM_SW1R",    mDARM_SW1,      0);
    addLscVbl("MICH_SW1R",    mMICH_SW1,      0);
    addLscVbl("PRC_SW1R",     mPRC_SW1,       0);
    addLscVbl("CARM_SW1R",    mCARM_SW1,      0);
    addLscVbl("DARM_SW2R",    mDARM_SW1A,     0);
    addLscVbl("MICH_SW2R",    mMICH_SW1A,     0);
    addLscVbl("PRC_SW2R",     mPRC_SW1A,      0);
    addLscVbl("CARM_SW2R",    mCARM_SW1A,     0);

    mDyn.addVbl("LscEpicsPi", mLscPi, M_PI);

    //----------------------------------  Define a load of filters
    string filterFile = mIFO + "LSC.txt";
    FilterDB fDB(filterFile.c_str());
    mAs1Pd.setFilter(fDB);
    mAs2Pd.setFilter(fDB);
    mAs3Pd.setFilter(fDB);
    mAs4Pd.setFilter(fDB);
    mPobPd.setFilter(fDB);
    mRflPd.setFilter(fDB);

    mDArmFm = fDB["DARM"];
    mCArmFm = fDB["CARM"];
    mPrcFm  = fDB["PRC"];
    mMichFm = fDB["MICH"];
}

//======================================  Set values
inline unsigned long 
bit_mask(int bLo, int bHi) {
    return (1<<(bHi+1)) - (1<<bLo);
}

static void
setSwit(const char* Name, FilterModule& fm, double sw1, double sw1a) {
    long sw =  (long(sw1) & bit_mask(4,15)) 
            + ((long(sw1a) & bit_mask(0, 7)) << 16);
    int mask = (sw >>  5) &   1; // fm1,  bits 4-5    sw1[ 4: 5]
    mask    |= (sw >>  6) &   2; // fm2,  bits 6-7    sw1[ 6: 7]
    mask    |= (sw >>  7) &   4; // fm3,  bits 8-9    sw1[ 8: 9]
    mask    |= (sw >>  8) &   8; // fm4,  bits 10-11  sw1[10:11]
    mask    |= (sw >>  9) &  16; // fm5,  bits 12-13  sw1[12:13]
    mask    |= (sw >> 10) &  32; // fm6,  bits 14-15  sw1[14:15]
    mask    |= (sw >> 11) &  64; // fm7,  bits 16-17  sw1a[ 0: 1]
    mask    |= (sw >> 12) & 128; // fm8,  bits 18-19  sw1a[ 2: 3]
    mask    |= (sw >> 13) & 256; // fm9,  bits 20-21  sw1a[ 4: 5]
    mask    |= (sw >> 14) & 512; // fm10, bits 22-23  sw1a[ 6: 7]
    if (mask != fm.getMask()) {
        fm.setMask(mask);
	cout << "Module: " << Name << " mask: " << mask << endl;
	fm.getFilter().dumpSosData(cout);
    }
}

void
LscEmul::getLscConfig(const Time& t0) {
    mDyn.getConfig(t0);
    setSwit("DArm", mDArmFm, mDARM_SW1, mDARM_SW1A);
    setSwit("CArm", mCArmFm, mCARM_SW1, mCARM_SW1A);
    setSwit("Prc", mPrcFm,  mPRC_SW1,  mPRC_SW1A);
    setSwit("Mich", mMichFm, mMICH_SW1, mMICH_SW1A);
}

LscEmul::LscPd::LscPd(const string& name) 
  : _name(name)
{
}

TSeries 
LscEmul::LscPd::getPortI(const TSeries& PdI, const TSeries& PdQ) {
    TSeries rc;
    double pi180 = _pi/180.0;

    double iGain = _I_gain * cos(_phase * pi180);
    if (!PdI.isEmpty() && iGain) {
        rc  = PdI;
	rc += _I_offset;
        rc  = _I_fm.filter(rc);
	rc *= iGain;
    }

    double qGain = _Q_gain * sin(_phase * pi180);
    if (!PdQ.isEmpty() && qGain) {
        TSeries temp  = PdQ;
	temp += _Q_offset;
        temp  = _Q_fm.filter(temp);
	temp *= qGain;
	rc   += temp;
    }
    return rc;
}

TSeries 
LscEmul::LscPd::getPortQ(const TSeries& PdI, const TSeries& PdQ) {
    TSeries rc;
    double pi180 = _pi/180.0;

    double qGain = _Q_gain * cos(_phase * pi180);
    if (!PdQ.isEmpty() && qGain) {
        rc  = PdQ;
	rc += _Q_offset;
        rc  = _Q_fm.filter(rc);
	rc *= qGain;
    }

    double iGain = - _I_gain * sin(_phase * pi180);
    if (!PdI.isEmpty() && iGain) {
        TSeries temp  = PdI;
	temp += _I_offset;
        temp  = _I_fm.filter(temp);
	temp *= iGain;
	rc   += temp;
    }
    return rc;
}

void 
LscEmul::LscPd::setup(DynConfig& f, const string& lscPfx) {
    string chanPfx = lscPfx + _name;
    f.addVbl(chanPfx+"_I_OFFSET", _I_offset, 0.0);
    f.addVbl(chanPfx+"_Q_OFFSET", _Q_offset, 0.0);
    f.addVbl(chanPfx+"_I_GAIN",   _I_gain,   0.0);
    f.addVbl(chanPfx+"_Q_GAIN",   _Q_gain,   0.0);
    f.addVbl(chanPfx+"_I_LIMIT",  _I_limit,  0.0);
    f.addVbl(chanPfx+"_Q_LIMIT",  _Q_limit,  0.0);
    f.addVbl(chanPfx+"_Phase",    _phase,    0.0);
    f.addVbl("LscEpicsPi",        _pi,       M_PI);
}

void 
LscEmul::LscPd::setFilter(const FilterDB& f) {
    if (f.exists(_name + "_I")) {
        _I_fm = f[_name + "_I"];
	_I_fm.setMask(1);
    }
    if (f.exists(_name + "_Q")) {
        _Q_fm = f[_name + "_Q"];
	_Q_fm.setMask(1);
    }
}
