#ifndef LSC_CALIB_HH
#define LSC_CALIB_HH

#include "FSeries.hh"
#include "TSeries.hh"
#include "CalibChan.hh"
#include <string>
#include <vector>

namespace xsil {
  class XSIL;
  class Xreader;
  class Xwriter;
}

/**  The LscCalib class provides access to the data needed to calibrate the
  *  asymmetric port (GW) data. 
  *  \brief Calibration data
  *  @author J. Zweizig
  *  @version $ID$
  */
class LscCalib {
public:
  /**  Construct an %LscCalib object using the specified name and file name.
    *  The name field contains a document name that is at present ignored.
    *  The file name may be specified as a null string (""), in which case,
    *  the calibration information will not be read until the read() or 
    *  readXml() method is called.
    *  \brief Construct an LscCalib object.
    *  \param name Name of calibration document
    *  \param file File from which the calibration data are to be read.
    *  \param chan Channel name
    *  \param gps  Calibration time.
    */
  LscCalib(const std::string& name, const std::string& file, 
	   const std::string& chan="", const Time& gps=Time(0));

  /**  Destroy an LscCalib object.
    */
  ~LscCalib(void);

  /**  Zero all internal fields of the calibration description.
    */
  void clear(void);

  /**  Get the name of the channel to which this calibration applies.
    *  \brief Get channel name.
    *  \return Constant pointer to channel name string.
    */
  const char* getChannel(void) const;

  /**  Get the level of debug printing.
    *  \brief Get the debug level.
    *  \return Debugging verbosity
    */
  int getDebug(void) const;

  /**  Get the name of this calibration document.
    *  \brief Get calibration document name.
    *  \return Calibration document name
    */
  const char* getName(void) const;

  /**  Get the calibration version. The version is typically a small integer
    *  but may contain other information as desired.
    *  \brief Get the calibration version.
    *  \return Calibration version string.
    */
  const char* getVersion(void) const;

  /**  Get the calibration version. The version is typically a small integer
    *  but may contain other information as desired.
    *  \brief Get the calibration version.
    *  \return Calibration version ID.
    */
  int getVersionID(void) const;

  /**  Get the calibration comment string.
    *  \brief Get the comment.
    *  \return Calibration comment string.
    */
  const char* getComment(void) const;

  /**  Get the name of the channel containing the calibration reference 
    *  waveform (assumed to be oneor more sinusoids) injected into the 
    *  ifo length control feeback loop.
    *  \brief Get the excitation channel name. The 
    *  \return Calibration line excitation channel.
    */
  const char* getEXCChannel(void) const;

  /**  Get the time at which the calibration was performed.
    *  \brief Get the calibration time.
    *  \return Time calibration was performed.
    */
  Time getCalibTime(void) const;

  /**  Get the calibration line frequency.
    *  \brief Get the calibration line frequency.
    *  \return Calibration line frequency
    */
  float getCalLineFreq(void) const;

  /**  Get the calibration line amplitude measured in the asymmetric port
    *  signal during the measurement of the nominal calibration.
    *  \brief Get the nominal calibration line amplitude in AsQ.
    *  \return Calibration line nominal readback amplitude
    */
  float getCalLineAmplASQ(void) const;

  /**  Get the calibration line amplitude measured in the excitation 
    *  channel during the measurement of the nominal calibration.
    *  \brief Get the nominal calibration line excitation amplitude.
    *  \return Calibration line nominal excitation amplitude
    */
  float getCalLineAmplEXC(void) const;

  /**  Get the cavity gain factor (alpha) at the specified time. A 
    *  runtime_error exception is thrown if the specified time is outside 
    *  of the TSeries  time range. 
    *  \brief Get the cavity factor.
    *  @param t Time at which the cavity factor is to be evaluated.
    *  @return Cavity gain factor.
    */
  double getCavFac(const Time& t) const;

  /**  Get the open loop gain factor (gamma) at the specified time. A 
    *  runtime_error exception is thrown if the specified time is outside 
    *  of the TSeries time range. 
    *  \brief Get the open-loop gain factor.
    *  \param t Time at which the open-loop gain factor is to be evaluated.
    *  \return Open loop gain factor.
    */
  double getOLGFac(const Time& t) const;

  /**  Get the start of the period for which this calibration is valid.
    *  \brief Get calibration validity start time.
    *  \return Calibration start time.
    */
  Time getStartTime(void) const;

  /**  Get the diuration of the period for which this calibration is valid.
    *  \brief Get calibration validity duration.
    *  \return Length of calibration validit in seconds.
    */
  Interval getDuration(void) const;

  /**  Get the number of channels that affect the open loop gain directly.
    *  \brief Get the number of gain channels.
    *  \return number of gain channels.
    */
  int getNGainChan(void) const;
 
  /**  Get the name of the ith gain channel.
    *  \brief Get the ith gain channel name.
    *  @param i Gain channel number.
    *  \return Pointer to the requested channel name
    */
  const char* getGainChan(int i) const;

  /**  Get the value of the gain channel during the reference calibration
    *  measurement.
    *  \brief Get ith channel reference value
    *  \param i Gain channel number.
    *  \return Reference value of the specified channel.
    */
  float getGainRefValue(int i) const;

  /**  Get the response function for the specified Cavity and Open-Loop gain 
    *  factors. Note that the response function is defined as 
    *  \[ R = (1 + OLG) / S \] where S is the Sensing function and OLG is the
    *  open loop gain. To convert a measured signal to a strain, one multiplies
    *  by the Response (R).
    *  @param alpha Cavity gain factor.
    *  @param gamma Open Loop Gain factor.
    *  @return Response function.
    */
  FSeries getResponse(double alpha, double gamma) const;

  /**  Get the response function at the specified time. The cavity and open
    *  loop gain factors are looked up at specified time in the internal 
    *  time series. The response function is then calculated using these 
    *  factors. 
    *  @param t Time at which the response function is to be evaluated.
    *  @return Response function.
    */
  FSeries getResponse(const Time& t) const;

  /**  Interpolate the sensing function in the specified frequency range
    *  (\a fMin - \a fMax) at points separated by the specified frequency 
    *  spacing (\a df). The returned frequency series is a single sided 
    *  series starting at \c f=0 with the specified spacing and points up 
    *  to (but not including) \a fMax. It is non-zero only in the specified 
    *  range. No interpolation is performed if the sensing function frequency 
    *  step is equal to the requested step. If interpolation is necessary, it 
    *  may be performed either linearly or logarithmically as specified by the
    *  \a logar parameter.
    *  \brief Interpolate the Sensing function
    *  \return Interpolated frequency series.
    *  \param fMin  Minimum non-zero frequency of returned series
    *  \param fMax  Maximum frequency of returned series
    *  \param df    Frequency step of returned series.
    *  \param logar If true, logarithmic interpolation.
    */
  FSeries interpSensing(double fMin, double fMax, double df, 
			bool logar=true) const;

  /**  Interpolate the open loop gain in the specified frequency range. The
    *  meaning of the arguments is identical to those in #interpSensing#.
    *  @return Interpolated frequency series.
    *  \brief Interpolate the open loop gain
    *  @param fMin  Minimum non-zero frequency of returned series
    *  @param fMax  Maximum frequency of returned series
    *  @param df    Frequency step of returned series.
    *  @param logar If true, logarithmic interpolation.
    */
  FSeries interpOpenLoopGain(double fMin, double fMax, double df, 
			     bool logar=true) const;

  /**  Assure global consitency of all fields.
    *  \brief Set undefined calibration fields
    */
  void prepare(void);

  /**  Set the calibration from the specified file. When frame data are 
    *  defined, this method will automatically determine the file type 
    *  (frame or xml) and call the appropriate method.
    *  \param file Reference to a frame or xml file path.
    *  \param title Calibration title
    *  \param chan Channel Name
    *  \param gps  Calibration time.
    */
  void read(const std::string& file, const std::string& title, 
	    const std::string& chan, const Time& gps=Time(0));

  /**  Set the calibration from the specified frame file.
    *  \brief read a calibration frame file.
    *  \param file Reference to a frame file path.
    *  \param title Calibration title
    *  \param chan Channel Name
    *  \param gps  Calibration time.
    */
  void readFrame(const std::string& file, const std::string& title, 
		 const std::string& chan, const Time& gps=Time(0));

  /**  Set the calibration from the specified xml file.
    *  \brief Read an xml file.
    *  \param file Reference to an xml file name string.
    *  \param title Calibration title
    *  \param chan Channel Name
    */
  void readXml(const std::string& file, const std::string& title, 
	       const std::string& chan);

  /**  Set the calibration from the xml document in the specified stream.
    *  \brief Read an xml stream.
    *  \param xw Reference to an open xreader.
    *  \param name Calibration title
    *  \param chan Channel Name
    */
  void readXml(xsil::Xreader& xw, const std::string& name, 
	       const std::string& chan="");

  /**  Return a reference to a time series containing Alpha values.
    *  \brief Get alpha time series.
    *  \return Constant reference to the alpha time series.
    */
  const TSeries& refAlpha(void);

  /**  Return a reference to a time series containing Alpha*Beta values.
    *  \brief Get alpha*beta time series.
    *  \return Constant reference to the alpha*beta (gamma) time series.
    */
  const TSeries& refAlphaBeta(void);

  /**  Return a reference to a time series containing the cavity factor
    *  (alpha) time series. Note that this is identical to refAlpha().
    *  \brief Get the cavity factor time series.
    *  \return Constant reference to the cavity factor time series.
    */
  const TSeries& refCavityFactor(void) const;

  /**  Return a reference to the open loop gain function frequency series.
    *  Note that the open-loop gain could be calculated from the response
    *  function and the Sensing function if it is not specified explicitly.
    *  The refOpenLoopGain function is not const to allow a future 
    *  implementation of this calculation.
    *  \brief Get the open loop gain.
    *  @return A constant reference to the Open Loop Gain FSeries.
    */
  const FSeries& refOpenLoopGain(void);

  /**  Return a reference to the response function frequency series.
    *  Note that the response function could be calculated from the open
    *  loop gain and the Sensing function if it is not specified explicitly.
    *  The refResponseFunction method is not const to allow a future 
    *  implementation of this calculation.
    *  \brief Get the open loop gain.
    *  @return A constant reference to the Respoinse Function FSeries.
    */
  const FSeries& refResponseFunction(void);

  /**  Return a reference to the sensing function (Frequency series).
    *  \brief Get the sensing function.
    *  \return Constant reference to the sensing transfer function.
    */
  const FSeries& refSensingFunction(void);

  /**  Set the open loop gain transfer function.
    *  \brief Set alpha time series.
    *  \param a Alpha time series.
    */
  void setAlpha(const TSeries& a);

  /**  Set the open loop gain factor time series.
    *  \brief Set the open loop gain factors.
    *  @param ab Gain factor time series.
    */
  void setAlphaBeta(const TSeries& ab);

  /**  Set the calibration measurement time.
    *  \brief Set the calibration time.
    *  @param t Calibration time.
    */
  void setCalibTime(const Time& t);

  /**  Specify all the parameters of a calibration line.
    *  \brief  Define a calibration line.
    *  @param exc     Excitation channel name.
    *  @param freq    Calibrationline frequency.
    *  @param amplAsq Nominal line amplitude in the calibrated channel.
    *  @param amplExc Nominal line amplitude in the excitation channel.
    */
  void setCalLine(const std::string& exc, float freq, float amplAsq, 
		  float amplExc);

  /**  Set the name of the channel for which the calibration is valid.
    *  \brief Set the channel name
    *  \param c Channel name string. 
    */
  void setChannel(const std::string& c);

  /**  Specify a comment describing this calibration.
    *  \brief Specify comment string.
    *  \param c Comment string.
    */
  void setComment(const std::string& c);

  /**  Set the debug flag
    *  \brief Set the debug level
    *  \param lvl Debugging printout verbosity level
    */
  void setDebug(int lvl);

  /**  Add a dynamic gain channel and specify its nominal amplitude.
    *  \brief Add a dunamic gain channel
    *  \param chan Channel name.  
    *  \param ampl Nominal amplitude value.  
    */
  void addGainChan(const std::string& chan, float ampl);

  /**  Set the document name for this calibration.
    *  \brief Set the ducument name.
    *  \param n Document name string.  
   */
  void setName(const std::string& n);

  /**  Set the open loop gain transfer function.
    *  \brief Set the OLG function.
    *  \param o OLG transfer function.  
    */
  void setOpenLoopGain(const FSeries& o);

  /**  Set the response function from a frequency series.
    *  \brief Set the response function.
    *  \param t Response transfer function.  
    */
  void setResponseFunction(const FSeries& t);

  /**  Set the response function from a frequency series.
    *  \brief Set the sensing function.
    *  \param t Sensing transfer function.  
    */
  void setSensingFunction(const FSeries& t);

  /**  Set the version string for this calibration.
    *  \brief Set calibration version string
    *  \param v Version string.
    */
  void setVersion(const std::string& v);

  /**  Set the Calibration groups numeric version ID for this calibration.
    *  \brief Set Calibration version
    *  \param id Calibration version number.
    */
  void setVersionID(int id);

  /**  Write the calibration as an Xml document to the specified file.
    *  \brief Write a calibration frame file.
    *  \param file Output file path
    *  \param fmt  Calibration frame file format.
    */
  void writeFrame(const std::string& file, 
		  CalibChanList::cal_format fmt=CalibChanList::cfmt_default);

  /**  Write the calibration as an Xml document to the specified file.
    *  \brief Write XML file
    *  \param file Xml file path.
    */
  void writeXml(const std::string& file);

  /**  Write the calibration as an Xml document to the specified Xsil writer.
    *  \brief Write XML file
    *  \param file Xml file writer.
    */
  void writeXml(xsil::Xwriter& file);

private:
  std::string mChannel;
  std::string mName;
  std::string mVersion;
  std::string mComment;
  int         mVersionID;

  Time     mStartTime;
  Interval mDuration;
  Time     mCalibTime;

  std::string mEXCChannel;
  float mCalLineFreq;
  float mCalLineAmplASQ;
  float mCalLineAmplEXC;

  std::vector<std::string> mGainChan;
  std::vector<float>       mGainRefValue;

  FSeries mSensingFunction;
  FSeries mOpenLoopGain;
  FSeries mResponseFunction;

  TSeries mAlpha;
  TSeries mAlphaBeta;

  int mDebug;
};

//======================================  inline methods
inline const char* 
LscCalib::getChannel(void) const {
    return mChannel.c_str();
}

inline int 
LscCalib::getDebug(void) const {
    return mDebug;
}

inline const char* 
LscCalib::getName(void) const {
    return mName.c_str();
}

inline const char* 
LscCalib::getVersion(void) const {
    return mVersion.c_str();
}

inline int
LscCalib::getVersionID(void) const {
    return mVersionID;
}

inline const char* 
LscCalib::getComment(void) const {
    return mComment.c_str();
}

inline Time
LscCalib::getStartTime(void) const {
    return mStartTime;
}

inline Time
LscCalib::getCalibTime(void) const {
    return mCalibTime;
}

inline Interval
LscCalib::getDuration(void) const {
    return mDuration;
}

inline const char* 
LscCalib::getEXCChannel(void) const {
    return mEXCChannel.c_str();
}

inline float 
LscCalib::getCalLineFreq(void) const {
    return mCalLineFreq;
}

inline float 
LscCalib::getCalLineAmplASQ(void) const {
    return mCalLineAmplASQ;
}

inline float 
LscCalib::getCalLineAmplEXC(void) const {
    return mCalLineAmplEXC;
}

inline int 
LscCalib::getNGainChan(void) const {
    return mGainChan.size();
}

inline const char* 
LscCalib::getGainChan(int i) const {
    return mGainChan[i].c_str();
}

inline float 
LscCalib::getGainRefValue(int i) const {
    return mGainRefValue[i];
}

inline FSeries 
LscCalib::interpSensing(double fmin, double fmax, double df, bool l) const {
    return mSensingFunction.interpolate(fmin, fmax, df, l);
}

inline FSeries 
LscCalib::interpOpenLoopGain(double fmin, double fmax, double df, bool l) const {
    return mOpenLoopGain.interpolate(fmin, fmax, df, l);
}

inline const TSeries& 
LscCalib::refAlpha(void) {
    return mAlpha;
}

inline const TSeries& 
LscCalib::refAlphaBeta(void) {
    return mAlphaBeta;
}

inline const FSeries& 
LscCalib::refOpenLoopGain(void) {
    return mOpenLoopGain;
}

inline const FSeries& 
LscCalib::refResponseFunction(void) {
    return mResponseFunction;
}

inline const FSeries& 
LscCalib::refSensingFunction(void) {
    return mSensingFunction;
}

#endif  // LSC_CALIB_HH
