/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//  Class FDCalibrate: Provides methods for calibrating ASQ power spectra
//  into units of physical displacement (nanometers or strain).
//    version:  0.4
//       date:  2005.03.31   
//     author:  Patrick J. Sutton (psutton@ligo.caltech.edu)
//
///////////////////////////////////////////////////////////////////////////


#ifndef FDCalibrate_HH
#define FDCalibrate_HH

#include "LscCalib.hh"
#include "FDFilter.hh"
#include "DaccAPI.hh"
#include "FSeries.hh"
#include "Time.hh"
#include "TSeries.hh"

/** Helper struct for FDCalibrate class.
  * Holds all information determined solely from the reference calibration.
  * @memo Holds information from reference calibration.
  * @author Patrick Sutton
  */
struct ReferenceCalibration
{
    //---------- Top-level stuff.

    /// Strain channel name.  Holds \c \<IFO\>:LSC-AS_Q, where \c \<IFO\> is 
    /// one of "H1", "H2", "L1", or "V1".
    std::string strain_channel_name;

    ///  Interferometer (one of H1, H2, or L1).
    std::string IFO;            

    /// Name of reference calibration file
    std::string ref_file_name; 

    /// Comment string from reference calibration file
    std::string ref_file_comment;

    //---------- Reference data for calibration line (for determining alpha)

    /// measured amplitude of calibration line in AS_Q at reference cal time.
    double ampl_asq_ref;

    /// injected amplitude of calibration line at reference calibration time.
    double ampl_exc_ref;
 
    /// Frequency of calibration line.
    double freq;

    /// Name of excitation channel in which calibration line is injected
    std::string exc_channel_name;

    //---------- Reference data for DARM gain (for determining beta).
  
    /// Reference values of darm gain channels
    double* darm_ref;

    /// Number of channels used to compute DARM gain.
    int num_darm_channels;

    /// Names of channels used to compute DARM gain.
    std::string* darm_channel_name;

    //---------- Stored alpha/beta calibration data (may be empty).

    /// Stored alpha values.
    TSeries Alpha;

    /// Stored alpha*beta values.
    TSeries AlphaBeta;

    //---------- Data on open-loop gain and sensing function. 
    //           Note that the actual OLG and SF data used by FDCalibrate 
    //           is not stored here, since it depends on what frequencies
    //           the user requests data for, and so is not determined 
    //           solely by the reference calibration.

    /// Minimum frequency for which calibration data is available.
    double Freq_min;

    /// Maximum frequencies for which calibration data is available.
    double Freq_max;

    /// Real part of open-loop gain at calibration-line freq.
    double reh;

    /// Imaginary part of open-loop gain at calibration-line freq.
    double imh;

    /// Unity gain frequency.
    double ugf;

    /// Minimum value of \c alpha*beta for which calibration is stable.
    double minab;

    /// Maximum value of \c alpha*beta for which calibration is stable.
    double maxab;
};
 
 
class Dacc;


/**  The FDCalibrate class is a basic low-level class to perform a
  *  frequency-domain calibration on FSeries (DFT) and FSpectrum (PSD) 
  *  objects.  It contains the information and methods required to 
  *  convert frequency series data from raw channel counts to physical 
  *  displacements in nanometers.
  *  @memo Class for calibrating FSeries and FSpectrum data.
  *  @author Patrick Sutton
  */
class FDCalibrate : public FDFilter {


public:

  /**  Default null constructor.
    *  @memo Default constructor.
    */
  FDCalibrate();


  /**  Constructor based on an uber reference-calibration file.   
    *  The frequency-dependent reference 
    *  calibration data is linearly interpolated to the specified 
    *  frequencies.  The response function is initialized to its  
    *  default value (alpha=1, beta=1).
    *  Alpha and beta values stored in the reference 
    *  file can be used instead of generating new calibrations by 
    *  setting the bool GenerateCalParam to false.
    *  @memo  Constructor using reference calibration file.
    *  @param access  Data accessor.
    *  @param refcal_file  File containing reference calibration data. 
    *  @param GenerateCalParam  Generate new alpha, beta parameters from data if true. 
    *  @param Fmin  Minimum frequency at which calibration is to be evaluated.
    *  @param Fstep  Spacing of frequencies at which calibration is to be evaluated.
    *  @param npoint  Number of frequencies at which calibration is to be evaluated.
    */
  FDCalibrate(DaccAPI *access, const char *refcal_file, bool GenerateCalParam,
	      double Fmin, double Fstep, int npoint);


  /**  Destroy an FDCalibrate object. 
    *  @memo Destructor.
    */
  virtual ~FDCalibrate(void);


  /**  Compute a new value for alpha using the gain ratio of the calibration
    *  line.
    *  \note Alpha depends on beta; be sure to call UpdateBeta() before this 
    *  method.  Alpha is set to zero if the current value if beta is not 
    *  physical as defined by IsBetaGood(). Calling ComputeResponseFunction() 
    *  in this case will set the response function to its reference value.
    *  \brief Compute a new value for alpha.
    *  \param ratio Ratio of the measured calibration-line amplitude in 
    *               the error signal to the expected value.
    */ 
  void ComputeAlpha(double ratio);


  /** Measure and return the amplitude of a line using a simple 
    * PSD-based estimate.  The procedure is
    * <ol>
    *   <li> Read in TSeries (preferably >~60 sec).
    *   <li> Compute PSD with Hanning window, 1Hz resolution.
    *   <li> Line power is the power in the bin containing the line freq, 
    *        and the nearest-neighbor bins (3 bins total)
    *        minus noise power in counted bins
    *        estimated from the average power in nearby bins.
    *   <li> Calculate line amplitude \f$ ampl = [ 2*dF*Line power ]^{0.5} \f$
    *        where \f$ dF = N_{windows}/T \f$ is the frequency resolution.
    * </ol>
    * @memo  Make PSD-based estimate of line amplitude.
    * @param ts  Pointer to TSeries containing data with the line.
    * @param Freq  Frequency of line.
    * @return Line amplitude.
    */
  float ComputeLineAmplitude(const TSeries *ts, double Freq) const;


  /** Compute response function strain(nm)->ASQ_counts.  This method 
    * computes and stores privately the detector response function 
    * using the last computed values of alpha and beta.  If the 
    * current alpha and beta are non-physical as defined by 
    * IsCalibrationGood(), the response function 
    * is set to its reference value (its value for alpha=1=beta).
    * The squared inverse transfer function \f$ ASQ^2 -> nm^2 \f$ can then
    * be applied to an FSpectrum by calling the inherited 
    * FDFilter::Apply(FSpectrum&) or     
    * FDFilter::Apply(FSpectrum&, FSpectrum&) methods.
    * @memo  Compute response function strain(nm)->ASQ.
    */
  void ComputeResponseFunction(void);


  /**  Return the current value of alpha. This is the last value calculated 
    *  or fetched by UpdateAlpha().
    *  @memo Alpha value.
    *  @return Current alpha value.
    */
  double GetAlpha(void) const;


  /**  Return the arm length of the interferometer, in nm. This can be used 
    *  to convert the calibrated displacement (in nm) to strain.
    *  @memo Return the arm length.
    *  @return Arm length in nm.
    */
  double GetArmLength(void) const;


  /**  Return the current value of beta.This is the last value calculated 
    *  or fetched by UpdateBeta().
    *  @memo Beta value.
    *  @return Current beta value.
    */
  double GetBeta(void) const;


  /**  Get the name of the channel to which this calibration applies.
    *  @memo Get channel name.
    *  @return Constant character pointer to the channel name.
    */
  const char* getChannel(void) const;
 
 
  /**  Get the comment string text from the reference calibration file.
    *  @memo  Get the comment string.
    *  @return Constant character pointer to the comment string.
    */
  const char* getComment(void) const;

  
  /**  Return the current measured calibration line amplitude in the
    *  channel to be calibrated (peak-to-peak amplitude).
    *  @memo Get calibration line amplitude.
    *  @return Calibration line amplitude.
    */
  double GetLineAmplitudeASQ(void) const;

  
  /**  Return the current measured calibration line amplitude in the
    *  Excitation channel. If the excitation channel is not available, 
    *  the nominal amplitude is returned.
    *  @memo Calibration line excitation amplitude.
    *  @return  Calibration line excitation amplitude.
    */
  double GetLineAmplitudeEXC(void) const;

  
  /**  Return the maximum value of \c alpha*beta for which calibration 
    *  is physical. 
    *  @memo Get Maximum phisical \c alpha*beta.
    *  @return Maximum phisical \c alpha*beta.
    */
  double GetMaxAlphaBeta(void) const;
  

  /**  Return the maximum frequency for which calibration is available.
    *  @memo Maximum calibration frequency.
    *  @return Maximum calibration frequency.
    */
  double GetMaxCalibF(void) const;


  /**  Return the minimum physical value of \c alpha*beta.
    *  @memo Minimum physical \c alpha*beta
    *  @return Lower limit for \c alpha*beta.
    */
  double GetMinAlphaBeta(void) const;


  /**  Return the minimum frequency for which calibratrion is available
    *  @memo Minimum calibration frequency.
    *  @return Minimum calibration frequency.
    */
  double GetMinCalibF(void) const;

  
  /**  Returns a pointer to a copy of the private member array containing 
    *  the amplitude of the open loop gain evaluated at NPoint frequency 
    *  values.
    *  @memo Open-loop gain amplitudes vs. frequency.
    *  @return Pointer to open-loop gain amplitudes vs. frequency.
    */
  const double* GetOLGAmp(void) const;
  
  
  /**  Returns a pointer to a copy of the private member array containing 
    *  the phase of the open loop gain evaluated at NPoint frequency values.
    *  @memo Open-loop gain phases vs. frequency.
    *  @return Pointer to open-loop gain phases vs. frequency.
    */
  const double* GetOLGPhase(void) const;
  
  
  /**  Returns the value of the current unity gain frequency
    *  @memo   Current unity gain frequency.
    *  @return Current unity gain frequency.
    */
  double GetUGF(void) const;


  /**  Returns true if \c alpha*beta is in physically stable range for 
    *  detector.
    *  @memo Test \c alpha*beta value.
    *  \return True if \c alpha*beta is in physically stable range
    */ 
  bool IsAlphaBetaGood(void) const;


  /**  Returns true if \f$ \beta > 0 \f$.
    *  @memo Test for valid beta.
    *  @return True if beta is valid.
    */
  bool IsBetaGood(void) const;


  /**  Returns true if calibration is physically acceptable; 
    *  i.e., if both IsAlphaBetaGood() and IsBetaGood() return true.
    *  @memo Test for physically acceptable alpha, beta.
    *  @return True if time-dependent parameters are physical.
    */ 
  bool IsCalibrationGood(void) const;


  /** Linear interpolate (x,y) data to values Y at X.
    * Using double arrays x, y of same length, linearly interpolate 
    * to calculate Y values at specified X. 
    * (This method is for reading and manipulating data from text 
    * files and has nothing in particular to do with calibration.)
    * @memo  Linear interpolate (x,y) data to values Y at X.
    * @param X  Pointer to array of points at which to interpolate (x,y).
    * @param Y  Pointer to array to hold data output from interpolation.
    * @param N  Number of points of data in arrays X, Y.
    * @param x  Pointer to array holding data to be interpolated.
    * @param y  Pointer to array holding data to be interpolated.
    * @param n  Number of points of data in arrays x, y.
    */
  void LinearInterpolation(const double *X,       double *Y, int N, 
			   const double *x, const double *y, int n) const;


  /**  Set alpha to the specified value.
    *  @memo Set current alpha value.
    *  @param value New alpha value.
    */
  void SetAlpha(double value);


  /**  Set the internally stored value of the arm length of the interferometer,
    *  based on the channel name in the calibration file.
    *  @memo Set arm length.
    */
  void SetArmLength(void);


  /**  Set beta to the specified value.
    *  @memo Set current beta value.
    *  @param value New beta value.
    */
  void SetBeta(double value);

  
  /**  This function updates the value of alpha stored in the FDCalibrate
    *  object from the "online" data.  It uses the privately stored 
    *  reference calibration data and data accessor, so it needs no 
    *  user-supplied arguments.   If the resulting value of alpha is 
    *  not real, alpha is reset to zero.
    *  WARNING: UpdateBeta() should be called before calling this
    *  method, since alpha depends on beta.
    *  @memo Calculate the current alpha value.
    */
  void UpdateAlpha(void);


  /** This function determines the values of alpha and beta 
    * for the specified time 
    * using stored calibration data 
    * from the reference calibration file.  
    * Zeros alpha and beta if data not available for the requested time.
    * @memo  Update calibration parameters alpha and beta.
    * @param t  GPS time for which alpha and beta are desired. 
    */ 
  void UpdateAlphaBeta(Time t);


  /** This function updates the value of beta stored in the FDCalibrate
    * object using "online" data.  
    * The new value is computed as the product of the averages 
    * of the supplied TSeries data objects, divided by the reference
    * value.  This method should be called before calling UpdateAlpha().
    * @memo  Update calibration parameter beta.
    */ 
  void UpdateBeta(void);

  
  /**  Updates the current unity gain using dynamic alphas and betas.
    *  The gain as a function of time is computed using:
    *  \f[

            G(f,t) = alpha * beta * G_0(t_0)
       \f]
    *  Where \f$ G_0(t_0)\f$ is simply the gain as a function of frequency 
    *  when the constructor is called. 
    */
  void UpdateUGF(void);

private:
  /**  Find the bin closest to the UGF bin. The data from vec[1:N] are
    *  scanned via binary search for the bin closes to the unity value.
    *  The function assumes that the gain is monotonically decreasing
    *  with bin number (frequency) in the range [1:N]. Unpredictable 
    *  results will be produced if this is not the case.
    *  @param N Number of bins
    *  @param vec Vetor of gain values.
    *  @param unity Unity gain value if not one.
    *  @return Bin number of UGF frequency.
    */
    int findUGFbin(int N, const double* vec, double unity=1.0) const;

private:

    // True if we are to generate new alpha, beta parameters on-the-fly.
    bool genCalParam;

    // Data accessor (so that FDCalibrate can access data on its own).
    DaccAPI* mDataAccess;

    // Current calibration parameters.
    double alpha;
    double beta;

    // Current line amplitude in ASQ and EXC.
    double ampl_asq;
    double ampl_exc;

    // Current gain.
    double gain;

    // Common frequencies at which calibration data is stored.
    double *f;

    // Number of frequencies at which calibration data is stored (size of f).
    int Npoint;

    // Response Function modulus and phase.  For convenience, keep the 
    // modulus also in an FSpectrum object (this is how it gets passed 
    // to parent FDFilter code).
    FSeries RF;

    // Open-Loop Gain Function.
    double *modH;
    double *phH;

    // Sensing Function.
    double *modC;
    double *phC;

    // IFO arm length.
    float arm_length;

    // Minimum, maximum frequencies for which calibration is applied
    // (the intersection of [Freq_min,Freq_max] and the frequency range 
    // of the AS_Q data, which is typically decimated to [0,4096]Hz).
    double freq_min, freq_max;

    // Frequency spacing at which calibration data is applied.
    double freq_step;

    // Reference calibration data.
    ReferenceCalibration mRefCal;

    // Name of window type to use in calculating the noise PSD.
    //std::string window_name; 

};

 
//======================================  inline methods
inline const char*
FDCalibrate::getChannel(void) const {
    return mRefCal.strain_channel_name.c_str();
}

inline const char*
FDCalibrate::getComment(void) const {
    return mRefCal.ref_file_comment.c_str();
}
 
inline double 
FDCalibrate::GetAlpha(void) const {
    return alpha;
}

inline double 
FDCalibrate::GetArmLength(void) const {
    return arm_length;
}

inline double 
FDCalibrate::GetBeta(void) const {
    return beta;
}

inline double 
FDCalibrate::GetLineAmplitudeASQ(void) const {
    return ampl_asq;
}

inline double 
FDCalibrate::GetLineAmplitudeEXC(void) const {
    return ampl_exc;
}

inline double 
FDCalibrate::GetMaxAlphaBeta(void) const {
    return mRefCal.maxab;
}

inline double 
FDCalibrate::GetMaxCalibF(void) const {
    return mRefCal.Freq_max;
}

inline double 
FDCalibrate::GetMinAlphaBeta(void) const {
    return mRefCal.minab;
}

inline double 
FDCalibrate::GetMinCalibF(void) const {
    return mRefCal.Freq_min;
}

inline const double* 
FDCalibrate::GetOLGAmp(void) const {
    return modH;
}

inline const double* 
FDCalibrate::GetOLGPhase(void) const {
    return phH;
}

inline double 
FDCalibrate::GetUGF(void) const {
    return mRefCal.ugf;
}

inline bool 
FDCalibrate::IsAlphaBetaGood(void) const {
    //----- Check that Alpha*Beta is in stable range.
    double ab = GetAlpha()*GetBeta(); 
    return (ab > mRefCal.minab) && (ab < mRefCal.maxab);
}

inline bool 
FDCalibrate::IsBetaGood(void) const {
    return GetBeta() > 0;
}

inline bool 
FDCalibrate::IsCalibrationGood(void) const
{
    return IsAlphaBetaGood() && IsBetaGood();
}

#endif  //----- FDCalibrate_HH

